//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Messaging_MessageReferrer_h)
#define Courier_Messaging_MessageReferrer_h

#include <Courier/Base.h>

namespace Courier { 
    class Message;
}

namespace Courier { 
    /// @addtogroup COURIER_MESSAGING
    /// @{
    /** The message referrer is an intrusive shared pointer for a message. */
    class MessageReferrer {
    public:
        ///
        MessageReferrer(Message * msg = 0);

        ///
        MessageReferrer(const MessageReferrer & msgRef);

        ///
        MessageReferrer & operator =(const MessageReferrer & msgRef);

        ///
        bool operator ==(const MessageReferrer & msgRef) const;

        ///
        ~MessageReferrer();

        /** Returns the message instance pointed by this message referrer.
            @return Pointer to message referenced by this message referrer instance. */
        FEATSTD_LINT_NEXT_EXPRESSION(1763,"the object itself is still const")
        Message * GetMessage() const { return mMsg; }
        FEATSTD_LINT_NEXT_EXPRESSION(1763,"the object itself is still const")
        Message * operator*() const { return mMsg; }

    private:
        ///
        Message * mMsg;
    };

    /** message referrer for const Message objects 
        @see MessageReferrer */
    class ConstMessageReferrer : private MessageReferrer {
    public:
        FEATSTD_LINT_CURRENT_SCOPE(1509,"base class destructor for class 'MessageReferrer' is not virtual => this is ok for this class")
        FEATSTD_LINT_CURRENT_SCOPE(1529,"Symbol 'Courier::ConstMessageReferrer::operator=(const Courier::ConstMessageReferrer &)' not first checking for assignment to this => this is ok for this class")
        ConstMessageReferrer(const Message *msg);
        ConstMessageReferrer(const ConstMessageReferrer &cRef);

        /** retrieve message object from referrer 
            @return the message object */
        FEATSTD_LINT_NEXT_EXPRESSION(1511, "base class privately inherited -> base class method not accessable")
        const Message* GetMessage() const { 
            return MessageReferrer::GetMessage();
        }

        /** retrieve message object from referrer 
            @return the message object */
        FEATSTD_LINT_NEXT_EXPRESSION(1511, "base class privately inherited -> base class method not accessable")
        const Message* operator*() const {
            return MessageReferrer::GetMessage();
        }

        ConstMessageReferrer& operator=(const ConstMessageReferrer &cRef) {
            MessageReferrer::operator=(cRef);
            return *this;
        }
    };

    /** Typed version of MessageReferrer
        the class supports both references to const or non-const messages
        @code
            TypedMessageReferrer<UpdateModelMsg> refMutable;            // define a referrer to non const UpdateModlMsg objects
            TypedMessageReferrer<const UpdateModelMsg> refImmutable;    // define a referrer to const UpdateModlMsg objects

            UpdateModelMsg *msg = *refMutable;                          // ok
            msg = *refImmutable;                                        // error!
        @endcode */
    template<typename T> class TypedMessageReferrer {
    public:
        /** constructs the message referrer with the given message object 
            @param msg the message object or 0 (null) */
        TypedMessageReferrer(T *msg = 0) : mReferrer(msg) {
        }

        TypedMessageReferrer(const TypedMessageReferrer &msgRef) : mReferrer(msgRef.mReferrer) {
        }


        /** Get the message object the referrer is referring to. 
            @return the message object or 0 (null) */
        inline T* GetMessage() const {
            return static_cast<T*>(mReferrer.GetMessage());
        }

        /** Get the message object the referrer is referring to. 
            @return the message object or 0 (null) */
        inline T* operator*() const {
            return static_cast<T*>(mReferrer.GetMessage());
        }

        inline TypedMessageReferrer& operator=(const TypedMessageReferrer &msgRef) {
            if (this != &msgRef) {
                mReferrer = msgRef.mReferrer;
            }
            return *this;
        }

    private:
        /** */
        template<typename X> struct ReferrerType {
            typedef MessageReferrer Type;
        };
        template<typename X> struct ReferrerType<const X> {
            typedef ConstMessageReferrer Type;
        };
        typename ReferrerType<T>::Type mReferrer;
    };


/// @}
}

#endif
