//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Messaging_MessageReceiver_h)
#define Courier_Messaging_MessageReceiver_h

#include <Courier/Base.h>

#include <Courier/Util/LinkedList.h>
#include <FeatStd/Util/FixedSizeString.h>
#include <Courier/Messaging/MessageReceiverPreprocessor.h>

// Forward declarations
namespace Courier {
    namespace Internal {
        class MessageRouter;
    }
    class MessageReferrer;
}

namespace Courier {

#if ! defined(COURIER_RECEIVER_MAX_COUNT)
    static const UInt32 cCOURIER_RECEIVER_MAX_COUNT = 16;
    #define COURIER_RECEIVER_MAX_COUNT Courier::cCOURIER_RECEIVER_MAX_COUNT
#else
    static const UInt32 cCOURIER_RECEIVER_MAX_COUNT = COURIER_RECEIVER_MAX_COUNT;
#endif

    /// @addtogroup COURIER_MESSAGING
    /// @{
    /** */
    class MessageReceiver {
    public:
        ///
        MessageReceiver();

        ///
        virtual ~MessageReceiver();

        /// Messaging receiver linked list node.
        Courier::Internal::LinkedListNode<MessageReceiver> MessageReceiverListNode;

        /** Activates the message receiver in the system. After calling this method
            the message router serves this receiver with appropriate messages (means
            the method Receive() is called).
            See also the more specialized ComponentMessageReceiver.
            \note Deactivation is done either via call of the Shutdown() - a protected method - or a call of the Deactivate method. */
        void Activate();

        /** Deactivates the message receiver in the system. After calling this method
            the message router stops serving this receiver with appropriate messages (means
            the method Receive() is called). */
        void Deactivate();

        /** Sets the name of the message receiver for logging purposes. The string will not be copied.
            @param name    Name of the component message receiver. */
        void SetName(const Char * name);

        /** Returns 0 or the name of the message receiver for logging purposes */
        const Char * GetName() const;

        ///
        bool IsShutdownTriggered() const { return mShutdownTriggered; }

        /** Interface to receive a message to be further processed.
            @note This interface is called in the message posting context.
            @param msgRef   Message reference object to receive.
            @return <em>true</em>   if message was successfully received,
                    <em>false</em>  if message reception failed. */
        virtual bool Receive(const MessageReferrer & /*msgRef*/) { return false; }

        /** Interface to start processing the message receiver. Usually called in the main loop of any thread.
            @note This interface has to be permanently called in a defined thread context.
            @return <em>true</em>   if processing was successful,
                    <em>false</em>  if a fatal failure occurred in processing. */
        virtual bool Process() { return false; }

        /** Attaches the given MessageReceiverPreprocessor to the message receiver.
            @param mp MessageReceiverPreprocessor to attach. */
        void AttachPreprocessor(MessageReceiverPreprocessor * mp);

        /** Detaches the given MessageReceiverPreprocessor from the message receiver.
            @param mp MessageReceiverPreprocessor to detach. */
        void DetachPreprocessor(MessageReceiverPreprocessor * mp);

    protected:
        /** The shutdown call de-registers the message receiver from the message router.
            Additionally the shutdown information flag is set. */
        virtual void Shutdown();

        /** Used to store the received message referrer in any storage (e.g. a message queue).
            Usually called in the Receive() interface to manage possible context switches.
            @param msgRef Message referrer to store.
            @return <em>true</em>   if message was successfully stored,
                    <em>false</em>  if storing the message failed. */
        virtual bool Store(const MessageReferrer & /*msgRef*/) { return false; }

        /** Used to fetch the next stored message referrer from storage (e.g. message queue).
            @note The returned message referrer is only valid, if the method finishes successful (returns true).
            @param msgRef Message referrer reference returned from storage if method finishes successful.
            @param waitForMessage If <em>true</em>, then the method shall only return if it can retrieve a message referrer from storage (blocking call).
                                  If <em>false</em>, then the method simply checks if a message referrer can be retrieved and returns immediately.
            @return <em>true</em>   If message was read and returned,
                    <em>false</em>  If no message was read. */
        virtual bool Fetch(MessageReferrer & /*msgRef*/, bool /*waitForMessage*/) { return false; }

        /** Iterates over all preprocessors attachted to the MessageReceiver. It is up to to the derived class, in our case ComponentMessageReceiver
            when this method is called. */
        void Preprocess();

    private:
        /// Linked list type for message preprocessing list
        typedef Internal::LinkedList<MessageReceiverPreprocessor, &MessageReceiverPreprocessor::MessageReceiverPreprocessorListNode> MessageReceiverPreprocessorList;

        /// Shutdown flag
        bool mShutdownTriggered;

        /// Name of the message receiver
        FeatStd::Internal::FixedSizeString<16> mName;

        /** Registers this instance of message receiver at the message router.*/
        void Register();

        /** De-registers this instance of message receiver from the message router. */
        void Deregister();

        /// List of message preprocessors 
        MessageReceiverPreprocessorList mMessageReceiverPreprocessorList;
    };

    /// @}
}

#endif
