//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Messaging_Message_h)
#define Courier_Messaging_Message_h

#include <Courier/Base.h>

#include <Courier/Platform/AtomicOp.h>
#include <Courier/Foundation/Component.h>
#include <Courier/Messaging/MessageSubscriberList.h>
#include <Courier/Messaging/MessagingTypes.h>
#include <Courier/Util/Identifier.h>
#include <Courier/Util/Traits.h>
#include <FeatStd/Util/StringBuffer.h>
#include <new>

namespace Courier { namespace UnitTest {
    class MessageTest;
}}

namespace Courier {
    class MessageReferrer;
    struct SerializationInfo;
}

/**
 *  Macros for message declaration.
 */
#define COURIER_MESSAGE_RTTI_DECLARATION( CLASS, BASECLASS, IDENTIFIER ) \
    FEATSTD_LINT_NEXT_EXPRESSION(1516, "message base class declaration") \
    typedef BASECLASS Base; \
    FEATSTD_LINT_NEXT_EXPRESSION(1516, "message identifier declaration") \
    static const ::FeatStd::SizeType ID = IDENTIFIER
#define COURIER_MESSAGE_SUBSCRIBER_LIST_SIZE( SUBSCRIBER_LIST_SIZE ) \
    FEATSTD_LINT_NEXT_EXPRESSION(1516, "message subscriber list size") \
    static const ::Courier::UInt8 SubscriberListSize = SUBSCRIBER_LIST_SIZE

namespace Courier {
    /// @addtogroup COURIER_MESSAGING
    /// @{
    /** Messages are entities routed through the system.
        Every message has an unique id and and may get the following kind of message category:
        - MessageCategory::Broadcast: For messages which should be broadcast to every application. No registration for
        these messages is needed!
        - MessageCategory::Focus: For messages, which should be send to the focused application/control. The applications
        do not need to register for messages which have the category MessageCategory::Focus.
        */
    class Message {
    public:
        /** Message scope mask */
        typedef UInt32 ScopeMask;

        /// Message Handle consists of message Id and sequence number.
        class Handle {
            public: 
                Handle() : mMsgId(0), mSeqNo(0) {}
                Handle(FeatStd::SizeType msgId, FeatStd::SizeType seqNo) : mMsgId(msgId), mSeqNo(seqNo) {}
                Handle(const Handle & handle) : mMsgId(handle.mMsgId), mSeqNo(handle.mSeqNo) {}
                bool IsValid() const { return (mMsgId!=0) && (mSeqNo!=0); }
                bool operator==(const Handle & handle) const { return (mMsgId==handle.mMsgId) && (mSeqNo==handle.mSeqNo); }
                bool operator!=(const Handle & handle) const { return (mSeqNo!=handle.mSeqNo) || (mMsgId!=handle.mMsgId); }
            private:    
                FeatStd::SizeType mMsgId; 
                FeatStd::SizeType mSeqNo;
        };

        /** Signature of message hook function */
        typedef bool (*MessageHookSignature)(Message * msg);

        ///
        Message();

        ///
        virtual ~Message();

        /** Returns the identifier of the message object.
            @return ID of message instance. */
        FeatStd::SizeType GetId() const {
            return GetMetadata().mId.mHash;
        }

        /** Returns the name of the message object.
            @return Name of the message. */
        const Char * GetName() const {
            return GetMetadata().mId.mStringId;
        }

        /** Defines, if the message has a modifiable subscription list.
            @return <em>true</em> Use modifiable subscription list,
                    <em>false</em> Use constant subscription list. */
        bool HasModifiableSubscriptionList() const {
            return (0 != GetMetadata().mCategory.mHasModifiableSubscriptionList);
        }

        /** Defines, if the message shall be distributed sequentially.
            @return <em>true</em> Use sequential distribution strategy,
                    <em>false</em> use parallel distribution strategy. */
        bool HasSequentialDistribution() const {
            return (0 != GetMetadata().mCategory.mHasSequentialDistribution);
        }

        /** Defines, if the message is defined to be sent as broadcast.
            @return <em>true</em> Broadcast message,
                    <em>false</em> use subscription list. */
        bool IsBroadcast() const {
            return (0 != GetMetadata().mCategory.mIsBroadcast);
        }

        /** Defines, which priority the message has.
            @return Priority of the message. */
        Priority::Enum GetPriority() const {
            return static_cast<Priority::Enum>(GetMetadata().mCategory.mPriority);
        }

        /** Returns the name of the message object.
            @return Name of the message. */
        MessageTag::Enum GetTag() const {
            return static_cast<MessageTag::Enum>(GetMetadata().mCategory.mTag);
        }

        /** Defines, if the message is not marked to have any special purposes (see MessageTag::Enum).
            @return <em>true</em> Message is not marked to have any special purpose,
                    <em>false</em> otherwise. */
        bool HasTag() const {
            return (MessageTag::None != GetMetadata().mCategory.mTag);
        }

        /** Defines, if the message is marked to carry data binding information.
            @return <em>true</em> Message is marked to send data binding information,
                    <em>false</em> otherwise. */
        bool IsDataBinding() const {
            return (MessageTag::DataBinding == GetMetadata().mCategory.mTag);
        }

        /** Defines, if the message is marked to be routed to an external instance.
            @return <em>true</em> Message is marked to be routed to an external instance,
                    <em>false</em> otherwise. */
        bool IsExternal() const {
            return (MessageTag::External == GetMetadata().mCategory.mTag);
        }

        /** Defines, if the message is marked to carry focus information.
            @return <em>true</em> Message is marked to send focus information,
                    <em>false</em> otherwise. */
        bool IsFocus() const {
            return (MessageTag::Focus == GetMetadata().mCategory.mTag);
        }

        /** Returns the scope mask of the message object.
            @return ScopeMask of message instance. */
        UInt32 GetScopeMask() const {
            FEATSTD_DEBUG_ASSERT(GetMetadata().mScopeMask > 0);
            return GetMetadata().mScopeMask;
        }

        /** Returns the subscription list valid for this type of message (either the static list or the local list or an empty list).
            @return Subscription list of this message type (defined by the category). */
        MessageSubscriberList GetSubscriberList();

        /** Returns the subscription list maximum size.
            @return The subscriber list size. */
        FeatStd::SizeType GetSubscriberListSize() const {
            return GetMetadata().mCategory.mSubscriptionListSize;
        }

        /** Checks, if target wants to handle the message.
            @returns true if target should be forwarded. */
        bool InScopeOf(const ScopeMask & targetScopeMask) const {
            return ((GetScopeMask() & targetScopeMask) > 0);
        }

        /** Returns the current message hook function
            @note The hooking functions GetMessageHook and SetMessageHook are not thread safe.
            @return The current message hook function pointer or 0 (null) if none set */
        static MessageHookSignature GetMessageHook();

        /** Sets a new message hook function
            The message hook function will be invoked with all posted messages before message distribution.
            @note The hooking functions GetMessageHook and SetMessageHook are not thread safe. The hook function
            may be invoked from different thread context.

            If the hooking function returns true, the given message is processed (distributed) normally. Message
            distribution will be abandoned if the hooking function return false.
            @attention In case the hooking function returns false, no reference counting will be activated. As consequence
            the caller has to care for the lifetime of the object (message object is NOT destroyed automatically).

            @param messageHook The pointer to the hooking function or 0 (null) to abandon hooking.
            @return The current set hook function or 0 (null) if no hook is set. */
        static MessageHookSignature SetMessageHook(MessageHookSignature messageHook);

        /** Posts the message (itself) to the message router.
            @param updateSubscriptionList   <em>true</em>   if static subscription list shall be copied to local subscription list,
                                            <em>false</em>  if the local subscription list shall not be updated.
            @return <em>true</em>   if message was successfully distributed,
                    <em>false</em>  message reception failed in at least one registered receiver. */
        bool Post(bool updateSubscriptionList = true);

        /** Returns the current index of the context specific recipient list (see comment on mCurrentRecipientIndex).
            @return The index to the context specific recipient list. */
        UInt8 CurrentRecipient() const {
            return mCurrentRecipientIndex;
        }

        /** Sets the current index of the context specific recipient list, this is used for Serialization and shall not be used otherwise. */
        void SetCurrentRecipient(UInt8 currentRecipientIndex) {
            mCurrentRecipientIndex = currentRecipientIndex;
        }

        /** Posts the message to next recipient in the context specific recipient list (see comment on mCurrentRecipientIndex).
            @return <em>true</em>   if message was successfully posted,
                    <em>false</em>  at least one distribution path failed. */
        bool PostToNextRecipient();

        /** Returns the serialization info of the message.
            @return <em>0</em> if no serialization info is available,
                    <em>SerializationInfo</em> if available for the message. */
        virtual const SerializationInfo * GetSerializationInfo() const;

        /** Returns if the message content is serializable.
            @return <em>true</em>   if the message is serializable,
                    <em>false</em>  if not. */
        bool IsSerializable() const;

        /** Returns the sequence number of the message.
            @return <em>FeatStd::SizeType</em> The sequence number. */
        FeatStd::SizeType GetSequenceNumber() const { return mSequenceNumber; }

        /** Sets the sequence number. Done only after deserialization of messages. */
        void SetSequenceNumber(FeatStd::SizeType seqNo) { mSequenceNumber = seqNo; }

        /** Returns the unique handle of the message.
            @return <em>Handle</em> The handle of the message. */
        Handle GetHandle() const { return Handle(GetId(),mSequenceNumber); }

#if defined(COURIER_MESSAGING_MONITOR_ENABLED)
        
        /// The monitoring data used for message monitoring feature.
        class MonitoringData {
            friend class Message;
            public:
                ///
                MonitoringData() : mPostTimestamp(0), mCyclesUnprocessed(0) {}
                ///
                MonitoringData(const MonitoringData & obj) : mPostTimestamp(obj.mPostTimestamp), mCyclesUnprocessed(obj.mCyclesUnprocessed) {}
                /** Initializes the monitoring data of the message. This is usually done when posting the message. */
                void Init();
                ///
                void IncCyclesUnprocessed() { ++mCyclesUnprocessed; }
                ///
                FeatStd::SizeType CyclesUnprocessed() const { return mCyclesUnprocessed; }
                /// Returns the timestamp of posting. Expired time in ms can be computed with Courier::Platform::Ticks(mPostTimestamp).GetExpiredTime(); 
                FeatStd::SizeType PostTimestamp() const { return mPostTimestamp; }
                ///  
                void SetPostTimestamp(FeatStd::SizeType val) { mPostTimestamp = val; }

            private:
                FeatStd::SizeType mPostTimestamp;
                FeatStd::SizeType mCyclesUnprocessed;
        };
           
        /// Returns a reference of the MonitoringData structure.
        MonitoringData & GetMonitoringData() { return mMonitoringData; }
#endif

#if defined(COURIER_MESSAGE_SERIALIZATION_ENABLED)
        /** Check if a message is marked for local (application internal) distribution only. 
            @note This information is only valid in case of parallel distribution of a subscription message.
            @return <em>true</em>   if message is marked for local distribution only,
                    <em>false</em>  otherwise. */
        bool IsMarkedForLocalDistributionOnly() const;

        /** Mark message for local (application internal) distribution only.
            @return <em>true</em>   if message was successfully marked for local distribution,
                    <em>false</em>  if message failed to be marked for local distribution (due to wrong message type). */
        bool MarkForLocalDistributionOnly();

        /** Returns the current checksum of the message.
            @return <em>checksum</em> the computed value, this is done usually when posting the message.
                    <em>0</em> if not computed, usually because the message is not posted. */
        FeatStd::SizeType GetChecksum() const { return mChecksumData.mChecksum; }

        /** Sets the checksum to the message. This is done during the serialization of the message. */
        void SetChecksum(FeatStd::SizeType checksum) { mChecksumData.mChecksum = checksum; }

        /** Marks the message as corrupted, so that no further distribution or checksum computation has to be done. */
        void SetCorrupted() { mChecksumData.mIsCorrupt = true; }

        /** Returns if the message is corrupt or not. */
        bool IsCorrupted() const { return mChecksumData.mIsCorrupt; }
#endif

    protected:
        /** Holds all the essential metadata information needed for message identification and categorization. */
        struct Metadata {
            struct Category {
                /// Use modifiable (true) or constant (false) subscription list.
                BitField mHasModifiableSubscriptionList : 1;
                /// Use sequential distribution strategy (true) or parallel (false).
                BitField mHasSequentialDistribution     : 1;
                /// Broadcast message (true) or use subscription list (false).
                BitField mIsBroadcast                   : 1;
                /// Priority Normal or High
                BitField mPriority                      : 1;
                /// Message tag information (see MessageTag::Enum).
                BitField mTag                           : 4;
                /// Maximum size of message's subscription list
                BitField mSubscriptionListSize          : 8;
            } mCategory;
            IdentifierStorage mId;
            UInt32 mScopeMask;
            const ComponentId * mStaticSubscriptionList;
        };

        /** Holds modifiable type specific data. */
        struct ModifiableMetadata {
            ModifiableMetadata() {
                *mSequenceNumber = 0; 
            }
            Platform::AtomicOp::Atomic mSequenceNumber;
        };

        /** Local stored subscriber list */
        ComponentId * mLocalSubscriberList;

        /** Initializes the local subscriber list with the default value. */
        void InitializeLocalSubscriberList(ComponentId * mSubscribers);

        /** Copies the type specific subscriber list to the object subscriber list. */
        void UpdateLocalSubscriberList();

        /** Resets the local subscriber list with the default value. */
        void ResetLocalSubscriberList();

        /** Adds a given component to the given subscription list and put it at the end resp. before a given component in the list list.
            @note This interface is used to manipulate the static, message type bound subscription list.
            @param subscriptionList Pointer to static subscription list.
            @param subscriptionListSize Size of static subscription list.
            @param componentId Component to subscribe.
            @param before The component in the list where the new component shall be inserted before.
            @return <em>true</em> Subscription was successful,
                    <em>false</em> Subscription failed or was not allowed. */
        static bool Subscribe(ComponentId * subscriptionList, UInt8 subscriptionListSize, ComponentId componentId, ComponentId before = ComponentType::Invalid);

        /** Removes a given component from the given subscription list.
            @note This interface is used to manipulate the static, message type bound subscription list.
            @param subscriptionList Pointer to static subscription list.
            @param subscriptionListSize Size of static subscription list.
            @param componentId Component to unsubscribe.
            @return <em>true</em> Unsubscription was successful,
                    <em>false</em> Unsubscription failed or was not allowed. */
        static bool Unsubscribe(ComponentId * subscriptionList, UInt8 subscriptionListSize, ComponentId componentId);

    private:
        /** Manipulates the reference counter. */
        friend class Courier::MessageReferrer;

        /** Test private methods. */
        friend class Courier::UnitTest::MessageTest;

        FEATSTD_MAKE_CLASS_UNCOPYABLE(Message);

#if defined(COURIER_MESSAGING_MONITOR_ENABLED)
        MonitoringData mMonitoringData;
#endif

#if defined(COURIER_MESSAGE_SERIALIZATION_ENABLED)
        ///
        struct ChecksumData {
            ChecksumData() : mChecksum(0), mIsCorrupt(false) {}
            FeatStd::SizeType mChecksum;
            bool mIsCorrupt;
        };
        ///
        ChecksumData mChecksumData;
#endif
        /// The sequence number of the message.
        FeatStd::SizeType mSequenceNumber;

        /** Internal reference counter. */
        Platform::AtomicOp::Atomic mRefCount;

        /** Pointer to next recipient, which can be in case of a sequential distribution of a subscription message
            the index to the message subscription list entry (accessed via GetSubscriptionList()), or
            in case of a sequential distribution of a broadcast message the index to the global list of message receiver
            (stored in the message router). 
            
            Additionally this flag is used for parallel distribution of a subscription message, when IPC is in place.
            When parallel distribution is used this flag usually is not used respectively set to 0. In case of IPC this 
            flag is set !=0, when a parallel distributed subscription message is received on an IPC channel, in order to mark it
            to be delivered only locally (application internal). If this marker is not set, then a received IPC message again 
            would send the message back to the sender, as it has no information, where it came from. */
        UInt8 mCurrentRecipientIndex;

        /** Abstract interface to define message metadata of derived message.
            @return Metadata of the message. */
        virtual const Metadata & GetMetadata() const = 0;

        /** Abstract interface to define modifiable message metadata of derived message.
            @return ModifiableMetadata of the message. */
        virtual Message::ModifiableMetadata & GetModifiableMetadata();

        /** Increments the reference counter. */
        void IncRef();

        /** Decrements the reference counter and destroys the message object,
            if the counter reaches 0. */
        void DecRef();

        /** Returns the reference counter of the message object.
            @return Reverence counter */
        UInt16 GetRefCounter() const {
            FEATSTD_DEBUG_ASSERT(*mRefCount <= 0x0000FFFF);
            return static_cast<UInt16>(*mRefCount);
        }

    public:
        /// Used in COURIER_MESSAGE_NEW
        static void * operator new (std::size_t size, void * ptr) throw() {
            FEATSTD_DEBUG_ASSERT(size > 0);
            FEATSTD_RELEASE_UNUSED(size);
            // NULL pointers are accepted! CTOR will not be called!
            return ptr;
        }

#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
        typedef FeatStd::StringBufferAppender<Courier::Message> MessageStringBufferAppender;
        template<typename T> friend struct FeatStd::StringBufferAppender;
        virtual ::FeatStd::UInt32 AppendTo(::FeatStd::StringBuffer& stringBuffer) const;
#endif

    private:
        static void * operator new (std::size_t) throw() { return 0; }
        static void * operator new[] (std::size_t) throw() { return 0; }
        static void * operator new[] (std::size_t, void *) throw() { return 0; }
    };

    /** Checks if given message has type T.
        @param msg Message to check
        @return <em>true</em> if message has given type
                <em>false</em> otherwise */
    template <class T> inline bool Equals(const Message & msg)
    {
        return (T::ID == msg.GetId());
    }

    /** Checks if given const message has type T.
        @param msg Pointer to message to check
        @return <em>true</em> if message has given type
                <em>false</em> otherwise */
    template <class T> inline bool Equals(const Message * msg)
    {
        return (Courier::Internal::PointerTrait<T>::ReferredType::ID == msg->GetId());
    }

    /** Function template message_cast is used to downcast a pointer of class Message
        to a pointer of some derived message class. It supports casting of both
        pointers to non-const and pointers to const messages.

        <em>message_cast</em> can be used only with pointers to Courier messages!

        @note A pointer to a const message will automatically be casted to a pointer
        to a non-const message derived class in case it is needed!
        Hence, this function template is not const correct.
        @code
            Messages::BroadcastMsg lBroadcastMsg = Messages::BroadcastMsg();

            Message * lMsg = &lBroadcastMsg;

            Messages::BroadcastMsg * lBroadcastMsgPtr = message_cast<Messages::BroadcastMsg *>(lMsg);
        @endcode
        @param msg   Message object (superclass)
        @return Down-casted message, if cast was successful,
                0, if cast was unsuccessful. */
    template <class T> inline T message_cast(Message * msg)
    {
        return ((0 == msg) || (!Equals<T>(msg))) ? 0 : static_cast<T>(msg);
    }

    /** @see message_cast(Message *) */
    template <class T> inline T message_cast(const Message * msg)
    {
        return ((0 == msg) || (!Equals<T>(msg))) ? 0 : static_cast<T>(msg);
    }

    FEATSTD_LINT_MESSAGE_STRING(1511, "Courier::Message::Subscribe(unsigned char *, unsigned char, unsigned char, unsigned char)", 
                                      "Subscribe method in generated message classes deliberate and not overwrites errornous overwrites of Message::Subscribe")

    FEATSTD_LINT_MESSAGE_STRING(1511, "*Courier::Message::Subscribe*",
        "Subscribe method in generated message classes deliberate and not overwrites errornous overwrites of Message::Subscribe")
    FEATSTD_LINT_MESSAGE_STRING(1511, "*Courier::Message::Unsubscribe*",
        "Subscribe method in generated message classes deliberate and not overwrites errornous overwrites of Message::Subscribe")

    /// @}
}

#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {
    template<> struct StringBufferAppender<Courier::Message>
    {
        static UInt32 Append(StringBuffer& stringBuffer, Courier::Message const & object)
        {
            return object.AppendTo(stringBuffer);
        }
    };
}
#endif
#endif
