//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Messaging_ComponentMessageReceiver_h)
#define Courier_Messaging_ComponentMessageReceiver_h

#include <Courier/Foundation/Component.h>
#include <Courier/Platform/MessageQueue.h>
#include <Courier/Messaging/MessageReceiver.h>
#include <Courier/Messaging/MessagingTypes.h>
#include <Courier/Platform/AtomicOp.h>

#if defined(COURIER_IPC_ENABLED)
#include <Courier/Ipc/IpcComponent.h>
#endif

namespace Courier {
    namespace UnitTest {
        class ComponentMessageReceiverWrapper;
    }

    /// @addtogroup COURIER_MESSAGING
    /// @{
    /** */
    class ComponentMessageReceiver : public MessageReceiver {
        typedef MessageReceiver Base;
    public:
        ///
        ComponentMessageReceiver(const Char * name = 0);

        ///
        virtual ~ComponentMessageReceiver();

        /** Attaches the given component to the message receiver and
            registers the new component to message receiver map entry to the message router.
            @param component Component to attach. */
        void Attach(Component * component);

        /** Detaches the given component from the message receiver
            and de-registers the component to message receiver map entry from the message router.
            @param component Component to detach. */
        void Detach(Component * component);

        /** Retrieve the maximum message queue reads per process cycle, which is currently used.
            @return Number of maximum message queue reads currently set. */
        UInt GetMaxMessageQueueReads() const;

        /** Sets the maximum message queue reads per process cycle (default is 1).
            @param lMaxMsgQReads    Number of maximum message queue reads. */
        void SetMaxMessageQueueReads(UInt lMaxMsgQReads);

        /** Receive a message to be further processed.
            @param msgRef   Message reference object to receive.
            @return <em>true</em>   if message was successfully received,
                    <em>false</em>  if message reception failed. */
        virtual bool Receive(const MessageReferrer & msgRef);

        /** Interface to start processing the message receiver. Usually called in the main loop of any thread.
            Before the messages are processed, attachted preprocessors are called using the MessageReceiver::Preprocess method.
            @note This interface has to be permanently called in a defined thread context.
            @note Method stops on first call to \a Fetch that returns false.
            @return <em>true</em>   if processing was successful,
                    <em>false</em>  if a fatal failure occurred in processing. */
        virtual bool Process();

        /**
            Sets the flag if the cycle time is allowed to be interrupted my a message.
            Set this flag to true if the cycle time has to be spent (according to the current cycle time type) completely.
            Otherwise, the Process method will return earlier if interrupted by a message.
            @param cycleTimeInterruptedByMessage the flag.
        */
        void SetCycleTimeInterruptedByMessage(bool cycleTimeInterruptedByMessage)
        {
            mCycleTimeInterruptedByMessage = cycleTimeInterruptedByMessage;
        }

        /** Sets the timeout operation mode and the timeout value in milliseconds.
            @param type the timeout mode. 
            @param timeMs the timeout value in milliseconds. */
        void SetCycleTime(ComponentMessageReceiverTiming::Enum type, UInt32 timeMs);

        ///
        UInt32 GetCycleTime() const { return mCycleTimeMs; }

    protected:
        /** All the component to message receiver mappings in the message router are dissolved and the
            component list in the message receiver is cleared. */
        virtual void Shutdown();

        /** This method executes all components, which are marked for continuous execution. */
        virtual void ExecuteComponents();

        /** Implements the routing strategy among the components for the incoming message, based on the message category definition.
            @param msg Message to route to the attached components.
            @return <em>true</em>   if routing was successful,
                    <em>false</em>  if a fatal failure occurred in routing. */
        virtual bool RouteMessage(Message & msg);

        /** Used to store the received message referrer in any storage (e.g. a message queue).
            Usually called in the Receive() interface to manage possible context switches.
            @param msgRef Message referrer to store.
            @return <em>true</em>   if message was successfully stored,
                    <em>false</em>  if storing the message failed. */
        virtual bool Store(const MessageReferrer & msgRef);

        /** Used to fetch the next stored message referrer from storage (e.g. message queue).
            @note The returned message referrer is only valid, if the method finishes successful (returns true).
            @param msgRef Message referrer reference returned from storage if method finishes successful.
            @param waitForMessage If <em>true</em>, then the method shall only return if it can retrieve a message referrer from storage (blocking call).
                                  If <em>false</em>, then the method simply checks if a message referrer can be retrieved and returns immediately.
            @return <em>true</em>   If message was read and returned,
                    <em>false</em>  if no message was read. */
        virtual bool Fetch(MessageReferrer & msgRef, bool waitForMessage)
        {
            return FetchImpl(msgRef, waitForMessage, false);
        }

        /** Used to fetch the next stored \b priority message (normal messages are ignored/skipped) referrer from storage (e.g. message queue).
            @note The returned message referrer is only valid, if the method finishes successful (returns true).
            @param msgRef Message referrer reference returned from storage if method finishes successful.
            @param waitForMessage If <em>true</em>, then the method shall only return if it can retrieve a message referrer from storage (blocking call).
                                  If <em>false</em>, then the method simply checks if a message referrer can be retrieved and returns immediately.
            @param priorityOnly If <em>true</em>, fetch only priority messages.
            @return <em>true</em>   If message was read and returned,
                    <em>false</em>  if no message was read. */
        virtual bool FetchPriority(MessageReferrer & msgRef, bool waitForMessage)
        {
            return FetchImpl(msgRef, waitForMessage, true);
        }

    private:
        bool FetchImpl(MessageReferrer & msgRef, bool waitForMessage, bool priorityOnly);

        /// Linked list type for component list
        typedef Internal::LinkedList<Component, &Component::ComponentListNode> ComponentList;

        /** True, if any component in the list has continuous execution enabled. */
        bool mContinuousExecutionEnabled;

        /// Maximum message queue reads per process cycle
        Platform::AtomicOp::Atomic mMaxMsgQReads;

        /** Standard message queue receiving all messages */
        Platform::MessageQueue mMsgQ;

#if defined(COURIER_IPC_ENABLED)
        IpcComponent * mIpcComponent;
#endif
        /// List of attached components
        ComponentList mComponentList;

        /** Searches a component based on the given component ID.
            @param componentId ID of component to search.
            @return Pointer to component if found, 0 otherwise. */
        Component * FindComponent(ComponentId componentId);

        /** De-registers component message receiver mapping from message router and
            clears the component list. */
        void ClearComponentList();

        /** Implements the broadcast routing strategy among the components for the incoming message.
            @param msg Message to route to the attached components.
            @return <em>true</em>   if processing was successful,
                    <em>false</em>  if a fatal failure occurred in processing. */
        bool DoBroadcastRouting(Message & msg);

        /** Implements the subscription routing strategy among the components for the incoming message.
            @param msg Message to route to the attached components.
            @return <em>true</em>   if processing was successful,
                    <em>false</em>  if a fatal failure occurred in processing. */
        bool DoSubscriptionRouting(Message & msg);

        /// Used timeout type
        ComponentMessageReceiverTiming::Enum mTimingType;

        /// Used message queue wait timeout in milliseconds.
        UInt32 mCycleTimeMs;

        /// Used message queue cycle time in ticks.
        UInt32 mCycleTimeTicks;

        /// Next tick (perfcounter units) for computation of the timeout in case UseCycleTimeout is used.
        UInt32 mNextTick;

        bool mCycleTimeInterruptedByMessage;

        friend class Courier::UnitTest::ComponentMessageReceiverWrapper;
    };

    /// @}
}

#endif
