//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ComponentMessageReceiver.h"

#include "MessageRouter.h"

#if defined(COURIER_MESSAGING_MONITOR_ENABLED)
    #include "MessagingMonitor.h"
#endif

#include <FeatStd/Platform/PerfCounter.h>

#include <Courier/Foundation/FoundationMsgs.h>
#include <Courier/Messaging/MessagingMsgs.h>

#include <Courier/Diagnostics/Trace.h>
#include <FeatStd/Platform/Thread.h>

// Local debug logging macros
#define LOG_COMPONENT_SEND_MSG(msg, comp)  \
    COURIER_LOG_DEBUG("Send %s to component %u", msg.GetName(), comp->GetId());
#define LOG_COMPONENT_MSG_CONSUMED(msg, comp, msgConsumed)  \
    COURIER_LOG_DEBUG("%s%s consumed by component %u", msg.GetName(), (msgConsumed ? "" : " not"), comp->GetId());
#define LOG_FORWARD_MSG_TO_NEXT_RECEIVER(msg, success)  \
    COURIER_LOG_DEBUG("Forward %s to next receiver%s succeeded", msg.GetName(), (success ? "" : " not"));

namespace Courier {

    COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Messaging);

    using namespace Diagnostics;

    // ------------------------------------------------------------------------
    ComponentMessageReceiver::ComponentMessageReceiver(const Char * name) :
        mContinuousExecutionEnabled(false), 
#if defined(COURIER_IPC_ENABLED)
        mIpcComponent(0),
#endif
        mTimingType(ComponentMessageReceiverTiming::Unused),
        mCycleTimeMs(0),
        mCycleTimeTicks(0),
        mNextTick(0),
        mCycleTimeInterruptedByMessage(false)
    {
        *mMaxMsgQReads = Internal::Limits<Platform::AtomicOp::ValueType>::Max();
        SetName(name);
    }

    // ------------------------------------------------------------------------
    ComponentMessageReceiver::~ComponentMessageReceiver()
    {
        // Detach and de-register all component message receiver mappings from message router
        ClearComponentList();
    }

    // ------------------------------------------------------------------------
    void ComponentMessageReceiver::Attach(Component * component)
    {
        FEATSTD_DEBUG_ASSERT(0 != component);
        if (0 != component) {
#if defined(COURIER_IPC_ENABLED)
            IpcComponentId ipcComponentId(component->GetId());
            if (ipcComponentId.GetComponentId() == ComponentId(ComponentType::Ipc)) {
                FEATSTD_DEBUG_ASSERT(0 == mIpcComponent);
                mIpcComponent = static_cast<IpcComponent*>(component);
                Internal::MessageRouter::RegisterComponentAtReceiver(this, component->GetId());
                mContinuousExecutionEnabled = mContinuousExecutionEnabled || component->IsContinuousExecutionEnabled();
                return;
            }
#endif
            mComponentList.Prepend(component);
            Internal::MessageRouter::RegisterComponentAtReceiver(this, component->GetId());
            mContinuousExecutionEnabled = mContinuousExecutionEnabled || component->IsContinuousExecutionEnabled();
        }
    }

    // ------------------------------------------------------------------------
    void ComponentMessageReceiver::Detach(Component * component)
    {
        FEATSTD_DEBUG_ASSERT(0 != component);
        if (0 != component) {
#if defined(COURIER_IPC_ENABLED)
            if (mIpcComponent == component) {
                mIpcComponent = 0;
                Internal::MessageRouter::DeregisterComponentFromReceiver(this, component->GetId());
                return;
            }
#endif
            (void)mComponentList.Remove(component);
            Internal::MessageRouter::DeregisterComponentFromReceiver(this, component->GetId());
        }
    }

    // ------------------------------------------------------------------------
    UInt ComponentMessageReceiver::GetMaxMessageQueueReads() const
    {
        return static_cast<UInt>(*mMaxMsgQReads);
    }

    // ------------------------------------------------------------------------
    void ComponentMessageReceiver::SetMaxMessageQueueReads(UInt lMaxMsgQReads)
    {
        (void)Platform::AtomicOp::Set(mMaxMsgQReads, static_cast<Platform::AtomicOp::ValueType>(lMaxMsgQReads));
    }

    // ------------------------------------------------------------------------
    bool ComponentMessageReceiver::Receive(const MessageReferrer & msgRef)
    {
        Message * lMsg = msgRef.GetMessage();
        bool lSuccess = (0 != lMsg);
        FEATSTD_DEBUG_ASSERT(lSuccess);

        // garantues calling of Execute method of the given Component.
        const TriggerComponentExecuteMsg * triggerComponentMsg = message_cast<const TriggerComponentExecuteMsg*>(lMsg);
        if(triggerComponentMsg!=0) {
            Component * component = FindComponent(triggerComponentMsg->GetComponentId());
            // ignore it because component is not existing.
            if(component==0) {
                return true;
            }
            // if set then ignore it.
            if(component->IsContinuousExecutionEnabled()) {
                return true;
            }
            // This flag is set to enable the "immediate" calling of Execute in case the component was disabled.
            component->EnableContinuousExecution(true);
        }

        // Valid message
        if (lSuccess) {
            COURIER_LOG_DEBUG("Receive %s in receiver 0x%p", lMsg->GetName(), this);
            // Store message referrer
            lSuccess = Store(msgRef);
        }
        // Message referrer with invalid content
        else {
            COURIER_LOG_FATAL("MessageReferrer carries no message!");
        }

        return lSuccess;
    }

    // ------------------------------------------------------------------------
#if defined(COURIER_MESSAGING_MONITOR_ENABLED)
    static void ComponentMessageReceiver_MessageObserve(void * data, const Message * msg)
    {
        if(msg!=0) {
            const_cast<Message*>(msg)->GetMonitoringData().IncCyclesUnprocessed();
            MessagingMonitor::EventMessageReceiverUnprocessed(msg, FeatStd::Internal::PointerToPointer<ComponentMessageReceiver*>(data));
        }
    }
#endif

    // ------------------------------------------------------------------------
    bool ComponentMessageReceiver::Process()
    {
        if (mCycleTimeInterruptedByMessage && (mCycleTimeTicks != 0)) {
            UInt32 now = FeatStd::Internal::PerfCounter::Now();
            if (now < mNextTick) {
                UInt32 timeToSleep = ((mNextTick - now) * FEATSTD_PERFCOUNTER_RESOLUTION) / 1000;
                FeatStd::Internal::Thread::Sleep(timeToSleep);
            }
        }

        MessageReferrer msgRef;
        Message * lMsg = 0;
        bool lSuccess = true;

        if(mTimingType==ComponentMessageReceiverTiming::Cycle) {
            mNextTick = (mNextTick==0) ? (FeatStd::Internal::PerfCounter::Now() + mCycleTimeTicks) : (mNextTick + mCycleTimeTicks);
        }
        else if ((mTimingType == ComponentMessageReceiverTiming::ResyncCycle) || (mTimingType == ComponentMessageReceiverTiming::MinResyncCycle))
        {
            mNextTick = FeatStd::Internal::PerfCounter::Now() + mCycleTimeTicks;
        }
        else {
            // MISRA compliance;
        }
        Preprocess();

        // Perform maximum count of reads on message queue
        UInt lMsgQReadCounter = 0;

        // Copy current maximum counter value on stack & use this value for this process cycle
        UInt lMaxMsgQReads = GetMaxMessageQueueReads();

        // Maximum number of message queue reads must be greater than 0, to be able to check for messages
        if (lMaxMsgQReads > 0) {
            // Calculate the maximum message queue reads for this loop cycle
            // based on the current queue size and given maximum value of message queue reads
            UInt lMsgQSize = static_cast<UInt>(mMsgQ.GetSize());
            UInt lMaxMsgQReadCount = ((lMsgQSize < lMaxMsgQReads) ? lMsgQSize : lMaxMsgQReads);

            do {
                // Check if message is available in queue (and wait for message if no component is marked for continuous execution)
                if (Fetch(msgRef, ! mContinuousExecutionEnabled)) {
                    lMsg = msgRef.GetMessage();
                    FEATSTD_DEBUG_ASSERT(0 != lMsg);
                    COURIER_LOG_DEBUG("Process %s in receiver 0x%p", lMsg->GetName(), this);

                    // Route received message
                    if (!RouteMessage(*lMsg)) {
                        lSuccess = false;
                    }

                    // Initiate shutdown for receiver
                    if (Equals<ShutdownMsg *>(lMsg)) {
                        COURIER_LOG_INFO("Shutdown initiated for receiver 0x%p", this);

                        // Shutdown message receiver and inform outer instance (caller) about termination
                        Shutdown();

                        // Discard messages still in message queue
                        MessageReferrer lMsgRef;
                        while (mMsgQ.Pop(lMsgRef)) {
                            COURIER_LOG_DEBUG("Discard %s", lMsgRef.GetMessage()->GetName());
                            // Increment current message queue read counter to terminate loop
                            ++lMsgQReadCounter;
                        }
                    } else {
                        const SetMessageReceiverCycleTimeMsg * setMsg = message_cast<const SetMessageReceiverCycleTimeMsg*>(lMsg);
                        if((setMsg!=0) && (0 != FindComponent(setMsg->GetComponentId()))) {
                            COURIER_LOG_INFO("SetMessageReceiverCycleTimeMsg received in 0x%p via component %u", this,setMsg->GetComponentId());
                            SetCycleTime(setMsg->GetTimingType(),setMsg->GetCycleTime());
                        }
                    }

                    // Increment current message queue read counter
                    ++lMsgQReadCounter;
                } 
                else {
                    break;
                }
            } while (lMsgQReadCounter < lMaxMsgQReadCount);

            UInt32 currentSize = static_cast<UInt32>(mMsgQ.GetSize());
            FEATSTD_UNUSED(currentSize); // if logging + messaging is disabled, variable/parameter is not used

#if defined(COURIER_MESSAGING_MONITOR_ENABLED)
            if(currentSize>0) {
                mMsgQ.Observe(this,ComponentMessageReceiver_MessageObserve);
            }
            if(lMsgQReadCounter>0) {
                MessagingMonitor::EventMessageReceiverProcess(this, lMsgQReadCounter, currentSize);
            }
#endif
            if (lMsgQReadCounter > 0) {
                COURIER_LOG_DEBUG("MsgQ reads: %u, queued: %u", lMsgQReadCounter, currentSize);
            }
        }

        // Trigger all components marked for execution
        ExecuteComponents();

        return lSuccess;
    }

    // ------------------------------------------------------------------------
//! [COURIER_TRACE_instrum]
    bool ComponentMessageReceiver::Store(const MessageReferrer & msgRef)
    {
        FEATSTD_DEBUG_ASSERT(msgRef.GetMessage() != 0);

        // Push message into message queue
        bool lSuccess = mMsgQ.Push(msgRef);
        if (lSuccess) {
            COURIER_TRACE_MESSAGE(TraceId::StoreMessage, msgRef.GetMessage());
        } else {
            COURIER_LOG_ERROR("Failed to store %s into msgQ", msgRef.GetMessage()->GetName());
        }
//! [COURIER_TRACE_instrum]
        return lSuccess;
    }

    // ------------------------------------------------------------------------
    bool ComponentMessageReceiver::FetchImpl(MessageReferrer & msgRef, bool waitForMessage, bool priorityOnly)
    {
        // This is always the default behavior: don't wait
        Platform::Ticks lMessageWaitTimeoutMs = Platform::Ticks(Platform::TicksType::Zero);

        // In case we have a fixed Constant wait time set that wait until timespan is finished
        if(mTimingType==ComponentMessageReceiverTiming::Constant) {
            lMessageWaitTimeoutMs = Platform::Ticks(mCycleTimeMs);
        // In case we have a cycle time then correct the wait time, make it shorter.
        }
        else if ((mTimingType == ComponentMessageReceiverTiming::Cycle) || (mTimingType == ComponentMessageReceiverTiming::ResyncCycle) || (mTimingType == ComponentMessageReceiverTiming::MinResyncCycle)) {
            if (mCycleTimeTicks != 0) {
                Int32 diffTicks = static_cast<Int32>(mNextTick)-static_cast<Int32>(FeatStd::Internal::PerfCounter::Now());
                Int32 diffMs = (diffTicks * FEATSTD_PERFCOUNTER_RESOLUTION) / 1000;
                if ((diffMs > 0) && (diffMs <= static_cast<Int32>(mCycleTimeMs))) {
                    lMessageWaitTimeoutMs.SetTime(static_cast<UInt32>(diffMs));
                }
            }
            if ((mTimingType == ComponentMessageReceiverTiming::MinResyncCycle) && waitForMessage) {
                lMessageWaitTimeoutMs = Platform::Ticks(Platform::TicksType::Infinite);
            }
        // In case we have the normal == old usecase wait until a message is received.
        } else {
            if(waitForMessage) {
                lMessageWaitTimeoutMs = Platform::Ticks(Platform::TicksType::Infinite);
            }
        } 

        // Check if message is available in queue
        const bool lRet = mMsgQ.Wait(msgRef, lMessageWaitTimeoutMs, priorityOnly);
        if (lRet) {
            FEATSTD_DEBUG_ASSERT(0 != msgRef.GetMessage());
            COURIER_TRACE_MESSAGE(TraceId::MessageFetched, msgRef.GetMessage());
        }
        return lRet;
    }

    // ------------------------------------------------------------------------
    void ComponentMessageReceiver::Shutdown()
    {
        // Detach and de-register all component message receiver mappings from message router
        ClearComponentList();

        // Call the shutdown processing from the base class
        Base::Shutdown();
    }

    // ------------------------------------------------------------------------
    void ComponentMessageReceiver::ExecuteComponents()
    {
        // Reset continuation flag for recalculation
        mContinuousExecutionEnabled = false;

        UInt32 currentTick = FeatStd::Internal::PerfCounter::Now();

        // Execute all marked components
        for (ComponentList::Iterator it = mComponentList.Begin(); it != mComponentList.End(); ++it) {
            Component * component = &*it;
            // Trigger component on every thread loop cycle, in case we have "unused" mode.
            if (mTimingType==ComponentMessageReceiverTiming::Unused) {
                if(component->IsContinuousExecutionEnabled()) {
                    component->Execute();
                }
            } else {
                // otherwise call it ...
                if(component->ShallExecute(currentTick)) {
                    component->Execute();
                }
            }

            component->PostProcess();

            // Check if any component is still marked for continuation (has to be calculated after components call of OnMessage and Execute)
            mContinuousExecutionEnabled = mContinuousExecutionEnabled || component->IsContinuousExecutionEnabled();
        }

#if defined(COURIER_IPC_ENABLED)
        if (0 != mIpcComponent) {
            if (mTimingType==ComponentMessageReceiverTiming::Unused) {
                if(mIpcComponent->IsContinuousExecutionEnabled()) {
                    mIpcComponent->Execute();
                }
            } else {
                // otherwise call it ...
                if(mIpcComponent->ShallExecute(currentTick)) {
                    mIpcComponent->Execute();
                }
            }

            // Check if any component is still marked for continuation (has to be calculated after components call of OnMessage and Execute)
            mContinuousExecutionEnabled = mContinuousExecutionEnabled || mIpcComponent->IsContinuousExecutionEnabled();
        }
#endif
    }

    // ------------------------------------------------------------------------
    bool ComponentMessageReceiver::RouteMessage(Message & msg)
    {
        bool lSuccess;

        const SetComponentMinExecutionCycleTimeMsg * setMsg = message_cast<const SetComponentMinExecutionCycleTimeMsg*>(&msg);
        if(setMsg!=0) {
            Component * component = FindComponent(setMsg->GetComponentId());
            if(component!=0) {
                component->SetMinExecutionCycleTime(setMsg->GetCycleTime(),setMsg->GetAllowMessageInterruption());
                return true;
            }
        }

        // Do broadcast routing among components
        if (msg.IsBroadcast()) {
            lSuccess = DoBroadcastRouting(msg);
        }
        // Handle subscription routing among components
        else {
            lSuccess = DoSubscriptionRouting(msg);
        }

        return lSuccess;
    }

    // ------------------------------------------------------------------------
    Component * ComponentMessageReceiver::FindComponent(ComponentId componentId)
    {
        if (componentId == static_cast<ComponentId>(ComponentType::Invalid)) {
            return 0;
        }

#if defined(COURIER_IPC_ENABLED)
        if(componentId==static_cast<Courier::ComponentId>(ComponentType::Ipc)) {
            return mIpcComponent;
        }

        IpcComponentId ipcComponentId(componentId);
        IpcProcessId processId = ipcComponentId.GetProcessId();
        if ((IpcComponent::GetLocalPid() != processId) &&  (processId != 0)) {
            return mIpcComponent;
        }
        componentId = ipcComponentId.GetComponentId();
#endif
        ComponentList::Iterator componentIterator = mComponentList.Begin();
        while ((componentIterator != mComponentList.End()) && (componentId != componentIterator->GetId())) {
            ++componentIterator;
        }

        return (componentIterator != mComponentList.End()) ? &*componentIterator : 0;
    }

    // ------------------------------------------------------------------------
    void ComponentMessageReceiver::ClearComponentList()
    {
#if defined(COURIER_IPC_ENABLED)
        if (0 != mIpcComponent) {
            Internal::MessageRouter::DeregisterComponentFromReceiver(this, mIpcComponent->GetId());
            mIpcComponent = 0;
        }
#endif
        // De-register component receiver map entry from message router
        for (ComponentList::Iterator componentIterator = mComponentList.Begin(); componentIterator != mComponentList.End(); ++componentIterator) {
            Internal::MessageRouter::DeregisterComponentFromReceiver(this, componentIterator->GetId());
            COURIER_LOG_DEBUG("Removed component receiver mapping (%u -> 0x%p)", componentIterator->GetId(), this);
        }

        // Empty component list in receiver
        COURIER_LOG_DEBUG("Clear component list in receiver 0x%p", this);
        mComponentList.Clear();
    }

    // ------------------------------------------------------------------------
    bool ComponentMessageReceiver::DoBroadcastRouting(Message & msg)
    {
        bool lSuccess = true;

        // Deliver message to all registered components
        for (ComponentList::Iterator componentIterator = mComponentList.Begin(); componentIterator != mComponentList.End(); ++componentIterator) {
            Component * component = &*componentIterator;
            // Execute component and prepare routing info for next call of this method
            LOG_COMPONENT_SEND_MSG(msg, component);

#if defined(COURIER_MESSAGING_MONITOR_ENABLED)
            MessagingMonitor::MeasurePoint mp;
#endif
            bool lMsgConsumed = component->ProcessMessage(msg);
#if defined(COURIER_MESSAGING_MONITOR_ENABLED)
            MessagingMonitor::EventMessageComponent(&msg,component->GetId(),mp,lMsgConsumed);
#endif
            LOG_COMPONENT_MSG_CONSUMED(msg, component, lMsgConsumed);
            FEATSTD_UNUSED(lMsgConsumed);
        }

        // In case of sequential broadcast, the message is forwarded to the next receiver
        if (msg.HasSequentialDistribution()) {
            lSuccess = msg.PostToNextRecipient();
            LOG_FORWARD_MSG_TO_NEXT_RECEIVER(msg, lSuccess);
        }

        return lSuccess;
    }

    // ------------------------------------------------------------------------
    bool ComponentMessageReceiver::DoSubscriptionRouting(Message & msg)
    {
        bool lSuccess = true;
        bool lMsgConsumed = false;

        // Get message subscriber list
        MessageSubscriberList lSubscriberList = msg.GetSubscriberList();

        // Do sequential subscription distribution
        if (msg.HasSequentialDistribution()) {
            // Forward message to all components, which are (in a row) subscribed to the message
            ComponentId lCurrentSubscriber = lSubscriberList.Current();

            // Check if current subscriber is handled in this message receiver (means is part of the message receiver's component list)
            Component * component = FindComponent(lCurrentSubscriber);

            // Forward message to all components which match the subscription list (in one block)
            while ((! lMsgConsumed) && (0 != component)) {
                // Execute component and prepare routing info for next call of this method
                LOG_COMPONENT_SEND_MSG(msg, component);

#if defined(COURIER_MESSAGING_MONITOR_ENABLED)
                MessagingMonitor::MeasurePoint mp;
#endif
                lMsgConsumed = component->ProcessMessage(msg);
#if defined(COURIER_MESSAGING_MONITOR_ENABLED)
                MessagingMonitor::EventMessageComponent(&msg,component->GetId(),mp,lMsgConsumed);
#endif

                LOG_COMPONENT_MSG_CONSUMED(msg, component, lMsgConsumed);

                // Increment subscription list pointer
                lCurrentSubscriber = lSubscriberList.Next();

                // Check if new subscriber is also part of receiver's component list
                component = FindComponent(lCurrentSubscriber);
            }

            // Forward message to the next component (if exists and message not already consumed) which is located in another receiver.
            // Subscription list must not be updated in this call of Message::Post()!
            if ((! lSubscriberList.End()) && (! lMsgConsumed)) {
                lSuccess = msg.Post(false);
                LOG_FORWARD_MSG_TO_NEXT_RECEIVER(msg, lSuccess);
            }
        }
        // Do parallel subscription distribution
        else {
            // Deliver message to all registered components
            for (ComponentList::Iterator componentIterator = mComponentList.Begin(); componentIterator != mComponentList.End(); ++componentIterator) {
                Component * component = &*componentIterator;

                // If component is part of message subscriber list, execute component
                if (lSubscriberList.Exists(component->GetId()))  {
                    // Execute component and prepare routing info for next call of this method
                    LOG_COMPONENT_SEND_MSG(msg, component);

#if defined(COURIER_MESSAGING_MONITOR_ENABLED)
                    MessagingMonitor::MeasurePoint mp;
#endif
                    lMsgConsumed = component->ProcessMessage(msg);
#if defined(COURIER_MESSAGING_MONITOR_ENABLED)
                    MessagingMonitor::EventMessageComponent(&msg,component->GetId(),mp,lMsgConsumed);
#endif
                    LOG_COMPONENT_MSG_CONSUMED(msg, component, lMsgConsumed);
                }
                FEATSTD_UNUSED(lMsgConsumed);
            }

#if defined(COURIER_IPC_ENABLED)
            // if an IPC component is registered pass message on as well, IPC component takes care if subscriber list
            // contains a remote component
            if (0 != mIpcComponent) {
                FEATSTD_LINT_NEXT_EXPRESSION(944, "lSuccess might be modified by some other thread")
                lMsgConsumed = mIpcComponent->ProcessMessage(msg) && lSuccess;
            }
#endif
        }
        COURIER_UNUSED(lMsgConsumed);
        return lSuccess;
    }

    // ------------------------------------------------------------------------
    void ComponentMessageReceiver::SetCycleTime(ComponentMessageReceiverTiming::Enum type, UInt32 timeMs) 
    { 
        if(type==ComponentMessageReceiverTiming::Unused) {
            timeMs = 0;
        } else {
            if(timeMs==0) {
                COURIER_LOG_ERROR("ComponentMessageReceiver '%s' TimingType(%d) must not be used with cycle time(%u)",GetName(), type, timeMs);
                return;
            }
        }
        
        COURIER_LOG_INFO("ComponentMessageReceiver '%s' TimingType(%d) cycle time(%u ms)",GetName(), type, timeMs);
        mNextTick = 0;
        mTimingType = type; 
        mCycleTimeMs = timeMs; 
        mCycleTimeTicks = (timeMs * 1000) / FEATSTD_PERFCOUNTER_RESOLUTION;
    }
}
