//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Foundation_Component_h)
#define Courier_Foundation_Component_h

#include <Courier/Base.h>
#include <Courier/Foundation/ComponentType.h>
#include <Courier/Platform/AtomicOp.h>

#include <Courier/Util/LinkedList.h>
#include <Courier/Platform/CriticalSection.h>
#if defined(COURIER_ENHANCED_ENABLED)
    #include <Courier/DataBinding/DataBindingFwd.h>
#endif

// Forward declarations
namespace Courier {
    class Message;
    class IComponentNotification;
}

namespace Courier {
    /// @addtogroup COURIER_FOUNDATION
    /// @{
    /** */
    class Component {
    public:
        /** Message Corruption Function prototype.
        @param  component the component which has detected the corrupted message.
        @param  message the corrupted message. */
        typedef void (*MessageCorruptionFct)(Component * component, const Message & message);

        /// Messaging receiver linked list node.
        Internal::LinkedListNode<Component> ComponentListNode;

        ///
        Component(ComponentId cid);

        ///
        virtual ~Component();

        static Component * Instance(ComponentId cid) {
            FEATSTD_DEBUG_ASSERT(cid < COURIER_COMPONENT_MAX_COUNT);
            return mInstance[cid];
        }

        /** Retrieve the ID of the component type.
            @return Component ID */
        ComponentId GetId() const { return mComponentId; }

        /// Tests if continuous flag is enabled.
        bool IsContinuousExecutionEnabled();

        /// Enables/disables continuous exceution.
        void EnableContinuousExecution(bool enable);

        /** Triggering component (every thread loop cycle), if continuous execution is enabled. */
        void Execute();

        /** Performs operations that have to be done for each processing of the component. */
        void PostProcess();

        /** Process given message in the component.
            This method calls the abstract virtual method OnMessage(), which implements the
            component specific routing strategy. It also sets the mContinuousExecution flag to true
            so that is guaranteed that Execute is called at least once.
            @param msg Message object to be processed.
            @return <em>true</em> if the message is consumed
                    <em>false</em> if the message shall be sent to following message receiving entities
                                   (valid only in case of sequential message distribution). */
        bool ProcessMessage(const Message & msg);

        /// Sets the optional notification object of the component.
        void SetComponentNotification(IComponentNotification * notification) {
            mComponentNotification = notification;
        }

        /** Sets the minimum execution time and if a previous received message may interrupt the minimum execution time.
            @param minimumExecutionTime the minimum execution time. 
            @param allowMessageInterrupt true if a message may interrupt the time. */
        void SetMinExecutionCycleTime(UInt32 minimumExecutionTime, bool allowMessageInterrupt);

        /// Returns the minium execution time.
        UInt32 GetMinExecutionCycleTime() const { return mMinimumExecutionTime; }

        /// Checks if it is time for execution.
        bool ShallExecute(UInt32 currentTick);

        /// Testable ShallExecute 
        static bool ShallExecute(UInt32 currentTick, UInt32 & oldTick, UInt32 cycleTime);

        /** Sets the message corruption callback. This allow implementing a corruption handler which is called 
            each time a corruption is detected. The checking is always done before OnMessage of the component is called.
            If a message is marked as corrupt distribution of the message will be stopped. The feature is only usable 
            if the FEATSTD_IPC_ENABLED feature is enabled.
            @param corruptionCallback the MessageCorruptionFct callback function. */
        static void SetMessageCorruptionCallback(MessageCorruptionFct corruptionCallback) {
            mMessageCorruptionCallback = corruptionCallback;
        } 

    private:
        /// List of component instances
        static Component * mInstance[COURIER_COMPONENT_MAX_COUNT];

        /// Callback which is called when an Message corruption appears.
        static MessageCorruptionFct mMessageCorruptionCallback; 

        /// Callback for notifications.
        IComponentNotification * mComponentNotification;

        /// Component id
        ComponentId mComponentId;
        
        /// Flag which indicates if calling Execute shall be done again until it is set (true = 1, false = 0).
        Platform::AtomicOp::Atomic mContinuousExecution;

        /// Allow Message Interrupt
        bool mAllowMessageInterrupt;

        /// Flag which indicates that a Message was processed
        bool mMessageProcessed;

        /// Minimum execution time
        UInt32 mMinimumExecutionTime;

        /// next execution time
        UInt32 mOldTick;

    protected:
        /** Hook for component specific execution call (triggered every thread loop cycle).
        @return <em>true</em> if subsequent Execute call of the component shall be made again.
                    <em>false</em> if calling Execute shall be stopped until a message starts calling Execute again. */
        virtual bool OnExecute() {
            return false;
        }

        /** Hook for component specific message processing, which can be customized for any derived component.
        @param msg Message object to be processed.
        @return <em>true</em> if the message is consumed
                <em>false</em> if the message shall be sent to following message receiving entities
                (valid only in case of sequential message distribution). */
        virtual bool OnMessage(const Message & msg) = 0;

        ///
        void AcquireDataLock() const;

        ///
        void ReleaseDataLock() const;

    };
    /// @}
}

#endif
