//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_TypedListInterfaces_h)
#define Courier_DataBinding_TypedListInterfaces_h

#include <Courier/DataBinding/DataBindingFwd.h>
#include <Courier/DataBinding/SharedAsyncListInterface.h>
#include <Courier/DataBinding/SyncListInterface.h>
#include <Courier/DataBinding/DataItemValue.h>

namespace Courier {  
/// @addtogroup COURIER_DATABINDING
/// @{

    // ========================================================================

    /**
        <summary>Typed asynchronous list interface.</summary>
        <seealso cref="AsyncListInterface"/>
     */
    template<typename T> class TypedAsyncListInterface : public AsyncListInterface {
        public:
            typedef T ListItemType;

            /**
                <summary>Typed version of Change. Request change to change the list item with given index.</summary>
                <param name="index">Zero-based index of the list item to change.</param>
                <param name="item">The item value.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            inline bool RequestChangeItem(UInt32 index, const T &item) {
                return RequestChange(index, DataItemValue(&item));
            }

            /**
                <summary>Typed version of RequestAdd. Request to add the given item to the list.</summary>
                <param name="index">Zero-based index where to insert the new item.</param>
                <param name="item">The item to insert.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            inline bool RequestAddItem(UInt32 index, const T &item) {
                return RequestAdd(index, DataItemValue(&item));
            }

    };


    // ========================================================================

    /**
        <summary>Typed shared asynchronous list interface.</summary>
        <seealso cref="SharedAsyncListInterface"/>
     */
    template<typename T> class TypedSharedAsyncListInterface : public SharedAsyncListInterface {
        public:
            typedef T ListItemType;

            /**
                <summary>Typed version of Change. Request change to change the list item with given index.</summary>
                <param name="index">Zero-based index of the list item to change.</param>
                <param name="item">The item value.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            inline bool RequestChangeItem(UInt32 index, const T &item) {
                return RequestChange(index, DataItemValue(&item));
            }

            /**
                <summary>Typed version of RequestAdd. Request to add the given item to the list.</summary>
                <param name="index">Zero-based index where to insert the new item.</param>
                <param name="item">The item to insert.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            inline bool RequestAddItem(UInt32 index, const T &item) {
                return RequestAdd(index, DataItemValue(&item));
            }
    };

    // ========================================================================

    /**
        <summary>Typed synchronise list interface.</summary>
        <seealso cref="SyncListInterface"/>
     */
    template<typename T> class TypedSyncListInterface : public SyncListInterface {
        public:
            typedef T ListItemType;

            /**
                <summary>Typed version of Item call. Gets an item from the list.</summary>
                <param name="index">Zero-based index of the requested item.</param>
                <param name="item">[in,out] The item.</param>
                <returns>true if the item could be retrieved, false otherwise.</returns>
             */
            bool GetItem(FeatStd::SizeType index, T &item) const {
                DataItemValue value(Item(index));
                const T *pValue = value.GetValue<T>();
                if (pValue != 0) {
                    item = *pValue;
                }
                return pValue != 0;
            }

            /**
                <summary>Typed version of Change. changes the item with the given index.</summary>
                <param name="index">Zero-based index of the item to change.</param>
                <param name="item">The new item.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            inline bool ChangeItem(FeatStd::SizeType index, const T &item) {
                return Change(index, DataItemValue(&item));
            }

            /**
                <summary>Typed version of Add. Adds an item to the list.</summary>
                <param name="index">Zero-based index where to insert the item.</param>
                <param name="item">The item to insert.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            inline bool AddItem(FeatStd::SizeType index, const T &item) {
                return Add(index, DataItemValue(&item));
            }

    };

//@}
}
#endif // Courier_DataBinding_TypedListInterfaces_h
