//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_TypeConverter_h)
#define Courier_DataBinding_TypeConverter_h

#include <Courier/Base.h>

#include <Courier/Util/ObjectList.h>
#include <Courier/DataBinding/DataItemType.h>
#include <Courier/DataBinding/DataItemValue.h>

namespace Courier {
    /// @addtogroup COURIER_DATABINDING
    /// @{
    class TypeConverterBase;

    /** signature of conversion function
    conversion functions convert from src to dst */
    typedef bool (*ConvertSignature)(const DataItemValue &dst, const DataItemValue &src);

    /** TypeConverterVtbl stores all TypeConverter relevant information */
    struct TypeConverterVtbl {
        DataItemTypeId mTypeIdT1;                       ///< DataItemTypeId of type 1
        DataItemTypeId mTypeIdT2;                       ///< DataItemTypeId of type 2
        ConvertSignature Convert;                       ///< function pointer being able to convert T1->T2 and T2->T1
    };

    namespace Internal {
        namespace DataBinding {
            /** TypeConverterRegistrar maintains references to all type converters in the system
            TypeConverterBase automatically will register the type converter to
            TypeConverterRegistrar */
            class TypeConverterRegistrar {
            public:
                /** finds a type converter being able to convert between t1 and t2 (and vice versa)
                @param t1 type 1
                @param t2 type 2
                @return type converter object being able to convert t1->t2 and t2->t1 */
                static TypeConverterBase* Locate(DataItemTypeId t1, DataItemTypeId t2);

                /** enlists a type converter to TypeConverterRegistrar
                @param tc pointer to type converter object
                @return true if type converter could be added, false otherwise */
                static bool Enlist(TypeConverterBase *tc);

                /** delists a type converter from TypeConverterRegistrar
                @param tc pointer to type converter object
                @return true if type converter could be removed, false otherwise */
                static bool Delist(TypeConverterBase *tc);
            };
        }   // private
    } // Internal

    /** type converters are object being able to convert between two types
    the types are denominated with T1 and T2. Conversion is bi-directional */
    class TypeConverterBase {
    public:
        /** dtor - delists the type converter from the TypeConverterRegistrar */
        ~TypeConverterBase();

        /** converts from src to dst
        @param dst mutable DataItemValue referencing an object of T1 or T2
        @param src DataItemValue referencing an object of T2 or T1
        @return true if conversion succeeded, false otherwise */
        bool Convert(DataItemValue &dst, const DataItemValue &src) const {
            return mVtblRef.Convert(dst, src);
        }

        /** returns DataItemTypeId of T1 */
        DataItemTypeId TypeIdT1() const {
            return mVtblRef.mTypeIdT1;
        }

        /** returns DataItemTypeId of T2 */
        DataItemTypeId TypeIdT2() const {
            return mVtblRef.mTypeIdT2;
        }

        /** returns the TypeConverterVtbl associated with this type converter */
        const TypeConverterVtbl& Vtbl() const {
            return mVtblRef;
        }

    protected:
        /** constructs a type converter using the given vtbl
        @param vtbl static type conversion information. Ownership and lifecycle management
        of vtbl remains at caller side. The caller must guarantee that vtbl is
        not destroyed before the type converter object */
        TypeConverterBase(const TypeConverterVtbl &vtbl);

        /** check if dst and src types match the given type converter types */
        static bool Check(const DataItemValue &dst, const DataItemValue &src,
            DataItemTypeId typeIdT1, DataItemTypeId typeIdT2);

    private:
        TypeConverterBase *mNext;                   ///< next pointer for TypeConverterRegistrar linked list
        const TypeConverterVtbl &mVtblRef;             ///< reference to the vtbl object

        /** returns next pointer for TypeConverterRegistrar linked list */
        TypeConverterBase* Next() const {
            return mNext;
        }

        /** sets next pointer for TypeConverterRegistrar linked list */
        void SetNext(TypeConverterBase *value) {
            mNext = value;
        }

        friend struct Internal::DefaultAccessor<TypeConverterBase>;

        FEATSTD_MAKE_CLASS_UNCOPYABLE(TypeConverterBase);
    };

    /** Helper class to ease type converter implementation.
    The class assumes the existance of the functions
    - bool TypeConvert(T1 &, const T2&)
    - bool TypeConvert(T2 &, const T1&)
    Following example shows the implementation of a type converter for
    std::string and DateTime
    \snippet CourierUnitTest/DataBinding/TypeConverterTest.cpp DataBinding_TypeConverterSample */
    template<typename T1, typename T2> class TypeConverter : public TypeConverterBase {
    public:
        /** unqualified T1 type */
        typedef typename DataItemType<T1>::UnqualifiedType UnqualifiedT1;
        /** unqualified T2 type */
        typedef typename DataItemType<T2>::UnqualifiedType UnqualifiedT2;

        /** ctor */
        TypeConverter() : TypeConverterBase(mVtbl) {
        }

    private:
        static const TypeConverterVtbl mVtbl;       ///< static initialized vtbl

        /** DoConvert converts from src to dst */
        static bool DoConvert(const DataItemValue &dst, const DataItemValue &src) {
            // ensure dst to be mutable and check of dst and src types match the converter types
            bool ok = Check(dst, src, mVtbl.mTypeIdT1, mVtbl.mTypeIdT2);
            FEATSTD_DEBUG_ASSERT(ok);
            if (ok) {
                if (dst.TypeId() == mVtbl.mTypeIdT2) {  // convert t1->t2
                    ok = TypeConvert(*dst.GetMutableValue<UnqualifiedT2>(), *src.GetValue<UnqualifiedT1>());
                }
                else {                              // convert t2->t1
                    ok = TypeConvert(*dst.GetMutableValue<UnqualifiedT1>(), *src.GetValue<UnqualifiedT2>());
                }
            }
            return ok;
        }
    };

    template<typename T1, typename T2> const TypeConverterVtbl TypeConverter<T1, T2>::mVtbl = {
        COURIER_DATA_ITEM_TYPENAME_ID(typename TypeConverter<FEATSTD_COMMA(T1, T2)>::UnqualifiedT1),
        COURIER_DATA_ITEM_TYPENAME_ID(typename TypeConverter<FEATSTD_COMMA(T1, T2)>::UnqualifiedT2),
        &TypeConverter<T1, T2>::DoConvert
    };

    //@}
}   // namespace

#endif // Courier_DataBinding_TypeConverter_h
