//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_SyncListInterface_h)
#define Courier_DataBinding_SyncListInterface_h

#include <Courier/DataBinding/DataBindingFwd.h>
#include <Courier/DataBinding/ListInterfaceBase.h>
#include <FeatStd/Util/Optional.h>

namespace Courier {  
/// @addtogroup COURIER_DATABINDING
/// @{

    /**
        <summary>Abstract interface for synchronous access to list collections.</summary>
     */
    class SyncListInterface : public ListInterfaceBase {
        public:
            /**
                <summary>Retrieve the item with the given index from the list.</summary>
                <param name="index">Zero-based index.</param>
                <returns>A valid DataItemValue object if index is within list bounds, an invalid otherwise.</returns>
             */
            virtual DataItemValue Item(FeatStd::SizeType index) const = 0;

            /**
            <summary>Retrieves if the item has been modified since the last call of ResetModified.</summary>
            <param name="index">Zero-based index.</param>
            <returns>The modified flag of the item.</returns>
            */
            virtual bool IsModified(FeatStd::SizeType index) const = 0;

            /**
            <summary>Sets the modified flag of the item.</summary>
            <param name="index">Zero-based index. If not provided then all items will be marked as modified.</param>
            */
            virtual void SetModified(FeatStd::Optional<FeatStd::SizeType> index) = 0;

            /**
            <summary>Resets the modified flag of the item.</summary>
            <param name="index">Zero-based index. If not provided then all items will be marked as not modified.</param>
            */
            virtual void ResetModified(FeatStd::Optional<FeatStd::SizeType> index) = 0;

            /**
                <summary>Replaces the item with the given index with the given value.</summary>
                <param name="index">Zero-based index.</param>
                <param name="value">The value.</param>
                <returns>true if the list is not readonly and index is within list bounds.</returns>
             */
            virtual bool Change(FeatStd::SizeType index, const DataItemValue &value) = 0;

            /**
                <summary>Adds the given value at the given index to the list.</summary>
                <param name="index">Zero-based index.</param>
                <param name="value">The value.</param>
                <returns>
                    true if index is within list bounds, the list is not readonly and list is not full.
                </returns>
             */
            virtual bool Add(FeatStd::SizeType index, const DataItemValue &value) = 0;

            /**
                <summary>Removes the item with the given index from the list.</summary>
                <param name="index">index of the item to remove.</param>
                <returns>true if index is in list bounds and list is not readonly.</returns>
             */
            virtual bool Remove(FeatStd::SizeType index) = 0;

            /**
                <summary>Moves item at index src to index dst.</summary>
                <param name="dst">Destination index.</param>
                <param name="src">Source index.</param>
                <returns>true if indices are in list bounds and list is not readonly</returns>
             */
            virtual bool Move(FeatStd::SizeType dst, FeatStd::SizeType src) = 0;

            /**
                <summary>Removes all items from the list.</summary>
                <returns>true if list is not readonly.</returns>
             */
            virtual bool Clear() = 0;
    };

//@}
}

#endif // Courier_DataBinding_SyncListInterface_h
