//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_SharedAsyncListInterface_h)
#define Courier_DataBinding_SharedAsyncListInterface_h

#include <Courier/DataBinding/AsyncListInterface.h>
#include <Courier/DataBinding/ListEventHandler.h>

namespace Courier {  
/// @addtogroup COURIER_DATABINDING
/// @{

    /**
        <summary>Defines a asynchronous list interface with SharedPointer reference counting.</summary>
        <seealso cref="AsyncListInterface"/>
     */
    class SharedAsyncListInterface : public AsyncListInterface
    {
        public:
            /**
                <summary>Set the callback function to be invoked when a list event happens.</summary>
                <param name="object">[in,out] The callback object or 0 (null).</param>
                <param name="CallbackFn">
                    [in,out] pointer to the member function of object.
                </param>
             */
            template<typename T> inline void SetEventCallback(T *object, void (T::*CallbackFn)(const ListEvent &listEvent))
            {
                mEventHandler.SetEventCallback(object, CallbackFn);
            }

            /**
                <summary>Resets the event callback.</summary>
             */
            void ResetEventCallback()
            {
                mEventHandler.ResetEventCallback();
            }

            void InvokeListEventCallback(const ListEvent &listEvent)
            {
                mEventHandler(listEvent);
            }

        protected:
            /**
                <summary>
                    Invoked when adding or removing a referrer to the object. Combines SharedPointer Retain and
                    Release function into a single virtual function call.
                </summary>
                <param name="addReferrer">true if a referrer is added, false otherwise.</param>
             */
            virtual void OnReferrerUpdate(bool addReferrer) = 0;

            /**
                <summary>Executes the list event action by calling the callback function.</summary>
                <param name="listEvent">The list event.</param>
             */
            virtual void OnListEvent(const ListEvent &listEvent) override
            {
                mEventHandler(listEvent);
            }

        public:
            void Retain()
            {
                OnReferrerUpdate(true);
            }

            void Release()
            {
                OnReferrerUpdate(false);
            }

        private:
            ListEventHandler mEventHandler;

    };

//@}
}

#endif // Courier_DataBinding_SharedAsyncListInterface_h
