//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_ModelBindingSource_h)
#define Courier_DataBinding_ModelBindingSource_h

#include <CanderaWidget/WidgetBase/WidgetMetaInfo.h>
#include <Courier/DataBinding/DataItemAccessor.h>
#include <Courier/Util/Limits.h>
#include <Courier/Visualization/FrameworkWidget.h>
#include <Candera/EngineBase/Common/AbstractNodePointer.h>


namespace Courier {
    /**
        <summary>
            Programmatically creates a binding between the specified data item and widget property. the life time of
            the binding object is controlled by the connected binding source and widget instance. As this function is
            creating a binding between a model data instance and a widget instance, it must be invoked in
            ViewComponent context only.
        </summary>
        <param name="itemKey">The of the source data item.</param>
        <param name="widgetType">Type name of the widget.</param>
        <param name="propertyName">Name of the widget property.</param>
        <param name="widgetInstance">[in,out] the widget instance.</param>
        <param name="typeConvert">(optional) controls if type conversion will be applied.</param>
        <param name="defaultValue">(optional) the default value.</param>
        <returns>true if the binding could be established, false otherwise.</returns>
     */
    extern bool CreateBinding(DataItemKey itemKey,
                              const Char *widgetType,
                              const Char *propertyName,
                              FrameworkWidget *widgetInstance,
                              bool typeConvert = true,
                              const DataItemValue &defaultValue = DataItemValue(),
                              const Candera::AbstractNodePointer& node = Candera::AbstractNodePointer());

    /**
        <summary>
            Programmatically creates a binding between the specified data item and widget property. the life time of
            the binding object is controlled by the connected binding source and widget instance. As this function is
            creating a binding between a model data instance and a widget instance, it must be invoked in
            ViewComponent context only.
        </summary>
        <param name="itemKey">The of the source data item.</param>
        <param name="wpmi">[in,out] Pointer to the widget property meta object.</param>
        <param name="widgetInstance">[in,out] the widget instance.</param>
        <param name="typeConvert">(optional) controls if type conversion will be applied.</param>
        <param name="defaultValue">(optional) the default value.</param>
        <returns>true if the binding could be established, false otherwise.</returns>
     */
    extern bool CreateBinding(DataItemKey itemKey,
                              const Candera::MetaInfo::WidgetPropertyMetaInfo *wpmi,
                              FrameworkWidget *widgetInstance,
                              bool typeConvert = true,
                              const DataItemValue &defaultValue = DataItemValue(),
                              const Candera::AbstractNodePointer& node = Candera::AbstractNodePointer());

//@}
}

namespace Courier { namespace Internal {
/// @addtogroup COURIER_DATABINDING
/// @{

    class ModelBindingSourceDelegationConnector;
    class AsyncModelBindingSource;

    /** defines a BindingSource that provides access to data items stored in the model component */
    class ModelBindingSource {
        public:
            virtual ~ModelBindingSource();

            virtual AsyncModelBindingSource* ToAsyncModelBindingSource()
            {
                return 0;
            }

            /** return the DataItemAccessor of the binding source */
            DataItemKey ItemKey() const {
                return mItemKey;
            }

            /**
                <summary> Gets the identifier of the component subscribed to this object. </summary>
                <returns> The component identifier. </returns>
             */
            ComponentId GetComponentId() const {
                return mComponentId;
            }

            /**
                <summary>Handles data item messages.</summary>
                <param name="msg">The message.</param>
             */
            virtual void OnDataItemMsg(const AbstractDataItemMsg *msg) = 0;

            /**
                <summary>Handles list event messages.</summary>
                <param name="msg">The list event message.</param>
             */
            virtual void OnListEventMsg(const ListEventMsg *msg) = 0;

            /** return the current value of the given data item
                @param itemKey the key of the data item
                @return a DataItemValue object referring to the current item value,
                        an invalid value object if not available. */
            virtual DataItemValue GetDataItemValue(DataItemKey itemKey) = 0;

            /** release all references to DataItemMsg objects */
            virtual void ReleaseDataItemMsgReference();

            virtual UInt16 DataRevision() const {
                return 0;
            }

            /**
                <summary>Creates a binding from the specified data item to widget property.</summary>
                <param name="itemKey">The data item key that is the source of the binding.</param>
                <param name="wmi">[in,out] pointer to the meta info object of the widget property.</param>
                <param name="widgetInstance">[in,out] the concrete widget instance.</param>
                <param name="typeConvert">(optional) specifies if the binding shall use use type conversion.</param>
                <param name="defaultValue">(optional) the default value.</param>
                <returns>true if the binding could be established, false otherwise.</returns>
             */
            virtual bool CreateBinding(DataItemKey itemKey,
                                       const ::Candera::MetaInfo::WidgetPropertyMetaInfo *wmi,
                                       FrameworkWidget *widgetInstance,
                                       bool typeConvert,
                                       const DataItemValue &defaultValue) = 0;

            void ProcessDataItemMsg(const AbstractDataItemMsg *msg);
            void ProcessListEventMsg(const ListEventMsg *msg);
            void ProcessReleaseDataItemMsgReference();

            virtual bool EnlistModelBindingSourceDelegationConnector(ModelBindingSourceDelegationConnector *modelBindingSourceDelegationConnector);
            virtual bool UnlistModelBindingSourceDelegationConnector(ModelBindingSourceDelegationConnector *modelBindingSourceDelegationConnector);

    protected:
            /**
                <summary> Constructor. </summary>
                <param name="itemKey"> Identifier for the item. </param>
             */
            ModelBindingSource(DataItemKey itemKey);

            /**
                <summary>
                    Sets a binding source map affinity. This supplies the binding source with a link to the
                    owning ModelBindingSourceMap.
                </summary>
                <param name="cid">the component identifier.</param>
             */
            void SetBindingSourceAffinity(ComponentId cid) {
                FEATSTD_DEBUG_ASSERT(cid <= ComponentType::CustomerLast);
                mComponentId = cid;
            }

            ModelBindingSourceDelegationConnector* mModelBindingSourceDelegationConnectorHead;

    private:
            ComponentId mComponentId;                           ///< id of the component subscribed to the binding source
            DataItemKey mItemKey;                               ///< item key to binding source associated hierarchy node

            friend class Internal::DataBinding::ComponentData;
            friend bool Courier::CreateBinding(DataItemKey, const Char*, const Char*, FrameworkWidget *, bool, const DataItemValue &defaultValue, const Candera::AbstractNodePointer& node);
            friend bool Courier::CreateBinding(DataItemKey, const Candera::MetaInfo::WidgetPropertyMetaInfo*, FrameworkWidget *, bool, const DataItemValue &defaultValue, const Candera::AbstractNodePointer& node);
    };

    class ModelBindingSourceDelegationConnector
    {
    public:
        virtual ~ModelBindingSourceDelegationConnector()
        {
        }

        virtual void DelegateDataItemMsg(const AbstractDataItemMsg *msg) = 0;
        virtual void DelegateListEventMsg(const ListEventMsg *msg) = 0;
        virtual void DelegateReleaseDataItemMsgReference() = 0;
        virtual void ProcessAddChangeRecord(const AsyncBinding &binding) const = 0;

        ModelBindingSourceDelegationConnector* Next()
        {
            return mNext;
        }

        void SetNext(ModelBindingSourceDelegationConnector* next)
        {
            mNext = next;
        }

    protected:
        ModelBindingSourceDelegationConnector() :
            mNext(0)
        {
        }

    private:
        ModelBindingSourceDelegationConnector* mNext;
    };
//@}
}}   // namespace

#endif // Courier_DataBinding_ModelBindingSource_h
