//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Courier/Platform/Memory.h>
#include <Courier/Diagnostics/Log.h>

#include "ListPropertyType.h"
#include "GenericObjectList.h"
#include "BindableProperty.h"

namespace Courier {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::DataBinding);

#if !defined(COURIER_LIST_ITEM_DESERIALIZATION_BUFFER_SIZE)
    /** Deserialization of list items from the asset requires to copy each list item to a
        temporary buffer. COURIER_LIST_ITEM_DESERIALIZATION_BUFFER_SIZE specifies the
        size in bytes of this temporary buffer.
        If the serialized size of an item execeeds the buffer size, the item de-serialization
        will fail. */
    static const UInt32 cCOURIER_LIST_ITEM_DESERIALIZATION_BUFFER_SIZE = 128;
#else
    static const UInt32 cCOURIER_LIST_ITEM_DESERIALIZATION_BUFFER_SIZE = COURIER_LIST_ITEM_DESERIALIZATION_BUFFER_SIZE;
#endif

    using namespace Internal;
    using namespace Internal::DataBinding;

    // ------------------------------------------------------------------------
    GenericListPropertyType::GenericListPropertyType()
    {
    }

    // ------------------------------------------------------------------------
    GenericListPropertyType::GenericListPropertyType(const SharedPointer<SharedAsyncListInterface> &list)
    {
        mList = list;
    }

    // ------------------------------------------------------------------------
    GenericListPropertyType::GenericListPropertyType(const GenericListPropertyType &prop) : AsyncListInterface(prop)
    {
        mList = prop.mList;
    }

    // ------------------------------------------------------------------------
    DataItemTypeId GenericListPropertyType::ListItemTypeId() const
    {
        return (mList != 0) ? mList->ListItemTypeId() : InvalidDataItemType::Ident();
    }

    // ------------------------------------------------------------------------
    FeatStd::SizeType GenericListPropertyType::Count() const
    {
        return (mList != 0) ? mList->Count() : ListInterfaceBase::UnknownQuantity();
    }

    // ------------------------------------------------------------------------
    FeatStd::SizeType GenericListPropertyType::Capacity() const
    {
        return (mList != 0) ? mList->Capacity() : ListInterfaceBase::UnknownQuantity();
    }

    // ------------------------------------------------------------------------
    bool GenericListPropertyType::EndOfList(FeatStd::SizeType index) const
    {
        return (mList == 0) || mList->EndOfList(index);
    }

    // ------------------------------------------------------------------------
    FeatStd::SizeType GenericListPropertyType::FragmentStartIndex() const
    {
        return (mList == 0) ? 0 : mList->FragmentStartIndex();
    }

    // ------------------------------------------------------------------------
    FeatStd::SizeType GenericListPropertyType::FragmentItemCount() const
    {
        return (mList == 0) ?  0 : mList->FragmentItemCount();
    }

    // ------------------------------------------------------------------------
    bool GenericListPropertyType::Request(FeatStd::SizeType index, FeatStd::SizeType nItems, bool updateOnly)
    {
        return (mList != 0) && mList->Request(index, nItems, updateOnly);
    }

    // ------------------------------------------------------------------------
    bool GenericListPropertyType::RequestChange(FeatStd::SizeType idx, const DataItemValue &item)
    {
        return (mList != 0) && mList->RequestChange(idx, item);
    }

    // ------------------------------------------------------------------------
    bool GenericListPropertyType::RequestChangeDataItem(FeatStd::SizeType idx, DataItemKey itemKey, const DataItemValue &item)
    {
        return (mList != 0) && mList->RequestChangeDataItem(idx, itemKey, item);
    }

    // ------------------------------------------------------------------------
    bool GenericListPropertyType::RequestAdd(FeatStd::SizeType idx, const DataItemValue &item)
    {
        return (mList != 0) && mList->RequestAdd(idx, item);
    }

    // ------------------------------------------------------------------------
    bool GenericListPropertyType::RequestRemove(FeatStd::SizeType idx)
    {
        return (mList != 0) && mList->RequestRemove(idx);
    }

    // ------------------------------------------------------------------------
    bool GenericListPropertyType::RequestMove(FeatStd::SizeType dst, FeatStd::SizeType src)
    {
        return (mList != 0) && mList->RequestMove(dst, src);
    }

    // ------------------------------------------------------------------------
    bool GenericListPropertyType::RequestClear()
    {
        return (mList != 0) && mList->RequestClear();
    }

    // ------------------------------------------------------------------------
    bool GenericListPropertyType::RequestPrefetch(FeatStd::SizeType startIndex, FeatStd::SizeType nItems)
    {
        return (mList != 0) && mList->RequestPrefetch(startIndex, nItems);
    }

    // ------------------------------------------------------------------------
    UInt32 GenericListPropertyType::Flags() const
    {
        return (mList != 0) ? mList->Flags() : UInt32(ListFlags::None);
    }
}

namespace Courier { namespace Internal { namespace DataBinding {
#if defined(CANDERA_META_DESCRIPTION)
    // ------------------------------------------------------------------------
    std::string ComposeListPropertyEditorString(const Char *listItemEditor)
    {
        std::string string("custom://ListEditor?ItemEditor=");
        string += listItemEditor;
        return string;
    }
#endif

    // ------------------------------------------------------------------------
    static SharedPointer<GenericObjectList> AllocObjectList(DataItemTypeId itemTypeId, FeatStd::SizeType nItems)
    {
        SharedPointer<GenericObjectList> objectList(FEATSTD_NEW(GenericObjectList));
        if ((!objectList.PointsToNull()) && (!objectList->Init(nItems, itemTypeId))) {
            // if the list could not be initialized, return a shared pointer reset to 0
            objectList.Release();
        }
        return objectList;
    }

    // ------------------------------------------------------------------------
    bool DeserializeListProperty(SharedPointer<GenericObjectList> &objectList,
                                 const Char *buf,
                                 DataItemTypeId itemTypeId,
                                 DeserializeValueSignature DeserializeValue)
    {
        bool ok = true;

        // lists are prefixed with U+FFFE
        FEATSTD_LINT_NEXT_EXPRESSION(743, "negative character constant required to check UTF8 encoded chars")
        if ((buf[0] == '\xEF') && (buf[1] == '\xBF') && (buf[2] == '\xBE')) {
            UInt32 nItems;
            // parse the number of list items from buffer
            buf = ParseHexValue(buf + 3, nItems);

            // skip ';' token
            FEATSTD_DEBUG_ASSERT(*buf == ';');
            ++buf;

            // if nItems > 0 allocate the generic object list object
            if (nItems > 0) {
                objectList = AllocObjectList(itemTypeId, nItems);
                ok = !objectList.PointsToNull();

                // parse the list items
                for (;;) {
                    // each serialized item is prefixed with its byte count in hex
                    UInt32 itemLength = 0;
                    buf = ParseHexValue(buf, itemLength);

                    FEATSTD_DEBUG_ASSERT(itemLength > 0);

                    // skip token terminating the length information
                    FEATSTD_DEBUG_ASSERT(*buf == ';');
                    ++buf;

                    if (itemLength > 0) {
                        FEATSTD_DEBUG_ASSERT(itemLength < (cCOURIER_LIST_ITEM_DESERIALIZATION_BUFFER_SIZE - 1));

                        //todo: change SC protocol to support serialized strings with embedded 0 bytes
                        // as most DataItem<T>::ConvertFrom functions will assume that the passed serialized
                        // data is 0 terminated, copy the serialized asset data to a temporary buffer and
                        // explicitly 0 terminate the string.
                        static Char deserializeBuffer[cCOURIER_LIST_ITEM_DESERIALIZATION_BUFFER_SIZE];

                        // check buffer size
                        ok = itemLength < (cCOURIER_LIST_ITEM_DESERIALIZATION_BUFFER_SIZE - 1);
                        if (ok) {
                            // copy from buf to deserializeBuffer and terminate with 0 byte
                            FeatStd::Internal::Memory::Copy(deserializeBuffer, buf, itemLength);
                            deserializeBuffer[itemLength] = '\0';
                        }
                        else {
                            COURIER_LOG_WARN("the serialized size of a list item exceeds COURIER_LIST_ITEM_DESERIALIZATION_BUFFER_SIZE");
                        }

                        // let the generic object list parse the value directly from the buffer
                        ok = ok && objectList->PushBack(deserializeBuffer, DeserializeValue);
                    }

                    if ((!ok) || (nItems == 1)) {
                        // terminate on parse error or all items read
                        break;
                    }

                    // decrement number of items
                    --nItems;

                    // preceed to next list item
                    buf += itemLength;
                }
            }
        }
        if ((! ok) && (objectList != 0)) {
            // on failure, delete the list object and return 0
            objectList.Release();
        }
        return ok;
    }
}}}
