//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_ListInterfaceBase_h)
#define Courier_DataBinding_ListInterfaceBase_h

#include <Courier/Util/Limits.h>
#include <Courier/DataBinding/DataBindingFwd.h>
#include <Courier/Diagnostics/Log.h>

namespace Courier {  
/// @addtogroup COURIER_DATABINDING
/// @{

    /**
        <summary>Interface shared between synchronous and asynchronous list interface.</summary>
     */
    class ListInterfaceBase {
        public:
            static const FeatStd::SizeType cInvalidListIndex = Internal::Limits<FeatStd::SizeType>::cMaxValue - 1;
            static const FeatStd::SizeType cUnknownQuantity = Internal::Limits<FeatStd::SizeType>::cMaxValue;
            static const FeatStd::SizeType cUnboundCount = Internal::Limits<FeatStd::SizeType>::cMaxValue;

            /**
                <summary>
                    Defines an alias representing type of the list item. all classes implementing a list interface
                    should provide a ListItemType type. Otherwise DataBinding type system is not able to perform
                    runtime type checks in binding process.
                </summary>
             */
            typedef void ListItemType;

            virtual ~ListInterfaceBase();

            /**
                <summary>
                    Synchronously returns the DataItemTypeId of the list elements
                </summary>
                <returns>DataItemTypeId of the list elements.</returns>
             */
            virtual DataItemTypeId ListItemTypeId() const = 0;

            /**
                <summary>
                    Synchronously gets the current number of elements in the list. if the list is unbound, the return
                    value is cUnboundCount. An unbound list is a list which Count value cannot be determined
                </summary>
                <returns>the number of list items in the list or cUnboundCount.</returns>
             */
            virtual FeatStd::SizeType Count() const = 0;

            /**
                <summary>
                    Synchronously gets the capacity of the list. If the list is unbound, Capacity will return
                    cUnboundCount.
                </summary>
                <returns>the maximum number of elements that can be stored in the list or cUnboundCount.</returns>
             */
            virtual FeatStd::SizeType Capacity() const = 0;

            /**
                <summary>Synchronously returns true if the given index beyond the end of the list.</summary>
                <param name="index">Zero-based index.</param>
                <returns>true if index is beyond end of the list, false otherwise.</returns>
             */
            virtual bool EndOfList(FeatStd::SizeType index) const;


            /**
                <summary>If list data is only available in fragments, the function returns the
                start index of the data fragment. In all other cases the function returns 0.</summary>
                <returns>The fragment start index.</returns>
             */
            virtual FeatStd::SizeType FragmentStartIndex() const;

            /**
                <summary>Gets the fragment end index.</summary>
                <returns>fragment end index.</returns>
             */
            FeatStd::SizeType FragmentEndIndex() const {
                return FragmentStartIndex() + FragmentItemCount();
            }

            /**
                <summary>Gets the fragment list item count.</summary>
                <returns>The fragment list item count.</returns>
             */
            virtual FeatStd::SizeType FragmentItemCount() const;

            /**
                <summary>Returns the invalid list index constant.</summary>
                <returns>constant value for invalid list index.</returns>
             */
            static FeatStd::SizeType InvalidListIndex() {
                return cInvalidListIndex;
            }

            /**
                <summary>Returns the unknown quantity constant.</summary>
                <returns>constant value for unknown quantity.</returns>
             */
            static FeatStd::SizeType UnknownQuantity() {
                return cUnknownQuantity;
            }

            /**
                <summary>Returns the invalid unique key constant.</summary>
                <returns>constant value for invalid unique list item key.</returns>
             */
            static FeatStd::SizeType UnboundCount() {
                return cUnboundCount;
            }
    };

    template<typename T> class TypedListInterfaceBase : public ListInterfaceBase {
        public:
            FEATSTD_LINT_NEXT_EXPRESSION(1516, "intented hiding of base class type")
            typedef T ListItemType;
    };

//@}
}

#endif // Courier_DataBinding_ListInterfaceBase_h
