//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_ListChangedNotifier_h)
#define Courier_DataBinding_ListChangedNotifier_h

#include <Courier/DataBinding/ListEventHandler.h>
#include <Courier/DataBinding/DataBindingMsg.h>

namespace Courier { 

    /**
        <summary>
            List event notifier. Convenience function for model component implementation. The class exposes a full
            set of list event notification functions. The function will result in messages sent to the other data
            consuming components.
        </summary>
     */
    class ListEventNotifier {
        public: 

            /**
                <summary>Constructor.</summary>
                <param name="itemKey">The item key of the list causing the events.</param>
             */
            ListEventNotifier(DataItemKey itemKey) : mItemKey(itemKey) { }

            /**
                <summary>Returns the item key of the list events are emitted for.</summary>
                <returns>Returns the item key of the list events are emitted for.</returns>
             */
            DataItemKey ItemKey() const {
                return mItemKey;
            }

            /**
                <summary>Notifies an item has been added to the list.</summary>
                <param name="itemIndex">Zero-based index of the new item.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool NotifyItemAdded(FeatStd::SizeType itemIndex) const {
                return PostMsg(ListEventType::AddedItem, 
                               itemIndex, 
                               ListInterfaceBase::InvalidListIndex());
            }

            /**
                <summary>Notifies an item has been modified modified.</summary>
                <param name="itemIndex">Zero-based index of the modified item.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool NotifyItemModified(FeatStd::SizeType itemIndex) const {
                return PostMsg(ListEventType::ModifiedItem, 
                               itemIndex, 
                               ListInterfaceBase::InvalidListIndex());
            }

            /**
                <summary>Notifies an item has been removed.</summary>
                <param name="itemIndex">Zero-based index of the removed item.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool NotifyItemRemoved(FeatStd::SizeType itemIndex) const {
                return PostMsg(ListEventType::RemovedItem, 
                               itemIndex, 
                               ListInterfaceBase::InvalidListIndex());
            }

            /**
                <summary>Notifies an item has been moved.</summary>
                <param name="newIndex">New zero-based index of the moved item.</param>
                <param name="oldIndex">Previous zero-based index of the item.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool NotifyItemMoved(FeatStd::SizeType newIndex, FeatStd::SizeType oldIndex) const {
                return PostMsg(ListEventType::MovedItem, 
                               newIndex, 
                               oldIndex);
            }

            /**
                <summary>Notifies data clients to refresh the list.</summary>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool NotifyRefreshList() const {
                return PostMsg(ListEventType::RefreshList, 
                               ListInterfaceBase::InvalidListIndex(), 
                               ListInterfaceBase::InvalidListIndex());
            }

            /**
                <summary>Notifies the list item count has changed.</summary>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool NotifyItemCountChanged() const {
                return PostMsg(ListEventType::ItemCountChanged, 
                               ListInterfaceBase::InvalidListIndex(), 
                               ListInterfaceBase::InvalidListIndex());
            }

            /**
                <summary>Notifies the list has been cleared.</summary>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool NotifyListCleared() const {
                return PostMsg(ListEventType::ListCleared, 
                               ListInterfaceBase::InvalidListIndex(), 
                               ListInterfaceBase::InvalidListIndex());
            }

            /**
                <summary>Notifies the specified event with given parameters.</summary>
                <param name="eventType">Type of the event.</param>
                <param name="newIndex">Zero-based new index.</param>
                <param name="oldIndex">Zero-based old index.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool Notify(ListEventType::Enum eventType, 
                        FeatStd::SizeType newIndex = ListInterfaceBase::InvalidListIndex(), 
                        FeatStd::SizeType oldIndex = ListInterfaceBase::InvalidListIndex()) const;

        private:
            DataItemKey mItemKey;

            /**
                <summary>Posts a list event message.</summary>
                <param name="eventType">Type of the event.</param>
                <param name="newIndex">Zero-based index of the newIndex.</param>
                <param name="oldIndex">Zero-based index of the oldIndex.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool PostMsg(ListEventType::Enum eventType, FeatStd::SizeType newIndex, FeatStd::SizeType oldIndex) const;
    };


}   // namespace 

#endif // Courier_DataBinding_ListChangedNotifier_h
