//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_ChangeRequestSet_h)
#define Courier_DataBinding_ChangeRequestSet_h

#include <Courier/DataBinding/DataBindingFwd.h>
#include <Courier/DataBinding/DataItemValue.h>
#include <Courier/DataBinding/Infrastructure.h>

#ifndef SIZE_OF_VOID_PTR
#define SIZE_OF_VOID_PTR sizeof(void*)
#endif
#include <CanderaPlatform/OS/MemoryPlatform.h>
#include <Courier/DataBinding/AsyncListInterface.h>

namespace Courier {
/// @addtogroup COURIER_DATABINDING
/// @{
    // ========================================================================

    /**
        <summary>
            Request encapsulates a data request sent from a component (controller / view) to
            the model component.
            Requests are received in form of a UpdateModelMsg that may contain multiple requests.
            The container that finally stores requests is Courier::Internal::DataBinding::ChangeSet.
        </summary>
     */
    class Request {
        public:
            /**
                <summary>Query if this object is valid.</summary>
                <returns>true if valid, false if not.</returns>
             */
            bool IsValid() const {
                return mItemKey != cInvalidDataItemKey;
            }

            /**
                <summary>Gets the item key of the data item which is request is addressed to.</summary>
                <returns>DataItemKey of the data item.</returns>
             */
            DataItemKey ItemKey() const {
                return mItemKey;
            }

            /**
                <summary>Gets the key of the associated binding source.</summary>
                <returns>DataItemKey of binding source.</returns>
             */
            DataItemKey BindingSourceKey() const {
                return Internal::DataBinding::DataItemAccessor::Root(mItemKey);
            }

            /**
                <summary>Gets the data revision the request was created for.</summary>
                <returns>data revision number.</returns>
             */
            UInt16 DataRevision() const {
                return mDataRevision;
            }

            /**
                <summary>
                    Gets the request type. Requests towards scalar data items have ListRequestType::None.
                </summary>
                <returns>the request type.</returns>
             */
            ListRequestType::Enum RequestType() const {
                return mRequestType;
            }

            /**
                <summary>
                    Returns the NewIndex parameter of the request.
                    <para>the parameter has following meanings depending on Request::RequestType value</para>
                    <list>
                        <item>ListRequestType::None: no meaning</item>
                        <item>ListRequestType::AddItem: the index of the new list item</item>
                        <item>ListRequestType::ChangeItem: the index of the item that should be modified</item>
                        <item>ListRequestType::RemoveItem: the index of the item to be removed from the list</item>
                        <item>ListRequestType::MoveItem: the new (destination) index of the item to be moved</item>
                        <item>ListRequestType::NewFragment: start of index range that should be sent with a list fragment</item>
                        <item>ListRequestType::ClearList: no meaning</item>
                        <item>ListRequestType::PrefetchItems: the first index of the items that shall be prefetched</item>
                        <item>ListRequestType::ChangeDataItem: the index of the item that should be modified</item>
                        </list>
                </summary>
                <returns>.</returns>
             */
            FeatStd::SizeType NewIndex() const {
                return mNewIndex;
            }

            /**
                <summary>
                    Returns the OldIndex parameter of the request.
                    <para>the parameter has following meanings depending on Request::RequestType value</para>
                    <list>
                        <item>ListRequestType::None: no meaning</item>
                        <item>ListRequestType::AddItem: no meaning</item>
                        <item>ListRequestType::ChangeItem: no meaning</item>
                        <item>ListRequestType::RemoveItem: no meaning</item>
                        <item>ListRequestType::MoveItem: the current (source) index of the list item that shall be moved</item>
                        <item>ListRequestType::NewFragment: the index after the last item that shall be sent with a fragment</item>
                        <item>ListRequestType::ClearList: no meaning</item>
                        <item>ListRequestType::PrefetchItems: the index after the last item that shall be prefetched</item>
                        <item>ListRequestType::ChangeDataItem: no meaning</item>
                    </list>
                </summary>
                <returns>OldIndex parameter.</returns>
             */
            FeatStd::SizeType OldIndex() const {
                return mOldIndex;
            }

            /**
                <summary>
                    Returns the inner data item key of the data item within the list item.
                    <para>the parameter has following meanings depending on Request::RequestType value</para>
                    <list>
                        <item>ListRequestType::None: no meaning</item>
                        <item>ListRequestType::AddItem: no meaning</item>
                        <item>ListRequestType::ChangeItem: no meaning</item>
                        <item>ListRequestType::RemoveItem: no meaning</item>
                        <item>ListRequestType::MoveItem: no meaning</item>
                        <item>ListRequestType::NewFragment: no meaning</item>
                        <item>ListRequestType::ClearList: no meaning</item>
                        <item>ListRequestType::PrefetchItems: no meaning</item>
                        <item>ListRequestType::ChangeDataItem: the inner data item key of the data item within the list item</item>
                    </list>
                </summary>
                <returns>OldIndex parameter.</returns>
             */
            DataItemKey InnerItemKey() const {
                return static_cast<DataItemKey>(mOldIndex);
            }
            /**
                <summary>Gets the list item value.
                    <para>the parameter has following meanings depending on Request::RequestType value</para>
                    <list>
                        <item>ListRequestType::None: no meaning - 0 (null)</item>
                        <item>ListRequestType::AddItem: reference to the list item that shall be added</item>
                        <item>ListRequestType::ChangeItem: reference to the list item that shall be modified</item>
                        <item>ListRequestType::RemoveItem: no meaning - 0 (null)</item>
                        <item>ListRequestType::MoveItem: no meaning - 0 (null)</item>
                        <item>ListRequestType::NewFragment: no meaning - 0 (null)</item>
                        <item>ListRequestType::ClearList: no meaning - 0 (null)</item>
                        <item>ListRequestType::PrefetchItems: no meaning - 0 (null)</item>
                        <item>ListRequestType::ChangeDataItem: the value of the inner data item within the list item</item>
                        </list>
                </summary>
                <returns>null if it fails, else the item.</returns>
             */
            template<typename T> inline const T* GetItem() const {
                return mValue.GetValue<T>();
            }

            /**
                <summary>Gets the item value.
                    <para>the parameter has following meanings depending on Request::RequestType value</para>
                    <list>
                        <item>ListRequestType::None: no meaning - invalid DataItemValue (null)</item>
                        <item>ListRequestType::AddItem: DataItemValue of the list item that shall be added</item>
                        <item>ListRequestType::ChangeItem: DataItemValue of the list item that shall be modified</item>
                        <item>ListRequestType::RemoveItem: no meaning - invalid DataItemValue</item>
                        <item>ListRequestType::MoveItem: no meaning - invalid DataItemValue</item>
                        <item>ListRequestType::NewFragment: no meaning - invalid DataItemValue</item>
                        <item>ListRequestType::ClearList: no meaning - invalid DataItemValue</item>
                        <item>ListRequestType::PrefetchItems: no meaning - invalid DataItemValue</item>
                        <item>ListRequestType::ChangeDataItem: the value of the inner data item within the list item</item>
                        </list>
                </summary>
                <returns>The item value.</returns>
             */
            const DataItemValue& GetItemValue() const {
                return mValue;
            }

        private:
            DataItemKey mItemKey;                       ///< the key of the data item addressed by the request
            UInt16 mDataRevision;                       ///< revision number the request is based on
            ListRequestType::Enum mRequestType;         ///< the request type
            FeatStd::SizeType mNewIndex;                ///< NewIndex parameter
            FeatStd::SizeType mOldIndex;                ///< OldIndex parameter
            DataItemValue mValue;                       ///< DataItemValue to the list item

            /**
                <summary>Request ctor.</summary>
                <param name="itemKey">The item key the request is addressed to.</param>
                <param name="dataRevision">The data revision number.</param>
             */
            Request(DataItemKey itemKey, UInt16 dataRevision);

            /**
                <summary>ItemKey setter.</summary>
                <param name="value">The value.</param>
             */
            void SetItemKey(DataItemKey value) {
                mItemKey = value;
            }

            /**
                <summary>Sets a data revision.</summary>
                <param name="value">The value.</param>
             */
            void SetDataRevision(UInt16 value) {
                mDataRevision = value;
            }

            /**
                <summary>Sets a request type.</summary>
                <param name="value">The value.</param>
             */
            void SetRequestType(ListRequestType::Enum value) {
                mRequestType = value;
            }

            /**
                <summary>Sets the new index parameter.</summary>
                <param name="value">The value.</param>
             */
            void SetNewIndex(FeatStd::SizeType value) {
                mNewIndex = value;
            }

            /**
                <summary>Sets the old index parameter.</summary>
                <param name="value">The value.</param>
             */
            void SetOldIndex(FeatStd::SizeType value) {
                mOldIndex = value;
            }

            /**
                <summary>Sets the item value.</summary>
                <param name="value">The value.</param>
             */
            void SetItemValue(const DataItemValue &value) {
                mValue = value;
            }

            friend class Internal::DataBinding::ChangeSet;
    };

#if !defined(COURIER_CHANGE_SET_BUFFER_SIZE)
    /**
        COURIER_CHANGE_SET_BUFFER_SIZE defines the size of a ChangeSet internal buffer in bytes.
        A ChangeSet is capable to store multiple requests. the number of requests that can be
        stored in a ChangeSet is controlled by this constant.
        A request in a change set has approximately 8 byte administrative data plus the parameter
        data of a request (NewIndex, OldIndex, ItemValue).
        The size of the ChangeSet buffer therefore shall be at least sizeof(largest data item) + 16
    */
    static const UInt32 cCOURIER_CHANGE_SET_BUFFER_SIZE = 256;
    #define COURIER_CHANGE_SET_BUFFER_SIZE Courier::cCOURIER_CHANGE_SET_BUFFER_SIZE
#else
    static const UInt32 cCOURIER_CHANGE_SET_BUFFER_SIZE = COURIER_CHANGE_SET_BUFFER_SIZE;
#endif

//@}
}

namespace Courier { namespace Internal { namespace DataBinding {
/// @addtogroup COURIER_DATABINDING
/// @{
    // ========================================================================

    /**
        <summary>
            ChangeSet is a generic container for request data.
            The size of the buffer ChangeSet is using can be controlled with COURIER_CHANGE_SET_BUFFER_SIZE.
        </summary>
     */
    class ChangeSet {
        public:

            /**
                <summary>Default constructor, creates an empty change set.</summary>
             */
            ChangeSet();

            /**
                <summary>Destructor.</summary>
             */
            ~ChangeSet();

            /**
                <summary>Gets number of requests contained in the ChangeSet.</summary>
                <returns>number of Requests in the ChangeSet.</returns>
             */
            UInt32 Count() const {
                return mRecordCount;
            }

            /**
                <summary>Gets a change request from the ChangeSet.</summary>
                <param name="changeIndex">Zero-based index of the change.</param>
                <returns>The change request. Invalid Request if index is out of bounds.</returns>
             */
            Request GetChangeRequest(FeatStd::SizeType changeIndex) const;

            /**
                <summary>Adds a change record for a non-list data item.</summary>
                <param name="itemKey">The item key of the data item.</param>
                <param name="dataRevision">The data revision.</param>
                <param name="value">The value the data item shall be changed to.</param>
                <returns>true if it succeeds, false if it fails (out of space).</returns>
             */
            bool AddRecord(DataItemKey itemKey, UInt16 dataRevision, const DataItemValue &value);

            /**
                <summary>Adds a non list data item change request.</summary>
                <param name="itemKey">The item key.</param>
                <param name="dataRevision">The data revision.</param>
                <returns>
                    A mutable DataItemValue object that can receive the value of the item. An invalid DataItemValue
                    object if out of space.
                </returns>
             */
            DataItemValue AddRecord(DataItemKey itemKey, UInt16 dataRevision);

            /**
                <summary>Adds a new list item change request.</summary>
                <param name="itemKey">The item key.</param>
                <param name="dataRevision">The data revision.</param>
                <param name="index">Zero-based index of the new list item.</param>
                <param name="value">The list item to be added.</param>
                <returns>true if it succeeds, false if it fails (out of space).</returns>
             */
            bool AddNewListItemRecord(DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType index, const DataItemValue &value);

            /**
                <summary>Adds a change list item request.</summary>
                <param name="itemKey">The item key.</param>
                <param name="dataRevision">The data revision.</param>
                <param name="index">Zero-based index of the list item to change.</param>
                <param name="value">The new list item value.</param>
                <returns>true if it succeeds, false if it fails (out of space).</returns>
             */
            bool AddChangeListItemRecord(DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType index, const DataItemValue &value);

            /**
                <summary>Adds a change list item request.</summary>
                <param name="itemKey">The item key.</param>
                <param name="dataRevision">The data revision.</param>
                <param name="innerItemKey">The inner data item key.</param>
                <param name="itemKey">The item key.</param>
                <param name="value">The new data item within the list item value.</param>
                <returns>true if it succeeds, false if it fails (out of space).</returns>
             */
            bool AddChangeListDataItemRecord(DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType index, FeatStd::SizeType innerItemKey, const DataItemValue &value);

            /**
                <summary>Adds a remove list item record.</summary>
                <param name="itemKey">The item key.</param>
                <param name="dataRevision">The data revision.</param>
                <param name="index">Zero-based index of the item to remove from the list.</param>
                <returns>true if it succeeds, false if it fails (out of space).</returns>
             */
            bool AddRemoveListItemRecord(DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType index);

            /**
                <summary>Adds a move list item record.</summary>
                <param name="itemKey">The item key.</param>
                <param name="dataRevision">The data revision.</param>
                <param name="dst">Destination index of the list item to be moved.</param>
                <param name="src">Source index of the list item to be moved.</param>
                <returns>true if it succeeds, false if it fails (out of space).</returns>
             */
            bool AddMoveListItemRecord(DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType dst, FeatStd::SizeType src);

            /**
                <summary>Adds a clear list record.</summary>
                <param name="itemKey">The item key.</param>
                <param name="dataRevision">The data revision.</param>
                <returns>true if it succeeds, false if it fails (out of space).</returns>
             */
            bool AddClearListRecord(DataItemKey itemKey, UInt16 dataRevision);

            /**
                <summary>Adds a prefetch list request record.</summary>
                <param name="itemKey">The item key.</param>
                <param name="dataRevision">The data revision.</param>
                <param name="startIndex">The start index of items to be prefetched.</param>
                <param name="nItems">The items number of items started with startIndex to be prefetched.</param>
                <returns>true if it succeeds, false if it fails (out of space).</returns>
             */
            bool AddPrefetchListRequestRecord(DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType startIndex, FeatStd::SizeType nItems);

            /**
                <summary>Adds a new fragment list request record.</summary>
                <param name="itemKey">The item key.</param>
                <param name="dataRevision">The data revision.</param>
                <param name="startIndex">The start index of items the fragment shall contain.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool AddNewFragmentListRequestRecord(DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType startIndex, FeatStd::SizeType nItems);

            /**
                <summary>Adds a list request record.</summary>
                <param name="itemKey">The item key.</param>
                <param name="dataRevision">The data revision.</param>
                <param name="requestType">Type of the request.</param>
                <param name="newIndex">NewIndex paramter - see Courier::Request::NewIndex.</param>
                <param name="oldIndex">OldIndex paramter  - see Courier::Request::OldIndex.</param>
                <param name="value">The value - see Courier::Request::GetItemValue.</param>
                <returns>true if it succeeds, false if it fails (out of space).</returns>
             */
            bool AddListRequestRecord(DataItemKey itemKey, UInt16 dataRevision, ListRequestType::Enum requestType,
                                      FeatStd::SizeType newIndex, FeatStd::SizeType oldIndex, const DataItemValue &value);

        private:

            /**
                <summary>
                    Internal administrative data structure. The ChangeSet buffer layout is:
                    <list>
                        <item>at the beginning of mBuffer a ChangeRecord data structure is provisioned for every
                        request</item>
                        <item>the data section is growing from mBuffer end downwards. each entry in the data section
                        consists
                        of a header and option item data (only for ListRequestType::AddItem and
                        ListRequestType::ChangeItem type requests.</item>
                    </list>
                </summary>
             */
            struct ChangeRecord {
                DataItemKey mItemKey;                   ///< the data item key of the request
                UInt16 mDataIndex;                      ///< (UInt32) index of the start of the data section for this request
            };

            /** the buffer size in SIZE_OF_VOID_PTR bit words */
            static const UInt32 cBufferSize = (cCOURIER_CHANGE_SET_BUFFER_SIZE + SIZE_OF_VOID_PTR - 1) / SIZE_OF_VOID_PTR;

            UInt32 mRecordCount;                        ///< number of records in the ChangeSet
            FeatStd::SizeType mBuffer[cBufferSize];                ///< the buffer used to store the request data

            /**
                <summary>Gets the end of data section index for the given change record.</summary>
                <param name="recordIndex">Zero-based index of the record.</param>
                <returns>The data section end index.</returns>
             */
            UInt32 GetDataEndIndex(FeatStd::SizeType recordIndex) const {
                // data section grows from buffer end downd - first end index is therefore cBufferSize
                return (recordIndex == 0) ? cBufferSize : GetChangeRecord(recordIndex - 1)->mDataIndex;
            }

            /**
                <summary>Gets neutralized pointer to the buffer.</summary>
                <returns>always retursn a valid pointer (non-null).</returns>
             */
            inline void* Buffer() {
                return mBuffer;
            }

            /**
                <summary>Gets neutralized pointer to the buffer.</summary>
                <returns>always retursn a valid pointer (non-null).</returns>
             */
            inline const void* Buffer() const {
                return mBuffer;
            }

            /**
                <summary>
                    Gets a pointer to the change record with the given index.
                    Change records grow from the beginning of mBuffer.
                </summary>
                <param name="index">Zero-based index of the.</param>
                <returns>null if it fails (index out of bounds), else the change record.</returns>
             */
            ChangeRecord* GetChangeRecord(FeatStd::SizeType index) {
                return (index <= mRecordCount) ? (FeatStd::Internal::PointerToPointer<ChangeRecord*>(Buffer()) + index) : 0;
            }

            /**
                <summary>
                    Gets a pointer to the change record with the given index.
                    Change records grow from the beginning of mBuffer.
                </summary>
                <param name="index">Zero-based index of the.</param>
                <returns>null if it fails (index out of bounds), else the change record.</returns>
             */
            const ChangeRecord* GetChangeRecord(FeatStd::SizeType index) const {
                return (index <= mRecordCount) ? (FeatStd::Internal::PointerToPointer<const ChangeRecord*>(Buffer()) + index) : 0;
            }

            /**
                <summary>Query if 'itemKey' is a list item.</summary>
                <param name="itemKey">The item key.</param>
                <returns>true if list item, false if not.</returns>
             */
            bool IsListItem(DataItemKey itemKey) const {
                return DataItemAccessor::Node(itemKey)->IsListNode();
            }

            /**
                <summary>Creates a new change record with the given data section size.</summary>
                <param name="itemKey">The item key.</param>
                <param name="dataSize">Size of the data section required by the request.</param>
                <returns>null if it fails (out of space), else pointer to ChangeRecord.</returns>
             */
            ChangeRecord* CreateNewChangeRecord(DataItemKey itemKey, FeatStd::SizeType dataSize);

            /**
                <summary>Intializes the data section for a ListRequestType::AddItem or ListRequestType::ChangeItem request.</summary>
                <param name="requestType">Type of the request.</param>
                <param name="itemKey">The item key.</param>
                <param name="dataRevision">The data revision.</param>
                <param name="index">Zero-based index of the the new list item.</param>
                <param name="value">The value of the new list item.</param>
                <returns>true if it succeeds, false if it fails (out of space).</returns>
             */
            bool AddNewOrModified(ListRequestType::Enum requestType, DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType index, const DataItemValue &value);

            /**
                <summary>Intializes the data section for a ListRequestType::AddItem or ListRequestType::ChangeItem request.</summary>
                <param name="requestType">Type of the request.</param>
                <param name="itemKey">The item key.</param>
                <param name="dataRevision">The data revision.</param>
                <param name="index">Zero-based index of the the new list item.</param>
                <param name="innerItemKey">The inner data item key.</param>
                <param name="value">The value of the new list item.</param>
                <returns>true if it succeeds, false if it fails (out of space).</returns>
             */
            bool AddChangeDataItem(ListRequestType::Enum requestType, DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType index, FeatStd::SizeType innerItemKey, const DataItemValue &value);

            /**
                <summary>
                    Adds a list data section with a single SizeType parameter.
                    This applies to requests of type ListRequestType::RemoveItem and ListRequestType::NewFragment
                </summary>
                <param name="requestType">Type of the request.</param>
                <param name="itemKey">The item key.</param>
                <param name="dataRevision">The data revision.</param>
                <param name="newIndex">NewIndex paramter (see Courier::Request::NewIndex).</param>
                <returns>true if it succeeds, false if it fails (out of space).</returns>
             */
            bool AddListDataHeader1(ListRequestType::Enum requestType, DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType newIndex);

            /**
                <summary>
                    Adds a list data section with a double SizeType parameter.
                    This applies to requests of type ListRequestType::MoveItem and ListRequestType::PrefetchItems
                </summary>
                <param name="requestType">Type of the request.</param>
                <param name="itemKey">The item key.</param>
                <param name="dataRevision">The data revision.</param>
                <param name="newIndex">The newindex.</param>
                <param name="oldIndex">Zero-based index of the old.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool AddListDataHeader2(ListRequestType::Enum requestType, DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType newIndex, FeatStd::SizeType oldIndex);

            ListRequestType::Enum RequestType(const ChangeRecord *cr) const;
            ListRequestType::Enum GetListRequestType(const ChangeRecord *cr) const;

            /**
                <summary>Gets the data header for the given ChangeRecord.</summary>
                <param name="cr">The ChangeRecord.</param>
                <returns>Always returns a valid pointer.</returns>
             */
            template<typename T> inline T* GetDataHeader(const ChangeRecord *cr) {
                return reinterpret_cast<T*>(&mBuffer[cr->mDataIndex]);
            }

            /**
                <summary>Gets the data header for the given ChangeRecord.</summary>
                <param name="cr">The ChangeRecord.</param>
                <returns>Always returns a valid pointer.</returns>
             */
            template<typename T> inline const T* GetDataHeader(const ChangeRecord *cr) const {
                return reinterpret_cast<const T*>(&mBuffer[cr->mDataIndex]);
            }

            friend struct UnitTest::ChangeSetTestAccessor;

            FEATSTD_MAKE_CLASS_UNCOPYABLE(ChangeSet);
    };

//@}
}}}

#endif // Courier_DataBinding_ChangeRequestSet_h
