//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ChangeSet.h"

namespace Courier {
    // ------------------------------------------------------------------------
    Request::Request(DataItemKey itemKey, UInt16 dataRevision) : mItemKey(itemKey), mDataRevision(dataRevision),
                                                                 mRequestType(ListRequestType::None), 
                                                                 mNewIndex(ListInterfaceBase::InvalidListIndex()),
                                                                 mOldIndex(ListInterfaceBase::InvalidListIndex())
    {
    }
}

namespace Courier { namespace Internal { namespace DataBinding {
    FEATSTD_LINT_NEXT_EXPRESSION(948, "operator '==' always evaluates to True  => Lint warning not true for each platform")
    FEATSTD_LINT_NEXT_EXPRESSION(944, "argument for operator '?' always evaluates to True  => Lint warning not true for each platform")
    template<typename T> struct AtomSizer {
        static const UInt32 cAtomCount = (sizeof(T) + SIZE_OF_VOID_PTR - 1) >> ((SIZE_OF_VOID_PTR == 4) ? 2 : 3);
    };

    // ------------------------------------------------------------------------
    static UInt32 ByteToAtomCount(UInt32 byteCount)
    {
        FEATSTD_LINT_NEXT_EXPRESSION(948, "operator '==' always evaluates to True  => Lint warning not true for each platform")
        FEATSTD_LINT_NEXT_EXPRESSION(944, "argument for operator '?' always evaluates to True  => Lint warning not true for each platform")
        return (byteCount + SIZE_OF_VOID_PTR - 1) >> ((SIZE_OF_VOID_PTR == 4) ? 2 : 3);
    }

    // ------------------------------------------------------------------------
    static inline UInt32 TypeAtomCount(DataItemTypeId typeId)
    {
        return ByteToAtomCount(typeId->Size());
    }

    // ========================================================================

    /**
        <summary>header for the data section of a Request stored in ChangeSet.</summary>
     */
    struct DataHeader {
        /**
            <summary>Initialises this object.</summary>
            <param name="headerSize">Size of the header.</param>
            <param name="dataRevision">The data revision.</param>
            <param name="requestType">Type of the request.</param>
         */
        void Init(UInt8 headerSize, UInt16 dataRevision, ListRequestType::Enum requestType);

        /**
            <summary>Gets the list request type.</summary>
            <returns>The list request type.</returns>
         */
        ListRequestType::Enum GetListRequestType() const {
            return ListRequestType::Enum(mListRequestType);
        }

        /**
            <summary>Sets a list request type.</summary>
            <param name="value">The value.</param>
         */
        void SetListRequestType(ListRequestType::Enum value) {
            mListRequestType = UInt8(value);
        }

        /**
            <summary>Gets the header atom count (ie. number of UInt32).</summary>
            <returns>.</returns>
         */
        UInt32 HeaderAtomCount() const {
            return mHeaderAtomCount;
        }

        /**
            <summary>
                Gets the data pointer. The data of a list item (ListRequestType::AddItem and
                ListRequestType::ChangeItem) is stored immediatly after the data header structure.
            </summary>
            <returns>null if it fails, else.</returns>
         */
        void* DataPtr() {
            void *p = this;
            return (FeatStd::Internal::PointerToPointer<FeatStd::SizeType*>(p) + mHeaderAtomCount);
        }

        /**
            <summary>
                Gets the data pointer. The data of a list item (ListRequestType::AddItem and
                ListRequestType::ChangeItem) is stored immediatly after the data header structure.
            </summary>
            <returns>null if it fails, else.</returns>
         */
        const void* DataPtr() const {
            const void *p = this;
            return (FeatStd::Internal::PointerToPointer<const FeatStd::SizeType*>(p) + mHeaderAtomCount);
        }

        UInt8 mListRequestType;                         ///< the list request type
        UInt8 mHeaderAtomCount;                         ///< the size of the header in 4 byte blocks
        UInt16 mDataRevision;                           ///< the data revision the request is based on
    };

    // ------------------------------------------------------------------------
    void DataHeader::Init(UInt8 headerSize, UInt16 dataRevision, ListRequestType::Enum requestType)
    {
        SetListRequestType(requestType);
        mDataRevision = dataRevision;
        mHeaderAtomCount = UInt8(ByteToAtomCount(headerSize));
    }

    // ========================================================================

    FEATSTD_LINT_NEXT_EXPRESSION(1790, "derived class just adds data members, no polymorphism involved")
    /**
        <summary>List data header with one index paramter.</summary>
        <seealso cref="DataHeader"/>
     */
    struct ListDataHeader1 : public DataHeader {
        FeatStd::SizeType mNewIndex;
    };

    // ========================================================================

    FEATSTD_LINT_NEXT_EXPRESSION(1790, "derived class just adds data members, no polymorphism involved")
    /**
        <summary>List data header for two index parameters.</summary>
        <seealso cref="ListDataHeader1"/>
     */
    struct ListDataHeader2 : public ListDataHeader1 {
        FeatStd::SizeType mOldIndex;
    };

    // ========================================================================

    // ------------------------------------------------------------------------
    ChangeSet::ChangeSet() : 
        mRecordCount(0)
    {
        Candera::MemoryPlatform::Set(&mBuffer[0], 0U, sizeof(mBuffer));
    }

    // ------------------------------------------------------------------------
    ChangeSet::~ChangeSet()
    {
        UInt32 i = mRecordCount;                        // inverse ctor sequence
        while (i > 0) {
            --i;
            // get change record
            ChangeRecord *cr = GetChangeRecord(i);
            // get pointer to the data type operators
            DataItemTypeId typeId = DataItemAccessor::Descriptor(cr->mItemKey).TypeId();
            // get data header
            DataHeader *dh = GetDataHeader<DataHeader>(cr);
            if (typeId->IsListType()) {
                if ((dh->GetListRequestType() == ListRequestType::AddItem) || (dh->GetListRequestType() == ListRequestType::ChangeItem)) {
                    // invoke dtor of the list item type
                    typeId->ListItemTypeId()->Destruct(dh->DataPtr());
                }
            }
            else {
                // invoke destructor on the item type
                typeId->Destruct(dh->DataPtr());
            }
        }
    }

    // ------------------------------------------------------------------------
    ListRequestType::Enum ChangeSet::RequestType(const ChangeRecord *cr) const
    {
        return GetDataHeader<DataHeader>(cr)->GetListRequestType();
    }

    // ------------------------------------------------------------------------
    bool ChangeSet::AddListRequestRecord(DataItemKey itemKey, UInt16 dataRevision,
                                         ListRequestType::Enum requestType, FeatStd::SizeType newIndex,
                                         FeatStd::SizeType oldIndex, const DataItemValue &value)
    {
        bool ok;
        switch(requestType) {
            case ListRequestType::AddItem:
                ok = AddNewListItemRecord(itemKey, dataRevision, newIndex, value);
                break;

            case ListRequestType::ChangeItem:
                ok = AddChangeListItemRecord(itemKey, dataRevision, newIndex, value);
                break;

            case ListRequestType::ClearList:
                ok = AddClearListRecord(itemKey, dataRevision);
                break;

            case ListRequestType::MoveItem:
                ok = AddMoveListItemRecord(itemKey, dataRevision, newIndex, oldIndex);
                break;

            case ListRequestType::PrefetchItems:
                ok = AddPrefetchListRequestRecord(itemKey, dataRevision, newIndex, oldIndex);
                break;

            case ListRequestType::NewFragment:
                ok = AddNewFragmentListRequestRecord(itemKey, dataRevision, newIndex, oldIndex);
                break;

            case ListRequestType::RemoveItem:
                ok = AddRemoveListItemRecord(itemKey, dataRevision, newIndex);
                break;

            case ListRequestType::ChangeDataItem:
                ok = AddChangeListDataItemRecord(itemKey, dataRevision, newIndex, oldIndex, value);
                break;

            case ListRequestType::None:
            default:
                ok = false;
                FEATSTD_DEBUG_FAIL();
                break;
        }
        return ok;
    }

    // ------------------------------------------------------------------------
    Request ChangeSet::GetChangeRequest(FeatStd::SizeType changeIndex) const
    {
        Request request(cInvalidDataItemKey, 0);
        if (changeIndex < mRecordCount) {
            const ChangeRecord *cr = GetChangeRecord(changeIndex);
            FEATSTD_DEBUG_ASSERT(cr != 0);

            const DataHeader *dh = GetDataHeader<DataHeader>(cr);
            FEATSTD_DEBUG_ASSERT(dh != 0);

            request.SetItemKey(cr->mItemKey);
            request.SetDataRevision(dh->mDataRevision);
            request.SetRequestType(dh->GetListRequestType());

            DataItemTypeId typeId = DataItemAccessor::Descriptor(cr->mItemKey).TypeId();

            if (typeId->IsListType()) {
                switch(dh->GetListRequestType()) {
                    case ListRequestType::AddItem:
                    case ListRequestType::ChangeItem: {
                        const ListDataHeader1 *ldh = FeatStd::Internal::PointerToPointer<const ListDataHeader1*>(dh);
                        FEATSTD_LINT_NEXT_EXPRESSION(1963, "template and non-template ctor have different semantics and argument list")
                        DataItemValue dataItemValue(typeId->ListItemTypeId(), dh->DataPtr());
                        request.SetItemValue(dataItemValue);
                        request.SetNewIndex(ldh->mNewIndex);
                        break;
                    }

                    case ListRequestType::ChangeDataItem: {
                        const ListDataHeader2 *ldh = FeatStd::Internal::PointerToPointer<const ListDataHeader2*>(dh);
                        request.SetNewIndex(ldh->mNewIndex);
                        // SetOldIndex will set the InnerItemKey
                        request.SetOldIndex(ldh->mOldIndex);
                        FEATSTD_LINT_NEXT_EXPRESSION(1963, "template and non-template ctor have different semantics and argument list")
                        DataItemValue dataItemValue(DataItemAccessor::Descriptor(request.InnerItemKey()).TypeId(), dh->DataPtr());
                        request.SetItemValue(dataItemValue);
                        break;
                    }

                    case ListRequestType::ClearList:
                        break;

                    case ListRequestType::MoveItem:
                    case ListRequestType::PrefetchItems:
                    case ListRequestType::NewFragment: {
                        const ListDataHeader2 *ldh = FeatStd::Internal::PointerToPointer<const ListDataHeader2 *>(dh);
                        request.SetNewIndex(ldh->mNewIndex);
                        request.SetOldIndex(ldh->mOldIndex);
                        break;
                    }

                    case ListRequestType::RemoveItem: {
                        const ListDataHeader1 *ldh = FeatStd::Internal::PointerToPointer<const ListDataHeader1 *>(dh);
                        request.SetNewIndex(ldh->mNewIndex);
                        break;
                    }

                    case ListRequestType::None:
                    default:
                        FEATSTD_DEBUG_FAIL();
                        request.SetRequestType(ListRequestType::None);
                        request.SetItemKey(cInvalidDataItemKey);
                        break;
                }
            }
            else {
                const void *dataPtr = dh->DataPtr();
                FEATSTD_LINT_NEXT_EXPRESSION(1963, "template and non-template ctor have different semantics and argument list")
                DataItemValue dataItemValue(typeId, dataPtr);
                request.SetItemValue(dataItemValue);
            }
        }
        return request;
    }

    // ------------------------------------------------------------------------
    ChangeSet::ChangeRecord* ChangeSet::CreateNewChangeRecord(DataItemKey itemKey, FeatStd::SizeType dataSize)
    {
        // get pointer to next free change record
        ChangeRecord *cr = GetChangeRecord(mRecordCount);
        UInt32 dataIndex = GetDataEndIndex(mRecordCount) - static_cast<UInt32>(dataSize);

        // if data fits into buffer (ChangeRequest and data section don't overlapp), increment number of records
        FEATSTD_LINT_NEXT_EXPRESSION(923, "this cast to long is required")
        FEATSTD_LINT_NEXT_EXPRESSION(925, "this cast is required")
        if ((dataIndex < cBufferSize) && (reinterpret_cast<FeatStd::SizeType>(static_cast<void*>(cr + 1)) <= reinterpret_cast<FeatStd::SizeType>(static_cast<void*>(&mBuffer[dataIndex])))) {
            // initialize change record
            cr->mItemKey = itemKey;
            // calculate the data index depending on the previous record data index minus current data object size
            cr->mDataIndex = UInt16(dataIndex);

            ++mRecordCount;
        }
        else {
            cr = 0;
        }

        return cr;
    }

    // ------------------------------------------------------------------------
    bool ChangeSet::AddRecord(DataItemKey itemKey, UInt16 dataRevision, const DataItemValue &value)
    {
        // get type operators
        DataItemTypeId typeId = value.TypeId();

        // fail on list items, invalid values, or type mismatches
        bool ok = (! IsListItem(itemKey)) &&
                  value.IsValid() &&
                  (typeId == DataItemAccessor::Descriptor(itemKey).TypeId());
        FEATSTD_DEBUG_ASSERT(ok);

        if (ok) {
            typedef DataHeader DhType;
            UInt32 atomCount = TypeAtomCount(typeId) + AtomSizer<DhType>::cAtomCount;

            // get a new change record and initialize it
            ChangeRecord *cr = CreateNewChangeRecord(itemKey, atomCount);
            ok = cr != 0;
            if (ok) {
                DhType *dh = GetDataHeader<DhType>(cr);
                dh->Init(sizeof(DhType), dataRevision, ListRequestType::None);
                // construct data in data section
                void *dataPtr = dh->DataPtr();
                typeId->Construct(dataPtr, value);
                ok = dataPtr != 0;
            }
        }
        return ok;
    }

    // ------------------------------------------------------------------------
    DataItemValue ChangeSet::AddRecord(DataItemKey itemKey, UInt16 dataRevision)
    {
        DataItemValue value;
        // get type operators
        DataItemTypeId typeId = DataItemAccessor::Descriptor(itemKey).TypeId();

        // fail on list items, invalid values, or type mismatches
        bool ok = !typeId->IsListType();
        FEATSTD_DEBUG_ASSERT(ok);

        if (ok) {
            typedef DataHeader DhType;
            UInt32 atomCount = TypeAtomCount(typeId) + AtomSizer<DhType>::cAtomCount;

            // get a new change record and initialize it
            ChangeRecord *cr = CreateNewChangeRecord(itemKey, atomCount);
            ok = cr != 0;
            if (ok) {
                DhType *dh = GetDataHeader<DhType>(cr);
                dh->Init(sizeof(DhType), dataRevision, ListRequestType::None);
                // construct data in data section
                typeId->Construct(dh->DataPtr(), value);
                value = DataItemValue(typeId, dh->DataPtr(), true);
            }
        }
        return value;
    }

    // ------------------------------------------------------------------------
    bool ChangeSet::AddNewOrModified(ListRequestType::Enum requestType,
                                     DataItemKey itemKey,
                                     UInt16 dataRevision,
                                     FeatStd::SizeType index,
                                     const DataItemValue &value)
    {
        DataItemTypeId typeId = value.TypeId();

        // fail on non list items, invalid values, or type mismatches
        bool ok = IsListItem(itemKey) &&
                  value.IsValid() &&
                  (value.TypeId() == DataItemAccessor::Descriptor(itemKey).TypeId()->ListItemTypeId());
        FEATSTD_DEBUG_ASSERT(ok);

        if (ok) {
            typedef ListDataHeader1 DhType;
            UInt32 atomCount = TypeAtomCount(typeId) + AtomSizer<DhType>::cAtomCount;
            ChangeRecord *cr = CreateNewChangeRecord(itemKey, atomCount);
            ok = cr != 0;
            if (ok) {
                DhType *dh = GetDataHeader<DhType>(cr);
                dh->Init(sizeof(DhType), dataRevision, requestType);
                dh->mNewIndex = index;

                typeId->Construct(dh->DataPtr(), value);
            }
        }
        return ok;
    }

    // ------------------------------------------------------------------------
    bool ChangeSet::AddChangeDataItem(ListRequestType::Enum requestType, DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType index, FeatStd::SizeType innerItemKey, const DataItemValue &value)
    {
        DataItemTypeId typeId = value.TypeId();

        // fail on non list items, invalid values, or type mismatches
        bool ok = IsListItem(itemKey) &&
            value.IsValid() &&
            (value.TypeId() == DataItemAccessor::Descriptor(DataItemKey(innerItemKey)).TypeId());
        FEATSTD_DEBUG_ASSERT(ok);

        if (ok) {
            typedef ListDataHeader2 DhType;
            UInt32 atomCount = TypeAtomCount(typeId) + AtomSizer<DhType>::cAtomCount;
            ChangeRecord *cr = CreateNewChangeRecord(itemKey, atomCount);
            ok = cr != 0;
            if (ok) {
                DhType *dh = GetDataHeader<DhType>(cr);
                dh->Init(sizeof(DhType), dataRevision, requestType);
                dh->mNewIndex = index;
                dh->mOldIndex = innerItemKey;
                typeId->Construct(dh->DataPtr(), value);
            }
        }
        return ok;
    }

    // ------------------------------------------------------------------------
    bool ChangeSet::AddNewListItemRecord(DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType index, const DataItemValue &value)
    {
        return AddNewOrModified(ListRequestType::AddItem, itemKey, dataRevision, index, value);
    }

    // ------------------------------------------------------------------------
    bool ChangeSet::AddChangeListItemRecord(DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType index, const DataItemValue &value)
    {
        return AddNewOrModified(ListRequestType::ChangeItem, itemKey, dataRevision, index, value);
    }

    // ------------------------------------------------------------------------
    bool ChangeSet::AddChangeListDataItemRecord(DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType index, FeatStd::SizeType innerItemKey, const DataItemValue &value)
    {
        return AddChangeDataItem(ListRequestType::ChangeDataItem, itemKey, dataRevision, index, innerItemKey, value);
    }

    // ------------------------------------------------------------------------
    bool ChangeSet::AddListDataHeader1(ListRequestType::Enum requestType, DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType newIndex)
    {
        bool ok = IsListItem(itemKey);
        FEATSTD_DEBUG_ASSERT(ok);

        if (ok) {
            typedef ListDataHeader1 DhType;
            ChangeRecord *cr = CreateNewChangeRecord(itemKey, AtomSizer<DhType>::cAtomCount);
            ok = cr != 0;
            if (ok) {
                DhType *dh = GetDataHeader<DhType>(cr);
                dh->Init(sizeof(DhType), dataRevision, requestType);
                dh->mNewIndex = newIndex;
            }
        }
        return ok;
    }

    // ------------------------------------------------------------------------
    bool ChangeSet::AddRemoveListItemRecord(DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType index)
    {
        return AddListDataHeader1(ListRequestType::RemoveItem, itemKey, dataRevision, index);
    }

    // ------------------------------------------------------------------------
    bool ChangeSet::AddListDataHeader2(ListRequestType::Enum requestType, DataItemKey itemKey,
                                       UInt16 dataRevision, FeatStd::SizeType newIndex, FeatStd::SizeType oldIndex)
    {
        bool ok = IsListItem(itemKey);
        FEATSTD_DEBUG_ASSERT(ok);

        if (ok) {
            typedef ListDataHeader2 DhType;
            ChangeRecord *cr = CreateNewChangeRecord(itemKey, AtomSizer<DhType>::cAtomCount);
            ok = cr != 0;
            if (ok) {
                DhType *dh = GetDataHeader<DhType>(cr);
                dh->Init(sizeof(DhType), dataRevision, requestType);
                dh->mNewIndex = newIndex;
                dh->mOldIndex = oldIndex;
            }
        }
        return ok;
    }

    // ------------------------------------------------------------------------
    bool ChangeSet::AddMoveListItemRecord(DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType dst, FeatStd::SizeType src)
    {
        return AddListDataHeader2(ListRequestType::MoveItem, itemKey, dataRevision, dst, src);
    }

    // ------------------------------------------------------------------------
    bool ChangeSet::AddClearListRecord(DataItemKey itemKey, UInt16 dataRevision)
    {
        bool ok = IsListItem(itemKey);
        FEATSTD_DEBUG_ASSERT(ok);

        if (ok) {
            typedef DataHeader DhType;
            ChangeRecord *cr = CreateNewChangeRecord(itemKey, AtomSizer<DhType>::cAtomCount);
            ok = cr != 0;
            if (ok) {
                DhType *dh = GetDataHeader<DhType>(cr);
                dh->Init(sizeof(DhType), dataRevision, ListRequestType::ClearList);
            }
        }
        return ok;
    }

    // ------------------------------------------------------------------------
    bool ChangeSet::AddPrefetchListRequestRecord(DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType startIndex, FeatStd::SizeType nItems)
    {
        return AddListDataHeader2(ListRequestType::PrefetchItems, itemKey, dataRevision, startIndex, startIndex + nItems);
    }

    // ------------------------------------------------------------------------
    bool ChangeSet::AddNewFragmentListRequestRecord(DataItemKey itemKey, UInt16 dataRevision, FeatStd::SizeType startIndex, FeatStd::SizeType nItems)
    {
        return AddListDataHeader2(ListRequestType::NewFragment, itemKey, dataRevision, startIndex, nItems);
    }
}}}
