//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_BindingSourceRevisiionStore_h)
#define Courier_DataBinding_BindingSourceRevisiionStore_h

#include <Courier/DataBinding/Infrastructure.h>

namespace Courier {
    /// @addtogroup COURIER_DATABINDING
    /// @{
    /**
    <summary>
    Binding source revision store. Maintains for each binding source a data revision number. The revision
    number is attached to all data binding messages from the model to other components. Change requests
    towards the model component carry the data revision number on which the changes should be done.
    It is up to the model to decide how change requests on old data revisions shall be handled.

    </summary>
    */
    class BindingSourceRevisionStore {
    public:
        /**
        <summary>Gets a revision for the specified binding source item key.</summary>
        <param name="bindingSourceItemKey">The binding source item key.</param>
        <returns>The revision binding source current revision number.</returns>
        */
        static UInt16 GetRevision(DataItemKey bindingSourceItemKey);

        /**
        <summary>Sets a binding source data revision number to the specified value.</summary>
        <param name="bindingSourceItemKey">The binding source item key.</param>
        <param name="dataRevision">The data revision.</param>
        */
        static void SetRevision(DataItemKey bindingSourceItemKey, UInt16 dataRevision);

        /**
        <summary>Increment the binding source data revision number.</summary>
        <param name="bindingSourceItemKey">The binding source item key.</param>
        <returns>The incremented revision number.</returns>
        */
        static UInt16 IncRevision(DataItemKey bindingSourceItemKey);

        /**
        <summary>
        Compares two data revision numbers to determine their relative ordering. the compare function
        handles number overflow. It is assumed, that a revision number faced an overflow if the difference
        of the two numbers is larger than 32767.
        </summary>
        <param name="revisionLeft">The left revision number in the comparison.</param>
        <param name="revisionRight">The right revision number in the comparison.</param>
        <returns>
        Negative if 'revisionLeft' is less than 'revisionRight', 0 if they are equal, or positive if it
        is greater.
        </returns>
        */
        static Int32 Compare(UInt16 revisionLeft, UInt16 revisionRight);

    protected:
        BindingSourceRevisionStore() {
            FEATSTD_DEBUG_ASSERT(mInstance == 0);
            FEATSTD_LINT_CURRENT_SCOPE(1938, "required to get global singleton instance")
                if (mInstance == 0) {                   // there can only be one revision store in the system
                    mInstance = this;
                }
        }

        ~BindingSourceRevisionStore() {
            FEATSTD_DEBUG_ASSERT(mInstance != 0);
            if (mInstance == this) {                // clear the instance pointer only if the object previously set it
                mInstance = 0;
            }
        }

        /**
        <summary>The polymorphic implementation of GetRevision.</summary>
        <param name="bindingSourceIndex">The binding source item index.</param>
        <returns>The revision binding source current revision number.</returns>
        */
        virtual UInt16 GetRevisionImpl(BindingSourceIndex bindingSourceIndex) const = 0;

        /**
        <summary>
        The polymorphic implementation of  SetRevision. Sets a binding source data revision number to the
        specified value.
        </summary>
        <param name="bindingSourceIndex">The binding source item index.</param>
        <param name="dataRevision">The data revision.</param>
        */
        virtual void SetRevisionImpl(BindingSourceIndex bindingSourceIndex, UInt16 dataRevision) = 0;

    private:
        static BindingSourceRevisionStore *mInstance;
    };

    /** compares two revision numbers including overflow handling.
    see also Courier::BindingSourceRevisionStore::Compare.
    @param revisionLeft left revision number in the comparision
    @param revisionRight right revision number in the comparision
    @return value less then 0 if revisionLeft < revisionRight, 0 on equality,
    value larger than 0 if revisionLeft > revisionRight */
    inline static Int32 CompareRevisions(UInt16 revisionLeft, UInt16 revisionRight)
    {
        return BindingSourceRevisionStore::Compare(revisionLeft, revisionRight);
    }

    namespace Internal { namespace DataBinding {
        FEATSTD_LINT_CTORDTOR_EFFECT(Courier::Internal::DataBinding::BindingSourceRevisionStoreImpl<*>)

            /**
            <summary>
            Binding source revision store implementation. The template argument specifies the total number of
            binding sources in the system.
            </summary>
            <seealso cref="BindingSourceRevisionStore"/>
            */
            template<UInt16 bindingSourceCount> class BindingSourceRevisionStoreImpl : public BindingSourceRevisionStore {
            public:
                static const UInt16 cBindingSourceCount = bindingSourceCount;

                /**
                <summary>Default constructor. Sets the instance pointer in BindingSourceRevisionStore</summary>
                */
                BindingSourceRevisionStoreImpl() {
                    FeatStd::Internal::Memory::Set(mRevision, 0, sizeof(mRevision));
                }

            protected:
                virtual UInt16 GetRevisionImpl(BindingSourceIndex bindingSourceIndex) const {
                    FEATSTD_DEBUG_ASSERT(bindingSourceIndex < cBindingSourceCount);
                    return (bindingSourceIndex < cBindingSourceCount) ? mRevision[bindingSourceIndex] : 0;
                }

                virtual void SetRevisionImpl(BindingSourceIndex bindingSourceIndex, UInt16 dataRevision) {
                    FEATSTD_DEBUG_ASSERT(bindingSourceIndex < cBindingSourceCount);
                    if (bindingSourceIndex < cBindingSourceCount) {
                        mRevision[bindingSourceIndex] = dataRevision;
                    }
                }

            private:
                UInt16 mRevision[cBindingSourceCount];  ///< The revision number array
        };

        //@}
    }}
}   // namespace

#endif // Courier_DataBinding_BindingSourceRevisiionStore_h
