//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_BindableProperty_h)
#define Courier_DataBinding_BindableProperty_h

FEATSTD_LINT_FILE(967, BindableProperty.h, "false positive - file has valid include guards.")

#include <Candera/System/MetaInfo/DataType.h>
#include <Candera/System/MetaInfo/PublicMacros.h>
#include <Candera/System/MetaInfo/MetaInfo.h>

#include <Courier/Base.h>
#include <Courier/Util/Traits.h>
#include <Courier/DataBinding/DataItemType.h>
#include <Courier/DataBinding/DataItemValue.h>
#include <Courier/DataBinding/TypeConverter.h>
#include <Courier/DataBinding/ListPropertyType.h>
#include <Courier/DataBinding/TypedListInterfaces.h>
#include <Candera/System/MetaInfo/BaseInternalMacros.h>

#if defined(__ghs__)
#pragma ghs nowarning 1931
#endif
namespace Courier {

class FrameworkWidget;

namespace Internal { namespace DataBinding {
/// @addtogroup COURIER_DATABINDING
/// @{
    // ========================================================================

    /** BindablePropertyCapability extends widget properties with data binding functionality
        Bindable properties return an instance of this capability object to provide extended
        property functionality.
        BindablePropertyCapability extends the normal capability object with functions to get
        and set the property value with a DataItemValue object and to retrieve the property
        type information. */
    class BindablePropertyCapability : public Candera::MetaInfo::PropertyCapability {
        public:
            /** returns this capability name */
            static const Char* Name() {
                return "urn://fme.fujitsu.com/Bindable";
            }

            /** returns true, if the given PropertyCapability object is of type BindablePropertyCapability
                @param cap the capability object
                @return if cap can be casted to BindablePropertyCapability, false otherwise */
            static bool IsBindable(const Candera::MetaInfo::PropertyCapability *cap) {
                return (cap != 0) && ((cap->CapabilityName() == Name()) || (FeatStd::Internal::String::CompareStrings(cap->CapabilityName(), Name()) == 0));
            }

            /** safe cast from given capability object to a BindablePropertyCapability object
                @param cap the capability object
                @return cap casted to BindablePropertyCapability or 0 (null) if cap is not Bindable */
            static const BindablePropertyCapability* ConvertFrom(const Candera::MetaInfo::PropertyCapability *cap) {
                return IsBindable(cap) ? static_cast<const BindablePropertyCapability*>(cap) : 0;
            }

            /** returns the capability name identifier */
            virtual const Char* CapabilityName() const {
                return Name();
            }

            /** reads property value from the property associated with this capability object
                @param object the widet instance owning the property
                @param to a mutable DataItemValue that receives the property value
                @param Convert an optional conversion function (see TypeConverter), may be 0 (null)
                @return true if the value could be retrieved, false if i.e. the DataItemValue was immutable or invalid */
            virtual bool Get(const FrameworkWidget *object, const DataItemValue &to, ::Courier::ConvertSignature Convert) const = 0;

            /** writes property value to the property associated with this capability object
                @param object the widet instance owning the property
                @param from a DataItemValue that refers to the property value to be written
                @param Convert an optional conversion function (see TypeConverter), may be 0 (null)
                @return true if the value could be written, false if i.e. the DataItemValue is invalid or types did not match */
            virtual bool Set(FrameworkWidget *object, const DataItemValue &from, ::Courier::ConvertSignature Convert) const = 0;

            virtual bool SetFrom(FrameworkWidget *target, const FrameworkWidget *source, const BindablePropertyCapability &cap) const = 0;

            /**
                <summary>
                    If the target does not accept a type, the source can query an accepted type to lookup a type
                    converter.
                </summary>
                <returns>type id of an accepted type.</returns>
             */
            virtual DataItemTypeId TypeId() const = 0;

        protected:
            bool SetFrom(FrameworkWidget *target, const FrameworkWidget *source, const DataItemValue& dataItemValue, const BindablePropertyCapability &cap) const
            {
                ConvertSignature convert = 0;
                if (TypeId() != cap.TypeId()) {
                    TypeConverterBase *tc = TypeConverterRegistrar::Locate(TypeId(), cap.TypeId());
                    if (tc == 0) {
                        return false;
                    }
                    convert = tc->Vtbl().Convert;
                }
                return cap.Get(source, dataItemValue, convert) && Set(target, dataItemValue, 0);
            }
    };

    /** casts given capability object to BindablePropertyCapability
        see BindablePropertyCapability::ConvertFrom
        @param cap the capability object
        @return a BindablePropertyCapability object or 0 (null) */
    inline const BindablePropertyCapability* Cast(const Candera::MetaInfo::PropertyCapability *cap) {
        return BindablePropertyCapability::ConvertFrom(cap);
    }

    /** Checks bindable properties type restrictions
        Invalid types for bindable properties:
        - arrays
        - pointer
        - references
        @tparam T the type to check */
    template<typename T> struct BindingTypeTypeCheck {
        // if you receive a compile error here, you defined a property with a pointer, array, or reference type.
        // DataBinding can not deal with this property type as ownership of the referred data is cannot be
        // handled. Use a reference type (an object that controls the lifetime of the referred data) to pass that
        // kind of data to bindable properties (see also DataBinding tutorial - reference types).
        typedef typename Internal::VolatileRemover<T>::Type UnqualifiedType;
        enum {
            /* LogicalNot is necessary to avoid "conditional expression is constant" issued by some compilers */
            Value = Internal::LogicalNot<Internal::LogicalOr<Internal::ArrayTrait<UnqualifiedType>::IsArray,
                                                             Internal::PointerTrait<UnqualifiedType>::IsPointer,
                                                             Internal::ReferenceTrait<UnqualifiedType>::IsReference
                                                            >::Value
                                        >::Value
        };
    };

    // ========================================================================

    /** concrete implementation of BindablePropertyCapability for a widget elementary property
        @param WidgetType the widget type onwing the property
        @param PropType the type of the property
        @param PropMetaType the type of the property meta information */
    template<typename WidgetType, typename PropType, typename PropMetaType>
    class ElementaryCapability : public BindablePropertyCapability {
        public:
            virtual bool Get(const ::Courier::FrameworkWidget *object, const DataItemValue &to, ConvertSignature Convert) const override {
                FEATSTD_COMPILETIME_ASSERT(BindingTypeTypeCheck<PropType>::Value);
                bool ok;
                if (Convert == 0) {                     // no conversion function supplied, call getter directly
                    ok = PropMetaType::GetterFn(static_cast<const WidgetType*>(object), to);
                }
                else {                                  // conversion function supplied
                    PropType val;                       // get property value to a temporary and
                    DataItemValue div(&val, true);      // invoke conversion function
                    ok = PropMetaType::GetterFn(static_cast<const WidgetType*>(object), div) &&
                         Convert(to, div);
                }
                return ok;
            }

            virtual bool Set(::Courier::FrameworkWidget *object, const DataItemValue &from, ConvertSignature Convert) const override {
                bool ok = false;
                if (Convert == 0) {                     // no conversion function supplied, get pointer to value from DataItemValue
                    const PropType *valPtr = from.GetValue<PropType>();
                    ok = valPtr != 0;                   // will be 0 if types don't match
                    if (ok) {                           // if value pointer could be retrieved, invoke setter function
                        PropMetaType::SetterFn(static_cast<WidgetType*>(object), *valPtr);
                    }
                }
                else {                                  // conversion function supplied,
                    PropType val;                       // temporary for converted value
                    DataItemValue div(&val, true);      // create a mutable item value referring to the temporary
                    ok = PropMetaType::GetterFn(static_cast<const WidgetType*>(object), div); // first get the original value to keep internal settings that may be required
                    ok = Convert(div, from);            // convert given value to temporary property value
                    if (ok) {                           // if conversion succeeded, invoke setter function
                        PropMetaType::SetterFn(static_cast<WidgetType*>(object), val);
                    }
                }
                return ok;
            }

            virtual bool SetFrom(FrameworkWidget *target, const FrameworkWidget *source, const BindablePropertyCapability &cap) const override {
                PropType val;
                DataItemValue dataItemValue(&val, true);
                return BindablePropertyCapability::SetFrom(target, source, dataItemValue, cap);
            }

            virtual DataItemTypeId TypeId() const override {
                return COURIER_DATA_ITEM_TYPENAME_ID(PropType);
            }

            /** returns the instance of the concrete capability object */
            static const ElementaryCapability* Instance() {
                static const ElementaryCapability capability;
                return &capability;
            }

        private:
            ElementaryCapability() {
            }
    };

    // ========================================================================

    /** concrete implementation of BindablePropertyCapability for a widget list property
        @param WidgetType the widget type onwing the property
        @param PropType the type of the property
        @param PropMetaType the type of the property meta information */
    template<typename WidgetType, typename ListItemType, typename PropMetaType>
    class ListCapability : public BindablePropertyCapability {
        public:
            virtual bool Get(const FrameworkWidget *object, const DataItemValue &to, ConvertSignature) const {
                BindingTypeTypeCheck<ListItemType>();
                FEATSTD_UNUSED2(object, to);
                return false;
            }

            virtual bool Set(FrameworkWidget *object, const DataItemValue &from, ConvertSignature) const {
                const SharedPointer<SharedAsyncListInterface> *p = from.GetValue<SharedPointer<SharedAsyncListInterface> >();
                if (p != 0) {
                    ListPropertyType<ListItemType> propVal(*p);
                    PropMetaType::SetterFn(static_cast<WidgetType*>(object), propVal);
                }
                return p != 0;
            }

            virtual bool SetFrom(FrameworkWidget *target, const FrameworkWidget *source, const BindablePropertyCapability &cap) const {
                ListItemType val;
                DataItemValue dataItemValue(&val, true);
                return BindablePropertyCapability::SetFrom(target, source, dataItemValue, cap);
            }

            virtual DataItemTypeId TypeId() const {
                return COURIER_DATA_ITEM_TYPENAME_ID(TypedSharedAsyncListInterface<ListItemType>);
            }

            /** returns the instance of the concrete capability object */
            static const ListCapability* Instance() {
                static const ListCapability capability;
                return &capability;
            }

        private:
            ListCapability() {
            }
    };

    template<typename WidgetType, typename PropType, typename PropMetaType> struct CapabilityTypeSelector {
        typedef ElementaryCapability<WidgetType, PropType, PropMetaType> Type;
    };

    template<typename WidgetType, typename T, typename PropMetaType> struct CapabilityTypeSelector<WidgetType, ListPropertyType<T>, PropMetaType> {
        typedef ListCapability<WidgetType, T, PropMetaType> Type;
    };

    /**
        <summary>Parse hash value (A-F,a-f,0-9).</summary>
        <param name="buffer">The buffer storing the ascii represenation of the hash value.</param>
        <param name="hashValue">[in,out] The hash value.</param>
        <returns>position of first character after the hash value</returns>
     */
    const Candera::Char* ParseHexValue(const Candera::Char *buffer, UInt32 &hashValue);

    /**
        <summary>Deserialize bindable property.</summary>
        <param name="buffer">The buffer holding the serialized bindable property value.</param>
        <param name="widget">[in,out] the widget instance.</param>
        <param name="wpmi">The widget property meta info object of the property.</param>
        <returns>
            number of bindings establised. if not all bindings could be establised, the inverted number of establised
            bindings is returned (ie. -2 for 2 bindings could be established).
        </returns>
     */
    Int32 DeserializeDataBindingProperty(const Char *buffer, Courier::FrameworkWidget *widget, const Candera::MetaInfo::WidgetPropertyMetaInfo *wpmi);

    /** defines the DataBinding property for the given widget type
        data bindings are defined in SceneComposer by the "virtual" property "Data Binding Definitions".
        This property is added automatically to the set of widget properties whenever a bindable property
        has been defined with CdaBindableProperty.
        The data binding property does not require getter or setter methods, as its sole purpose is to
        de-serialized binding information for bindings to other properties.
        The data binding property must be edited with its own custom editor. This custom editor receives a list
        of all widget properties that are bindable. */
    class BindablePropertyMetaInfoBase : public Candera::MetaInfo::WidgetPropertyMetaInfo {
        public:
            BindablePropertyMetaInfoBase();

            /** the setter only de-serializes the binding information
                no setter method on the widget will be invoked. */
            virtual bool Set(Candera::WidgetBase *object, const Candera::Char *value);

            /** animations are not applicable to this property
                @return returns always 0*/
            virtual Candera::Animation::AnimationPropertySetter::SharedPointer CreateAnimationPropertySetter(Candera::WidgetBase*, Candera::Int, Candera::Int*);

            /** this method will be invoked in SceneComposer scenario only to retrieve the default value of the property
                the data binding property has no default, thus the an empty string will be returned to SC */
            virtual bool Get(const Candera::WidgetBase *object, Candera::Char *buf, Candera::UInt buf_size);

            static const Char* GetBindablePropertyName();

#ifdef CANDERA_META_DESCRIPTION
            /** grant the property its own category */
            virtual const Char* GetCategory() const;

            /** define a readable name for the property */
            virtual const Char* GetReadableName() const;

            /** supply property documentation */
            virtual const Char *GetDescription() const;

            /** returns the editor string for the data binding property
                the editor urn includes also a list of widget properties that are bindable
                @return the custom editor string */
            virtual const Candera::Char* GetEditor() const;
#endif
    };

    template<typename CurrentWidget> class BindablePropertyMetaInfo : public BindablePropertyMetaInfoBase {
        public:
    };

    template<typename T> struct BindablePropertyDetector {
        enum { Value = BindablePropertyDetector<typename T::TypeAggregator>::Value };
    };

    template<typename T1, typename T2> struct BindablePropertyDetector< Candera::MetaInfo::Aggregatable<T1, T2> > {
        enum { Value = Courier::Internal::IsDerivedFrom<T1, BindablePropertyMetaInfoBase>::Value };
    };

    template<typename T1, typename T2> struct BindablePropertyDetector< Candera::MetaInfo::TypeAggregator<T1, T2> > {
        enum { 
            Value = Courier::Internal::LogicalOr<BindablePropertyDetector<T1>::Value,
                                                 BindablePropertyDetector<T2>::Value>::Value
        };
    };

    template<typename BaseType> struct BindablePropertyDetector< Candera::MetaInfo::NullAggregatable<BaseType> > {
        enum { Value = false };
    };

    /** Injects the DataBinding property meta info type to the aggregated list of widget property types.

        As the DataBinding property meta info must only added once, the type aggregation is first checked
        if the property has already been added before. This means we need to dive into type iterating
        business.

        Background:

        For each widget property a type derived from WidgetPropertyMetaInfo is
        declared. These property types are aggregated with Candera::MetaInfo::TypeAggregator. The TypeAggregator
        takes two types that exposes an Aggregatable interface and defines a new type that exposes certain
        function on that types (registration etc.). A (simplified) sample on a generated type aggregations looks
        like this:

        @code
        TypeAggregator&lt;
            TypeAggregator&lt;NullAggregatableType, FirstPropertyMetaInfoObject&gt;, SecondPropertyMetaInfoObject
        &gt;
        @endcode

        This describes the aggregated property meta info of two properties. The first template paramater is
        again a TypeAggregator with FirstProperty meta information type. This way the meta info object types
        are aggregated in a nested hierarchy.

        As the types passed to TypeAggregator must expose an Aggregatable interface, property meta info types
        added to the type aggregation are wrapped into Aggregatable<T>.

        To detect if the bindable property already has been injected to the property meta info aggregation,
        we first to tag the binable property. The tag is BindablePropertyTag which is used as base class for
        BindablePropertyMetaInfoBase. The type trait Courier::Internal::IsDerivedFromwe can detect if a type
        has been derived from from a specified type.

        All that needs to be done is to iterate through the property meta info type aggregation and to check
        each properties WidgetPropertyMetaInfo if it has been derived from BindablePropertyTag. */
    template<typename CurrentWidget, typename PropBaseType, typename T, bool cDataBindingPropertyDefined = BindablePropertyDetector<T>::Value>
    struct BindingPropertyInjector {
        typedef T Type;
    };

    template<typename CurrentWidget, typename PropBaseType, typename T>
    struct BindingPropertyInjector<CurrentWidget, PropBaseType, T, false> {
        Candera::MetaInfo::Internal::AggregationInstanceRef<BindablePropertyMetaInfo<CurrentWidget> > _CdaUniqueName(Item);
    };

//@}
}}} // namespace

/// @addtogroup COURIER_DATABINDING
/// @{
    // ========================================================================

/** extends Cda macros with bindable support
    @param PropName the name of the property
    @param WidgetType the type of the widget hosting the property
    @param PropType the type of the property
    @param PropGetter getter function to read the value of the property
    @param PropSetter getter function to write the value of the property */
#define CourierPrivateBindingCapability(PropName, WidgetType, PropType, PropGetter, PropSetter) \
        static inline bool GetterFn(const WidgetType *widget, const ::Courier::DataItemValue &dst) { \
            return dst.SetValue(widget->PropGetter()); \
        } \
        static inline void SetterFn(WidgetType *widget, const PropType &value) { \
            widget->PropSetter(value); \
        } \
        virtual const ::Candera::MetaInfo::PropertyCapability* GetCapability() const { \
            typedef typename ::Courier::Internal::DataBinding::CapabilityTypeSelector<WidgetType, PropType, Cda_##PropName>::Type CapType; \
            return CapType::Instance(); \
        }

/** define a bindable property.
    use as CdaProperty macro
    @param name the property name
    @param type the property type
    @param getter the getter function to read the property value
    @param setter the setter function to read the property value */
#define CdaBindableProperty(name, type, getter, setter) \
        CdaProperty(name, type, getter, setter) \
        CourierPrivateBindingCapability(name, CurrentType, type, getter, setter)

/** ends the definition of a bindable property */
#define CdaBindablePropertyEnd() CdaPropertyEnd()

//@}

#if defined(__ghs__)
#pragma ghs endnowarning
#endif

#endif // Courier_DataBinding_BindableProperty_h
