//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Candera/System/MetaInfo/DataType.h>
#include <Candera/System/MetaInfo/PublicMacros.h>
#include <Candera/System/MetaInfo/MetaInfo.h>
#include <Courier/Diagnostics/Log.h>
#include <FeatStd/Platform/String.h>

#include "ModelBindingSource.h"
#include "BindableProperty.h"
#include "MetaInfoUtil.h"

#if defined(CANDERA_META_DESCRIPTION)
#   include <string>
#   include <map>
#endif
#include <CanderaBehavior/BehaviorBase/Behavior.h>

namespace Courier { namespace Internal { namespace DataBinding {
    using namespace Internal;

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::DataBinding);

    // ========================================================================

#if defined(CANDERA_META_DESCRIPTION)
    /**
        <summary>Helper function to compose data binding editor string.</summary>
        <param name="wpmi">The Widget Property MetaInfo object to create the editor string for.</param>
        <returns>the editor string for the given widget property.</returns>
     */
    static std::string ComposeDataBindingEditorString(const Candera::MetaInfo::WidgetPropertyMetaInfo *wpmi)
    {
        using namespace Candera::MetaInfo;
        FEATSTD_DEBUG_ASSERT(wpmi != 0);

        const Candera::MetaInfo::WidgetMetaInfo *wmi = GetWidgetMetaInfo(wpmi);
        FEATSTD_DEBUG_ASSERT(wmi != 0);

        bool firstProperty = true;
        std::string editorString("custom://DataBindingEditor?WidgetName=");
        editorString += wmi->GetName();

        for (Int32 i = 0; i < wmi->GetItemCount(); ++i) {
            wpmi = wmi->GetItem(i);
            const PropertyCapability *cap = wpmi->GetCapability();
            if ((cap != 0) && BindablePropertyCapability::IsBindable(cap)) {
                const BindablePropertyCapability *bindableCap = BindablePropertyCapability::ConvertFrom(cap);
                if (firstProperty) {
                    editorString += ",Properties=";
                    firstProperty = false;
                }
                else {
                    editorString += "+";
                }
                editorString += wpmi->GetName();
                editorString += ":IsList=";
                editorString += (bindableCap->TypeId()->IsListType()) ? "1" : "0";
            }
        }
        return editorString;
    }

    const Candera::Char* BindablePropertyMetaInfoBase::GetEditor() const
    {
        static std::map<const void*, std::string> sEditorStrings;
        std::map<const void*, std::string>::iterator editorString = sEditorStrings.find(this);
        if (editorString == sEditorStrings.end()) {
            sEditorStrings[this] = Courier::Internal::DataBinding::ComposeDataBindingEditorString(this);
            editorString = sEditorStrings.find(this);
        }
        return editorString->second.c_str();
    }

#endif

    // ------------------------------------------------------------------------
    const Candera::Char* ParseHexValue(const Candera::Char *buffer, UInt32 &hashValue)
    {
        const Candera::Char *p = buffer;
        hashValue = 0;
        for (;;) {
            Candera::Char v;
            if ((*p >= '0') && (*p <= '9')) {
                v = *p - '0';
            }
            else if ((*p >= 'A') && (*p <= 'F')) {
                v = (*p - 'A') + '\x0A';
            }
            else if ((*p >= 'a') && (*p <= 'f')) {
                v = (*p - 'a') + '\x0A';
            }
            else {
                break;
            }
            hashValue <<= 4U;
            hashValue += UInt32(UInt8(v));
            ++p;
            if (FeatStd::Internal::PointerDiff(p, buffer) >= 8) {
                break;
            }
        }
        return p;
    }

    // ------------------------------------------------------------------------
    Int32 DeserializeDataBindingProperty(const Candera::Char *buffer,
                                         Courier::FrameworkWidget *widget,
                                         const Candera::MetaInfo::WidgetPropertyMetaInfo *wpmi)
    {
        // lookup the widget meta info object
        const Candera::MetaInfo::WidgetMetaInfo *wmi = GetWidgetMetaInfo(wpmi);
        FEATSTD_DEBUG_ASSERT((wmi != 0) && (buffer != 0));

        bool succeeded = true;
        bool ok = (wmi != 0) && (buffer != 0);
        Int32 bindingCount = 0;

        FEATSTD_LINT_SYMBOL(613, wmi, "wmi is checked for 0 at function entry")

        while(ok && (*buffer != '\0')) {
            // parse the name - delimted by @ from data item hash value
            const Char *propertyName = buffer;
            while ((*buffer != '\0') && (*buffer != '@')) {
                ++buffer;
            }

            // lookup the property with propertyName in widget meta info object
            FeatStd::OffsetType nameLen = FeatStd::Internal::PointerDiff(buffer, propertyName);
            FEATSTD_DEBUG_ASSERT(nameLen >= 0);
            wpmi = 0;
            for (Int32 i = 0; (wpmi == 0) && (i < wmi->GetItemCount()); ++i) {
                const Candera::MetaInfo::WidgetPropertyMetaInfo *wpmiTmp = wmi->GetItem(i);

                const Char *wpmiTmpName = (wpmiTmp != 0) ? wpmiTmp->GetName() : "";
                FeatStd::SizeType nameLenSize = static_cast<FeatStd::SizeType>(nameLen);
                if ((FeatStd::Internal::String::ComparePartial(propertyName, wpmiTmpName, nameLenSize) == 0) && (wpmiTmpName[nameLenSize] == '\0')) {
                    wpmi = wpmiTmp;
                }
            }

            // parse hex value even if wpmi has not been found
            UInt32 hashValue;
            buffer = ParseHexValue(buffer + 1, hashValue);

            if (wpmi != 0) {
                // if everything fine so far, create the binding object
                // first locate the DataItemHierarchyNode of the bound data item
                const DataItemHierarchyNode *node = DataItemAccessor::Locate(hashValue);
                ok = node != 0;
                // then create the binding
                Candera::Behavior* behavior = Candera::Dynamic_Cast<Candera::Behavior*>(static_cast<Candera::WidgetBase*>(widget));
                Candera::AbstractNodePointer abstractNodePointer;
                if (0 != behavior) {
                    abstractNodePointer = behavior->GetNode();
                }
                ok = ok && CreateBinding(node->mItemKey, wpmi, widget, true, DataItemValue(), abstractNodePointer);
            }
            if ((!ok) || (wpmi == 0)) {
                COURIER_LOG_WARN("Failed to create binding to property %s::%s (%s)",
                                  wmi->GetName(),
                                  ((wpmi != 0) ? wpmi->GetName() : "<unknown property>"),
                                  buffer);
                // let the function return false as not all bindings could be established
                succeeded = false;
            }
            else {
                ++bindingCount;
            }
            if (*buffer != '\0') {
                // if there are multiple bindings, the next input token must be ';'
                // if not, the input string is malformed and we bail out to avoid further harm
                ok = *buffer == ';';
                ++buffer;
            }
            // collect the general function return code, we succeeded only if every binding could
            // be correctly established
            succeeded = succeeded && ok;
        }

        return succeeded ? bindingCount : -bindingCount;
    }

    // ========================================================================

    static const Char *cBindablePropertyName = "BindableProperty";

    // ------------------------------------------------------------------------
    BindablePropertyMetaInfoBase::BindablePropertyMetaInfoBase() :
        Candera::MetaInfo::WidgetPropertyMetaInfo(cBindablePropertyName)
    {
    }

    // ------------------------------------------------------------------------
    bool BindablePropertyMetaInfoBase::Set(Candera::WidgetBase *object, const Candera::Char *value)
    {
        FEATSTD_DEBUG_ASSERT(value != 0);
        if ((value != 0) && (*value != '\0')) {
            (void)Courier::Internal::DataBinding::DeserializeDataBindingProperty(value, object, this);
        }
        // return false will result in a second call of set due to the dependency loading mechanism of the default asset loader
        // therefore always return true.
        return true;
    }

    // ------------------------------------------------------------------------
    Candera::Animation::AnimationPropertySetter::SharedPointer BindablePropertyMetaInfoBase::CreateAnimationPropertySetter(Candera::WidgetBase*, Candera::Int, Candera::Int*)
    {
        return Candera::Animation::AnimationPropertySetter::SharedPointer(0);
    }

    // ------------------------------------------------------------------------
    bool BindablePropertyMetaInfoBase::Get(const Candera::WidgetBase *object, Candera::Char *buf, Candera::UInt buf_size)
    {
        FEATSTD_UNUSED2(object, buf_size);
        FEATSTD_DEBUG_ASSERT((buf != 0) && (buf_size > 0));
        if ((buf != 0) && (buf_size > 0)) {
            *buf = '\0';
        }
        return true;
    }

    // ------------------------------------------------------------------------
    const Char* BindablePropertyMetaInfoBase::GetBindablePropertyName()
    {
        return cBindablePropertyName;
    }

#ifdef CANDERA_META_DESCRIPTION
    // ------------------------------------------------------------------------
    const Char* BindablePropertyMetaInfoBase::GetCategory() const
    {
        return "Data Binding";
    }

    // ------------------------------------------------------------------------
    const Char* BindablePropertyMetaInfoBase::GetReadableName() const
    {
        return "Data Binding Definitions";
    }

    // ------------------------------------------------------------------------
    const Char* BindablePropertyMetaInfoBase::GetDescription() const
    {
        return "Define bindings from data items in the model to bindable properties of this widget";
    }
#endif
}}}
