//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_BindableParameterCapability_h)
#define Courier_DataBinding_BindableParameterCapability_h

#if defined(CANDERA_GLOBALIZATION_ENABLED)

#include <FeatStd/Base.h>

#include <CanderaGlobalization/DataBinding/ParameterizedLocalizableText.h>
#include <Courier/DataBinding/DataItemValue.h>

#include <FeatStd/Util/String.h>

/// @addtogroup COURIER_DATABINDING
/// @{
namespace Courier { namespace Internal {
    class BindableParameterCapabilityBase {
    public:
        virtual bool Get(const Candera::Globalization::Internal::ParameterizedLocalizableText *text, const DataItemValue &to, ::Courier::ConvertSignature Convert) const = 0;
        virtual bool Set(Candera::Globalization::Internal::ParameterizedLocalizableText *text, const DataItemValue &from, ::Courier::ConvertSignature Convert) = 0;
    };

    template <typename T>
    class BindableParameterCapability : public BindableParameterCapabilityBase {
    public:
        BindableParameterCapability(UInt8 parameterIndex) {
            mParameterIndex = parameterIndex;
        }

        /** reads parameter value from the text parameter associated with this capability object
            @param object the text instance owning the parameter
            @param to a mutable DataItemValue that receives the property value
            @param Convert an optional conversion function (see TypeConverter), may be 0 (null)
            @return true if the value could be retrieved, false if i.e. the DataItemValue was immutable or invalid */
        virtual bool Get(const Candera::Globalization::Internal::ParameterizedLocalizableText *text, const DataItemValue &to, ::Courier::ConvertSignature Convert) const {
            const DataItemValue& parameterValue = text->GetParameterValue(mParameterIndex);

            bool ok = to.SetValue(parameterValue);
            if (!ok && Convert) {
                ok = Convert(to, parameterValue);
            }

            return ok;
        }

        /** writes parameter value to the text parameter associated with this capability object
            @param object the text instance owning the parameter
            @param from a DataItemValue that refers to the property value to be written
            @param Convert an optional conversion function (see TypeConverter), may be 0 (null)
            @return true if the value could be written, false if i.e. the DataItemValue is invalid or types did not match */
        virtual bool Set(Candera::Globalization::Internal::ParameterizedLocalizableText *text, const DataItemValue &from, ::Courier::ConvertSignature Convert) {
            T value;
            DataItemValue convertedValue(TypeId(), &value, true);
            bool ok = SetOrConvert(convertedValue, from, Convert);
            ok = ok && text->SetValue(mParameterIndex, convertedValue);
            return ok;
        }

    private:
        UInt8 mParameterIndex;

        bool SetOrConvert(const DataItemValue &to, const DataItemValue &from, ::Courier::ConvertSignature Convert) const
        {
            bool ok = to.SetValue(from);
            if ((! ok) && (Convert != 0)) {
                ok = Convert(to, from);
            }
            return ok;
        }

        DataItemTypeId TypeId() const {
            return DataItemType<T>::Ident();
        }
    };

    typedef BindableParameterCapability<bool> BindableBoolParameterCapability;
    typedef BindableParameterCapability<Int32> BindableInt32ParameterCapability;
    typedef BindableParameterCapability<Float> BindableFloatParameterCapability;
    typedef BindableParameterCapability<FeatStd::String> BindableStringParameterCapability;
}}
/// @}

#endif

#endif // !defined(Courier_DataBinding_BindableParameterCapability_h)
