//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Courier/Messaging/MessageReferrer.h>
#include <Courier/Platform/Memory.h>
#include <Courier/Util/FixedSizeList.h>
#include "AsyncModelBindingSource.h"
#include "AsyncBinding.h"
#include "BindableProperty.h"
#include "AsyncWidgetPropertyBindings.h"

namespace Courier {

    COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::DataBinding);

    using namespace Candera::MetaInfo;

namespace Internal {

    using namespace DataBinding;

    // in order to use short names for the binding object class names,
    // all binding objects are defined in Async name space.
    using namespace DataBinding::Async;

    UpdateModelMsg *AsyncModelBindingSource::mCachedUpdateModelMsg = 0;

    // ========================================================================

    // ------------------------------------------------------------------------
    AsyncGenericModelBindingSource::AsyncGenericModelBindingSource(DataItemKey itemKey) :
        ModelBindingSource(itemKey),
        mCurrentDataItemMsg(0)
    {
    }

    // ------------------------------------------------------------------------
    AsyncGenericModelBindingSource::~AsyncGenericModelBindingSource()
    {
    }

    // ------------------------------------------------------------------------
    const AbstractDataItemMsg* AsyncGenericModelBindingSource::CurrentDataItemMsg() const
    {
        return *mCurrentDataItemMsg;
    }

    // ------------------------------------------------------------------------
    void AsyncGenericModelBindingSource::OnDataItemMsg(const AbstractDataItemMsg *msg)
    {
        FEATSTD_DEBUG_ASSERT(msg != 0);
        // create a reference on the message,
        // previous message is released by the referrer
        mCurrentDataItemMsg = TypedMessageReferrer<const AbstractDataItemMsg>(msg);
        COURIER_LOG_DEBUG("new mCurrentDataItemMsg bs=%u (rev %u)", msg->GetItemKey(), msg->GetDataRevision());
    }

    // ------------------------------------------------------------------------
    void AsyncGenericModelBindingSource::OnListEventMsg(const ListEventMsg *msg)
    {
        FEATSTD_UNUSED(msg);
    }

    // ------------------------------------------------------------------------
    bool AsyncGenericModelBindingSource::CreateBinding(DataItemKey,
                                                       const ::Candera::MetaInfo::WidgetPropertyMetaInfo *,
                                                       FrameworkWidget *, bool ,const DataItemValue &)
    {
        return false;
    }

    // ------------------------------------------------------------------------
    UInt16 AsyncGenericModelBindingSource::DataRevision() const
    {
        const AbstractDataItemMsg *msg = CurrentDataItemMsg();
        return (msg != 0) ? msg->GetDataRevision() : 0;
    }

    // ------------------------------------------------------------------------
    DataItemValue AsyncGenericModelBindingSource::GetDataItemValue(DataItemKey itemKey)
    {
        const AbstractDataItemMsg *msg = CurrentDataItemMsg();
        return (msg != 0) ? msg->GetItemValue(itemKey) : DataItemValue();
    }

    // ------------------------------------------------------------------------
    void AsyncGenericModelBindingSource::ReleaseDataItemMsgReference()
    {
        mCurrentDataItemMsg = TypedMessageReferrer<const AbstractDataItemMsg>(0);
    }

    // ========================================================================

    // ------------------------------------------------------------------------
    AsyncModelBindingSource::AsyncModelBindingSource(DataItemKey itemKey) :
        AsyncGenericModelBindingSource(itemKey),
        mBindingHead(0)
    {
        FEATSTD_LINT_NEXT_EXPRESSION(1938, "the access to the global data here is intended")
        mCachedUpdateModelMsg = 0;
    }

    // ------------------------------------------------------------------------
    AsyncModelBindingSource::~AsyncModelBindingSource()
    {
        // should not happen, the object gets destroyed
        if (mCachedUpdateModelMsg != 0) {
            COURIER_LOG_WARN("pending update message");
            // try to post the message
            if (!mCachedUpdateModelMsg->Post()) {
                // if the post fails, delete the message
                Platform::MessageFactory::Destroy(mCachedUpdateModelMsg);
                COURIER_LOG_ERROR("Post(UpdateModelMsg) failed");
            }
            mCachedUpdateModelMsg = 0;
        }

        // if binding object linked to this source exist, unbind (destroy) the objects
        while(mBindingHead != 0) {
            mBindingHead->Destroy();
        }
    }

    // ------------------------------------------------------------------------
    void AsyncModelBindingSource::DoPostProcessing()
    {
        // post the pending update message. if the post fails, the binding source
        // will automatically request post processing in the next message loop
        // iteration. Thus it is save to ignore the post return value.
        (void) PostCachedUpdateModelMsg();
    }

    // ------------------------------------------------------------------------
    UpdateModelMsg* AsyncModelBindingSource::GetCachedUpdateModelMsg()
    {
        // re-use existing update model message or create new one
        if (mCachedUpdateModelMsg == 0) {
            mCachedUpdateModelMsg = COURIER_MESSAGE_NEW(UpdateModelMsg)();
            COURIER_LOG_DEBUG("new UpdateModelMsg");
            if (mCachedUpdateModelMsg == 0) {
                COURIER_LOG_ERROR("failed to create UpdateModelMsg");
            }
        }
        return mCachedUpdateModelMsg;
    }

    // ------------------------------------------------------------------------
    bool AsyncModelBindingSource::PostCachedUpdateModelMsg()
    {
        bool ok = mCachedUpdateModelMsg == 0;
        if (!ok) {
            // if an UpdateMessage is pending, try to post the message
            COURIER_LOG_DEBUG("sending UpdateModelMsg (%u requests)", mCachedUpdateModelMsg->RequestCount());
            FEATSTD_LINT_NEXT_EXPRESSION(613, "pointer to mCachedUpdateModelMsg is checked at function entry")
            ok = mCachedUpdateModelMsg->Post();
            if (ok) {
                // clear message reference on success
                mCachedUpdateModelMsg = 0;
            }
            else {                                      // log error and
                COURIER_LOG_ERROR("Post(ModelUpdateMsg) failed");
            }
        }
        return ok;
    }

    // ------------------------------------------------------------------------
    bool AsyncModelBindingSource::AddChangeRecord(const AsyncBinding &binding) const
    {
        bool ok;
        UInt32 tries = 0;

        COURIER_LOG_DEBUG("ItemKey=%u, prop=%s", binding.GetItemKey(), binding.GetTargetName());

        do {
            // the given binding causes a model update -> create UpdateModelMsg (or get existing one)
            UpdateModelMsg *msg = GetCachedUpdateModelMsg();
            ok = msg != 0;
            if (ok) {
                // get the change set from the message and add a new record
                ChangeSet &changeSet = msg->GetChangeSet();
                DataItemValue value(changeSet.AddRecord(binding.GetItemKey(), DataRevision()));

                if ((tries == 1) && (!value.IsValid())) {
                    COURIER_LOG_WARN("failed to add change record, increase COURIER_CHANGE_SET_BUFFER_SIZE");
                }

                // let the binding write the property value to the change set
                ok = value.IsValid() && binding.GetTargetValue(value);
                if (!ok) {
                    // if adding the record failed, the message is not capable to
                    // hold the data. thus post the message and try again with fresh
                    // message
                    (void) PostCachedUpdateModelMsg();
                }
            }
            ++tries;
        } while((! ok) && (tries < 2));

        if (!ok) {
            COURIER_LOG_ERROR("failed to create change set record");
        }

        return ok;
    }

    // ------------------------------------------------------------------------
    bool AsyncModelBindingSource::AddListRequestRecord(DataItemKey itemKey,
                                                       ListRequestType::Enum requestType,
                                                       FeatStd::SizeType newIndex, FeatStd::SizeType oldIndex,
                                                       const DataItemValue &value) const
    {
        COURIER_LOG_DEBUG("ItemKey=%u, req=%u, newIndex=%u, oldIndex=%u", itemKey, UInt32(requestType), newIndex, oldIndex);
        bool ok;
        UInt32 tries = 0;
        do {
            UpdateModelMsg *msg = GetCachedUpdateModelMsg();
            ok = msg != 0;
            if (ok) {
                ChangeSet &changeSet = msg->GetChangeSet();
                ok = changeSet.AddListRequestRecord(itemKey, DataRevision(), requestType, newIndex, oldIndex, value);

                if ((tries == 1) && (! ok)) {
                    COURIER_LOG_WARN("failed to add list request, increase COURIER_CHANGE_SET_BUFFER_SIZE");
                }

                if (!ok) {
                    // if adding the record failed, the message is not capable to
                    // hold the data. thus post the message and try again with fresh
                    // message
                    (void) PostCachedUpdateModelMsg();
                }
            }
            ++tries;
        } while((! ok) && (tries < 2));
        return ok;
    }

    // ------------------------------------------------------------------------
    const SyncListInterface* AsyncModelBindingSource::GetListInterface(DataItemKey itemKey) const
    {
        const SyncListInterface *slif = 0;
        const AbstractDataItemMsg *dataItemMsg = CurrentDataItemMsg();
        DataItemValue value;
        if (dataItemMsg != 0) {
            // if message is available get the data item from the message
            value = dataItemMsg->GetItemValue(itemKey);
            // if the value is valid and an sync list interface
            if (value.IsValid() && (value.TypeId()->ListType() == ListInterfaceType::Synchronous)) {
                // cast to sync list interface
                slif = FeatStd::Internal::PointerToPointer<const SyncListInterface*>(value.ImmutablePointer());
            }
        }
        return slif;
    }

    // ------------------------------------------------------------------------
    DataItemValue AsyncModelBindingSource::GetListItem(DataItemKey itemKey, FeatStd::SizeType index) const
    {
        const SyncListInterface *slif = GetListInterface(itemKey);
        return (slif != 0) ? slif->Item(index) : DataItemValue();
    }

    // ------------------------------------------------------------------------
    void AsyncModelBindingSource::OnDataItemMsg(const AbstractDataItemMsg *msg)
    {
        COURIER_LOG_DEBUG("BS=%u", ItemKey());
        AsyncGenericModelBindingSource::OnDataItemMsg(msg);
        // iterate all associated bindings
        for (BindingObjectListIt it(mBindingHead); it != 0; ++it) {
            if (!(*it)->OnDataItemMsg(msg)) {
                COURIER_LOG_ERROR("failed update widget property %s", (*it)->GetTargetName());
            }
        }
    }

    // ------------------------------------------------------------------------
    void AsyncModelBindingSource::OnListEventMsg(const ListEventMsg *msg)
    {
        COURIER_LOG_DEBUG("BS=%u, req=%u, newIndex=%u, oldIndex=%u", ItemKey(),
                          UInt32(msg->GetEventType()), msg->GetNewIndex(), msg->GetOldIndex());
        FEATSTD_DEBUG_ASSERT(msg != 0);
                                                    // create ListEvent with msg values
        ListEvent listEvent(msg->GetEventType(),
                            msg->GetNewIndex(),
                            msg->GetOldIndex(),
                            DataItemValue());

        switch(listEvent.EventType()) {// set event item on add and modify events
            case ListEventType::AddedItem:
            case ListEventType::ModifiedItem:
                listEvent.SetItem(GetListItem(msg->GetItemKey(), msg->GetNewIndex()));
                break;

            case ListEventType::ItemCountChanged: {
                const SyncListInterface *slif = GetListInterface(msg->GetItemKey());
                listEvent.SetNewIndex((slif != 0) ? slif->Count() : ListInterfaceBase::UnknownQuantity());
                break;
            }

            case ListEventType::ListCleared:            // nothing to with those
            case ListEventType::MovedItem:
            case ListEventType::NewFragment:
            case ListEventType::None:
            case ListEventType::RefreshList:
            case ListEventType::RemovedItem:
            case ListEventType::RequestedItem:
            default:
                break;
        }
                                                    // iterate associated bindings
        for (BindingObjectListIt it(mBindingHead); it != 0; ++it) {
            if ((*it)->GetItemKey() == msg->GetItemKey()) {
                (*it)->ProcessListEvent(listEvent); // forward the ListEvent to the binding object
            }
        }
    }

    // ------------------------------------------------------------------------
    bool AsyncModelBindingSource::CreateBinding(DataItemKey itemKey,
                                                const WidgetPropertyMetaInfo *wmi,
                                                FrameworkWidget *widgetInstance,
                                                bool typeConvert,
                                                const DataItemValue &defaultValue)
    {
        Courier::Internal::DataBinding::Async::AsyncWidgetPropertyBinding *binding;
        switch(DataItemAccessor::Node(itemKey)->NodeType()) {
            case DataItemHierarchyNodeType::List:
                binding = FEATSTD_NEW(ListBinding);
                break;

            case DataItemHierarchyNodeType::Elementary:
                binding = FEATSTD_NEW(ElementaryBinding);
                break;

            case DataItemHierarchyNodeType::Compound:
            default:
                binding = 0;
                break;
        }

        if (binding != 0) {
            bool initialized = false;
            if (defaultValue.IsValid()) {
                initialized = binding->Init(this, itemKey, wmi, widgetInstance, typeConvert, defaultValue);
            }
            else {
                initialized = binding->Init(this, itemKey, wmi, widgetInstance, typeConvert, GetDataItemValue(itemKey));
            }
            if (!initialized) {
                FEATSTD_DELETE(binding);
                binding = 0;
            }
        }

        if (binding != 0) {
            COURIER_LOG_DEBUG("Item(%u)->%s", UInt32(itemKey), wmi->GetName());
        }

        FEATSTD_LINT_CURRENT_SCOPE(429, "binding object will be released on reference counting or widget / binding source dtor")
        return binding != 0;
    }

    // ------------------------------------------------------------------------
    bool AsyncModelBindingSource::Enlist(Internal::AsyncBinding *binding)
    {
        return BindingObjectList::Add(mBindingHead, binding);
    }

    // ------------------------------------------------------------------------
    bool AsyncModelBindingSource::Unlist(Internal::AsyncBinding *binding)
    {
        return BindingObjectList::Remove(mBindingHead, binding);
    }
}}   // namespace
