//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_AsyncBindingBase_h)
#define Courier_DataBinding_AsyncBindingBase_h

#include <CanderaWidget/WidgetBase/WidgetMetaInfo.h>
#include <Courier/Util/FixedSizeList.h>
#include <Courier/Diagnostics/ContextCheck.h>
#include <Courier/DataBinding/DataBindingFwd.h>
#include <Courier/DataBinding/Infrastructure.h>
#include <Courier/DataBinding/TypeConverter.h>
#include <Courier/DataBinding/SharedAsyncListInterface.h>
#include <Courier/DataBinding/BindingBase.h>
#include <Courier/DataBinding/AsyncBinding.h>

namespace Courier { 

    /// @addtogroup COURIER_DATABINDING
    /// @{
    namespace Internal {
#if !defined(COURIER_LIST_EVENT_QUEUE_SIZE)
    /** ListEvent queue size
        To avoid recursive ListEvent processing, the binding objects will use an internal static queue
        to store list events triggered by processing of the current list event (nested events). This
        constant defines the size of the fifo queue.
        The total amount of memory requiered for the queue is COURIER_LIST_EVENT_QUEUE_SIZE * sizeof(ListEvent) */
    static const UInt32 cCOURIER_LIST_EVENT_QUEUE_SIZE = 5;
    #define COURIER_LIST_EVENT_QUEUE_SIZE Courier::Internal::cCOURIER_LIST_EVENT_QUEUE_SIZE
#else
    static const UInt32 cCOURIER_LIST_EVENT_QUEUE_SIZE = COURIER_LIST_EVENT_QUEUE_SIZE;
#endif

#if !defined(COURIER_TYPE_CONVERSION_BUFFER_SIZE)
    /** type conversion requires the creation of a temporary object receiving the result
        of the type conversion call. The buffer is shared for all type conversion and is
        dimensioned according the definition of this preprocessor constant.
        if a type involved in type conversion does not fit into this buffer,
        the binding will not be established. */
    static const UInt32 cCOURIER_TYPE_CONVERSION_BUFFER_SIZE = 128;
    #define COURIER_TYPE_CONVERSION_BUFFER_SIZE Courier::Internal::cCOURIER_TYPE_CONVERSION_BUFFER_SIZE
#else
    static const UInt32 cCOURIER_TYPE_CONVERSION_BUFFER_SIZE = COURIER_TYPE_CONVERSION_BUFFER_SIZE;
#endif
    
    /**
     * Default implementation of an asynchronous binding.
     */
    class AsyncBindingBase : public BindingBase, public AsyncBinding {
    public:
        FEATSTD_LINT_CURRENT_SCOPE(1748, "the non-virtual base class 'Binding' has no members")
        static const UInt32 cListEventQueueSize = cCOURIER_LIST_EVENT_QUEUE_SIZE;
        static const UInt32 cTypeConversionBufferSize = cCOURIER_TYPE_CONVERSION_BUFFER_SIZE;

        FEATSTD_LINT_NEXT_EXPRESSION(1510,"base class 'AsyncBinding' and 'Binding' has no destructor AsyncBinding and Binding are only interfaces")
        virtual ~AsyncBindingBase();

        virtual void Destroy();

        virtual DataItemKey GetItemKey() const {
            COURIER_CHECK_SCOPE_AFFINITY();
            return mItemKey;
        }

        virtual void SetTargetModifiedTag();
        virtual void ProcessListEvent(const ListEvent &listEvent);
        DataItemValue GetModelValue() const;
        virtual bool OnDataItemMsg(const AbstractDataItemMsg *msg);

        virtual bool GetTargetValue(const DataItemValue &value) const = 0;
        virtual bool SetTargetValue(const DataItemValue &value) = 0;

        COURIER_DEFINE_SCOPE_AFFINITY(ComponentThreadContext<ComponentType::View>);

    protected:
        AsyncBindingBase();

        virtual Binding* GetBindingTargetHostNext() {
            COURIER_CHECK_SCOPE_AFFINITY();
            return BindingBase::GetBindingTargetHostNext();
        }

        virtual void SetBindingTargetHostNext(Binding* next) {
            COURIER_CHECK_SCOPE_AFFINITY();
            BindingBase::SetBindingTargetHostNext(next);
        }

        virtual AsyncBinding* GetBindingSourceNext() {
            COURIER_CHECK_SCOPE_AFFINITY();
            return mBindingSourceNext;
        }

        virtual void SetBindingSourceNext(AsyncBinding* next) {
            COURIER_CHECK_SCOPE_AFFINITY();
            mBindingSourceNext = next;
        }

        virtual void SetReadOnly(bool readonly) {
            COURIER_CHECK_SCOPE_AFFINITY();
            mIsReadOnly = readonly;
        }

        virtual bool IsReadOnly() const {
            COURIER_CHECK_SCOPE_AFFINITY();
            return mIsReadOnly;
        }

        virtual bool IsUpdating() const {
            COURIER_CHECK_SCOPE_AFFINITY();
            return mIsUpdating;
        }

        /** sets the IsUpdating flag */
        virtual void SetUpdating(bool value) {
            COURIER_CHECK_SCOPE_AFFINITY();
            mIsUpdating = value;
        }

        virtual bool IsInitialized() const {
            COURIER_CHECK_SCOPE_AFFINITY();
            return mInitialized;
        }

        virtual void SetInitialized(bool initialized) {
            COURIER_CHECK_SCOPE_AFFINITY();
            mInitialized = initialized;
        }

        /** resets the tag target modified tag */
        virtual void ResetTargetModifiedTag() {
            COURIER_CHECK_SCOPE_AFFINITY();
            mTargetModified = false;
        }

        FEATSTD_LINT_NEXT_EXPRESSION(1763,"the object itself is still const")
        Internal::AsyncModelBindingSource* GetBindingSource() const {
            COURIER_CHECK_SCOPE_AFFINITY();
            return mBindingSource;
        }

        bool UpdateModifiedFlag(bool setterResult) {
            COURIER_CHECK_SCOPE_AFFINITY();
            if (setterResult && static_cast<bool>(mTargetModified)) {
                mTargetModified = false;
            }
            return setterResult;
        }

        bool TargetInitialized() const {
            COURIER_CHECK_SCOPE_AFFINITY();
            return mTargetInitialized;
        }

        void SetTargetInitialized(bool value) {
            COURIER_CHECK_SCOPE_AFFINITY();
            mTargetInitialized = value;
        }

        bool Setup(AsyncModelBindingSource *bindingSource, DataItemKey itemKey);
        virtual bool OnTargetInitialized();

        /**
            <summary>Gets the current data item message from the binding source.</summary>
            <returns>
                null if the binding source did not receive a DataItemMsg so far, else the message object.
            </returns>
            */
        const AbstractDataItemMsg* CurrentDataItemMsg() const;

        /**
            <summary>
                Gets the cached update model message from the binding source object. See
                Courier::AsyncModelBindingSource::GetCachedUpdateModelMsg for details.
            </summary>
            <returns>
                null the creation of the cached message fails, else the cached update model message.
            </returns>
            */
        UpdateModelMsg* GetCachedUpdateModelMsg() const;

        /**
            <summary>Gets the data revision number from the current DataItemMsg.</summary>
            <returns>The data revision.</returns>
            */
        UInt16 GetDataRevision() const;

        /**
            <summary>Query a type convert function is defined for this binding.</summary>
            <returns>true if convert function is set, false if not.</returns>
            */
        bool HasConvertFn() const {
            COURIER_CHECK_SCOPE_AFFINITY();
            return mConvert != 0;
        }

        /**
            <summary>Gets the convert function set in the binding object.</summary>
            <returns>The type convert function.</returns>
            */
        ConvertSignature GetConvertFn() const {
            COURIER_CHECK_SCOPE_AFFINITY();
            return mConvert;
        }

        /**
            <summary>invokes the type convert function.</summary>
            <param name="dst">Destination for the conversion.</param>
            <param name="src">Source for the conversion.</param>
            <returns>true if conversion succeeded, false if it fails.</returns>
            */
        bool InvokeConvertFn(const DataItemValue &dst, const DataItemValue &src) const {
            COURIER_CHECK_SCOPE_AFFINITY();
            return (mConvert != 0) && mConvert(dst, src);
        }

        /**
            <summary>
                Sets up the type converter. If source and target binding types don't match, this function will
                try to locate and set a type converter in the binding object.
            </summary>
            <param name="typeId1">The first type identifier.</param>
            <param name="typeId2">The second type identifier.</param>
            <returns>true if a type converter could be found, false otherwise.</returns>
            */
        bool SetupTypeConverter(DataItemTypeId typeId1, DataItemTypeId typeId2);

     private:
         BitField mInitialized       :  1;                  ///< indicates if object has been initialized
         BitField mTargetModified    :  1;                  ///< flag indicating that the target has changed the data item value
         BitField mIsUpdating        :  1;                  ///< true if the binding is just updating the widget property value
         BitField mIsReadOnly        :  1;                  ///< true if data can be read only from data item
         BitField mTargetInitialized :  1;                  ///< true, if OnWidgetInitialized has been invoked on this object
         DataItemKey mItemKey;                              ///< the data item reference
         ConvertSignature mConvert;                         ///< pointer to type conversion function or null if no type conversion
         Internal::AsyncModelBindingSource *mBindingSource; ///< pointer to the binding source the binding belongs to
         AsyncBinding *mBindingSourceNext;                  ///< linked list node to register binding in the source
    };

    class UpdateFlagGuard {
    public:
        UpdateFlagGuard();
        UpdateFlagGuard(AsyncBinding *binding, bool *updateResult);
        ~UpdateFlagGuard();

    private:
        bool *mUpdateResult;
        AsyncBinding *mBinding;

        FEATSTD_LINT_NEXT_EXPRESSION(1704, "class is uncopyable, thus copy ctor must be privat")
            UpdateFlagGuard(const UpdateFlagGuard&);
        UpdateFlagGuard& operator=(const UpdateFlagGuard&);
    };

    }
    //@}
}

#endif // !defined(Courier_DataBinding_AsyncBindingBase_h)
