//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AsyncBindingBase.h"

#include <Courier/DataBinding/AsyncModelBindingSource.h>


namespace Courier {

    namespace Internal {

    using namespace DataBinding;

    FEATSTD_LINT_CTORDTOR_EFFECT(Courier::Internal::DataBinding::Async::UpdateFlagGuard)

    // ------------------------------------------------------------------------
    AsyncBindingBase::AsyncBindingBase() : mInitialized(false), mTargetModified(false), mIsUpdating(false), mIsReadOnly(false), mTargetInitialized(false),
                                           mItemKey(cInvalidDataItemKey), mConvert(0), mBindingSource(0), mBindingSourceNext(0)                  
    {
        COURIER_CHECK_SCOPE_AFFINITY();
    }

    // ------------------------------------------------------------------------
    bool AsyncBindingBase::Setup(AsyncModelBindingSource *bindingSource, DataItemKey itemKey) 
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        if (IsInitialized()) {
            FEATSTD_DEBUG_FAIL();
            return false;
        }

        FEATSTD_DEBUG_ASSERT(itemKey != cInvalidDataItemKey);
        FEATSTD_DEBUG_ASSERT(bindingSource != 0);
        bool ok = (itemKey != cInvalidDataItemKey) && (bindingSource != 0);

        // the binding is read only if the item value itself or the whole binding source is read only
        const DataItemHierarchyNode *hn = DataItemAccessor::Node(itemKey);
        FEATSTD_DEBUG_ASSERT(hn != 0);
        SetReadOnly((hn != 0) && (hn->IsReadOnly()));

        if (ok) {
            mItemKey = itemKey;
            mBindingSource = bindingSource;
            ok = bindingSource->Enlist(this);
        }

        return ok;
    }

    // ------------------------------------------------------------------------
    AsyncBindingBase::~AsyncBindingBase()
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        if (mBindingSource != 0) {
            (void) mBindingSource->Unlist(this);
        }

        mConvert = 0;
        mBindingSource = 0;
        mBindingSourceNext = 0;
    }


    // ------------------------------------------------------------------------
    void AsyncBindingBase::SetTargetModifiedTag()
    {
        COURIER_CHECK_SCOPE_AFFINITY();
    }

    // ------------------------------------------------------------------------
    void AsyncBindingBase::Destroy()
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        return FEATSTD_DELETE(this);
    }

    // ------------------------------------------------------------------------
    bool AsyncBindingBase::SetupTypeConverter(DataItemTypeId typeId1, DataItemTypeId typeId2)
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        // try locate converter typeId1<->typeId2
        TypeConverterBase *tc = TypeConverterRegistrar::Locate(typeId1, typeId2);
        if (tc != 0) {                              // if locate succeeded, set the type converter
            mConvert = tc->Vtbl().Convert;          // in the binding object
        }
        return tc != 0;
    }

    // ------------------------------------------------------------------------
    const AbstractDataItemMsg* AsyncBindingBase::CurrentDataItemMsg() const
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        return (mBindingSource != 0) ? mBindingSource->CurrentDataItemMsg() : 0;
    }

    // ------------------------------------------------------------------------
    UInt16 AsyncBindingBase::GetDataRevision() const
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        const AbstractDataItemMsg *msg = CurrentDataItemMsg();
        return (msg != 0) ? msg->GetDataRevision() : 0;
    }

    // ------------------------------------------------------------------------
    UpdateModelMsg* AsyncBindingBase::GetCachedUpdateModelMsg() const
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        return AsyncModelBindingSource::GetCachedUpdateModelMsg();
    }

    // ------------------------------------------------------------------------
    DataItemValue AsyncBindingBase::GetModelValue() const
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        const AbstractDataItemMsg *msg = CurrentDataItemMsg();
        return (msg != 0) ? msg->GetItemValue(mItemKey) : DataItemValue();
    }


    // ------------------------------------------------------------------------
    void AsyncBindingBase::ProcessListEvent(const ListEvent &)
    {
        COURIER_CHECK_SCOPE_AFFINITY();
    }

    // ------------------------------------------------------------------------
    bool AsyncBindingBase::OnTargetInitialized()
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        // set the value from the model if already available.
        if (!TargetInitialized()) {
            // if the update fails, trust that subsequent updates wont fail
            DataItemValue value(GetModelValue());
            if (value.IsValid()) {
                (void) SetTargetValue(value);
            }
            SetTargetInitialized(true);
        }
        return true;
    }

    // ------------------------------------------------------------------------
    bool AsyncBindingBase::OnDataItemMsg(const AbstractDataItemMsg *msg)
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        bool ok;
        if (msg->ItemRequiresUpdate(GetItemKey())) {
            DataItemValue value(msg->GetItemValue(GetItemKey()));
            ok = value.IsValid() && SetTargetValue(value);
        }
        else {
            ok = true;
        }
        return ok;
    }


    // ========================================================================

    // ------------------------------------------------------------------------
    UpdateFlagGuard::UpdateFlagGuard() : mUpdateResult(0), mBinding(0)
    {
    }

    // ------------------------------------------------------------------------
    UpdateFlagGuard::UpdateFlagGuard(AsyncBinding *binding, bool *updateResult) : mUpdateResult(updateResult), mBinding(0)
    {
        FEATSTD_DEBUG_ASSERT(updateResult != 0);
        if (!binding->IsUpdating()) {
            mBinding = binding;
            binding->SetUpdating(true);
        }
    }

    // ------------------------------------------------------------------------
    UpdateFlagGuard::~UpdateFlagGuard()
    {
        if (mBinding != 0) {
            FEATSTD_DEBUG_ASSERT(mBinding->IsUpdating());
            mBinding->SetUpdating(false);
            if ((mUpdateResult != 0) && (*mUpdateResult)) {
                mBinding->ResetTargetModifiedTag();
            }
            mBinding = 0;
        }
        mUpdateResult = 0;
    }

    // ========================================================================
}}
