//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TRANSITION_STRATEGY_H)
#define CANDERA_TRANSITION_STRATEGY_H

#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/Environment.h>
#include <Candera/System/Container/SingleLinkedList.h>
#include <CanderaTransitions/Transition.h>

namespace Candera {

class EventListener;
class Event;

namespace Transitions {

/**
*  @brief The default transition strategy which also serves as the base class from which to inherit custom transition strategies.
*  To enable execution of transitions the Update method must be called on a regular basis.
*  TransitionStrategy composes Transitions from Requests and Rules and executes these transitions.
*/
class TransitionStrategy : public CanderaObject
{
public:
    FEATSTD_TYPEDEF_BASE(CanderaObject);
    FEATSTD_RTTI_DECLARATION();

    TransitionStrategy();

    virtual ~TransitionStrategy();

    static TransitionStrategy& GetInstance();
    static void SetInstance(TransitionStrategy& transitionStrategy);

    /**
     *  Create a new Transition from a Request and a set of Rules.
     *  @param request The transition request from which a concrete Transition should be created, RequestFragments are removed from the Request if they are handled successfully.
     *  @param rules The ruleset from which a concrete Transition should be created.
     *  @param activeTransitions output parameter that Will hold the currently active transition set.
     *  @return A shared pointer to the created transition.
     */
    virtual Transition::SharedPointer Start(Request& request, Rule::Set& rules, Transition::Set& activeTransitions);

    /**
     *  Continue execution of the currently active transition set.
     *  @param worldTime  The world time in milliseconds.
     *  @param deltaTime  The elapsed time since the last call to Update, in milliseconds.
     */
    void UpdateWithMilliseconds(TimeType worldTime, TimeType deltaTime) { FEATSTD_UNUSED(worldTime); Update(deltaTime); }

    /**
     *  Continue execution of the currently active transition set.
     *  @param deltaTime the elapsed time since the last call to Update, in milliseconds.
     */
    virtual void Update(TimeType deltaTime);

    /**
     *  Registers an EventListener to events from this transition strategy; a listener can only be added once.
     *  @param listener a shared pointer to the given EventListener instance.
     *  @return true if adding the listener succeeded, false otherwise.
     */
    bool AddListener(EventListener& listener);

    /**
     *  Remove an EventListener from this transition strategy.
     *  @param listener a shared pointer to the given EventListener instance.
     *  @return true if the listener was found and removed, false otherwise.
     */
    bool RemoveListener(EventListener& listener);

    /**
    *  Sets the AnimationTimeDispatcher for AnimationPlayers
    *  @param timeDispatcher AnimationTimeDispatcher to set.
    */
    void SetAnimationTimeDispatcher(Animation::AnimationTimeDispatcher::SharedPointer& timeDispatcher) const;

    /**
    *  Retrieves the AnimationTimeDispatcher
    *  @return the AnimationTimeDispatcher
    */
    Animation::AnimationTimeDispatcher::SharedPointer GetAnimationTimeDispatcher() const;

protected:

    /**
     *  Begin composing a transition.
     */
    static void BeginTransition();


    /**
     *  Creates and adds a DefaultTransitionFragment for the given request without a rule, for instance if no rule could be matched.
     *  Will attempt to remove the given requestFragment from the request.
     *  Must be called between BeginTransition and EndTransition.
     *
     *  @param request The request from which to remove the request fragment.
     *  @param requestFragment The request fragment from which this transition will be created. Function will fail if points to null
     *  @return true if creating and adding the fragment was successful, false otherwise. 
     */
    static bool AddDefaultTransitionFragment(Request& request, const RequestFragment::SharedPointer& requestFragment);

    /**
     *  Creates and adds a transition fragment to the current transition via the factory defined in the given rule.
     *  Will attempt to remove the given requestFragment from the request.
     *  Must be called between BeginTransition and EndTransition.
     *
     *  @param request The request from which to remove the request fragment.
     *  @param requestFragment The request fragment from which this transition will be created. Function will fail if points to null
     *  @param rule The rule from which this transition fragment was created, should hold a pointer to a factory to create the fragment.
     *  @return true if creating and adding the fragment was successful, false otherwise.
     */
    static bool AddTransitionFragment(Request& request, const RequestFragment::SharedPointer& requestFragment, const Rule::SharedPointer& rule);

    /**
     *  Creates and adds a transition fragment to the current transition via the factory defined in the given rule.
     *  Will attempt to remove sourceRequestFragment and destinationRequestFragment from the request.
     *  Must be called between BeginTransition and EndTransition.
     *
     *  @param request The request from which to remove the source and destination fragments.
     *  @param sourceRequestFragment The source request fragment from which this transition will be created. Function will fail if points to null
     *  @param destinationRequestFragment The request fragment matched to the destination from which this transition will be created. Function will fail if points to null
     *  @param rule The rule from which this transition fragment was created, should hold a pointer to a factory to create the fragment.
     *  @return true if creating and adding the fragment was successful, false otherwise.
     */
    static bool AddTransitionFragment(Request& request, const RequestFragment::SharedPointer& sourceRequestFragment, const RequestFragment::SharedPointer& destinationRequestFragment, const Rule::SharedPointer& rule);

    /**
     *  Ends the current transition composition, returning a shared pointer to the finished transition.
     *  @return a shared pointer to the composed transition.
     */
    static Transition::SharedPointer EndTransition();

    /**
     *  Adds a transition to the beginning of the active transition set.
     *  @param transition A shared pointer to a transition instance.
     */
    void Prepend(const Transition::SharedPointer& transition);

    /**
     *  Notify listeners of an event.
     */
    void Notify(const Event& event);

    Transition::Set m_activeTransitions; ///< Currently running transitions.
    TimeType m_lastWorldTime;            ///< Last known world time; in milliseconds.

private:
    void UpdateInternal(TimeType deltaTime);
    static bool AddTransitionFragmentInternal(const RequestFragment::SharedPointer& requestFragment, const Rule::SharedPointer& rule, TransitionFragmentFactory& factory);
    bool FindReversibleFragments(Request& request, const RequestFragment::SharedPointer& requestFragment, RequestFragment::SharedPointer& nextFragment, const Rule& rule);
    void ApplyEarlyLateDelay(const Transition::Set& transitionSet) const;
    void GetMaximumEarlyFinishTimes(const TransitionFragment* transitionFragment, Float& maxEarlyActivationFinishTime, Float& maxEarlyDeactivationFinishTime) const;
    void AddEarlyFinishTimesToLateTimelines(TransitionFragment* transitionFragment, Float maxEarlyActivationFinishTime, Float maxEarlyDeactivationFinishTime) const;
    bool InjectBidirectionalReversedRules(Rule::Set& rules) const;

    typedef Internal::SingleLinkedList<EventListener*> EventListenerContainer;
    EventListenerContainer m_eventListeners;
};

/// @}
}   // namespace Transitions
}   // namespace Candera

#endif // CANDERA_TRANSITION_STRATEGY_H
