//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TransitionStrategy.h"

#include <FeatStd/Util/StaticObject.h>

#include <CanderaTransitions/Trigger.h>
#include <CanderaTransitions/TransitionFragment.h>
#include <CanderaTransitions/TransitionEvents.h>
#include <CanderaTransitions/DefaultTransitionFactory.h>

#include <Candera/System/MemoryManagement/CanderaHeap.h>

#ifdef FEATSTD_LOG_ENABLED
#include <Candera/System/Diagnostics/Log.h>
#endif

namespace Candera {

using namespace Transitions;

FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaTransitions);

FEATSTD_RTTI_DEFINITION(Candera::Transitions::TransitionStrategy, CanderaObject)

static TransitionStrategy& DefaultTransitionStrategyGetInstance() {
    FEATSTD_UNSYNCED_STATIC_OBJECT(TransitionStrategy, s_defaultTransitionStrategyGetInstance);
    return s_defaultTransitionStrategyGetInstance;
}

static TransitionStrategy*& TransitionStrategyGetInstance() {
    FEATSTD_UNSYNCED_STATIC_OBJECT(TransitionStrategy*, s_transitionStrategyGetInstance, 0);
    return s_transitionStrategyGetInstance;
}

TransitionStrategy::TransitionStrategy() :
Base(),
m_activeTransitions(),
m_lastWorldTime(0),
m_eventListeners()
{
}

TransitionStrategy::~TransitionStrategy()
{
    m_eventListeners.Clear(); //cleanup listeners
}

TransitionStrategy& TransitionStrategy::GetInstance()
{
    if (0 == TransitionStrategyGetInstance()) {
        return DefaultTransitionStrategyGetInstance();
    }
    return *TransitionStrategyGetInstance();
}

void TransitionStrategy::SetInstance(TransitionStrategy& transitionStrategy)
{
    TransitionStrategyGetInstance() = &transitionStrategy;
}

static TransitionFragment::SharedPointer& LastTransitionFragment()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(TransitionFragment::SharedPointer, s_lastTransitionFragment);
    return s_lastTransitionFragment;
}

static TransitionFragment::SharedPointer s_initLastTransitionFragment = LastTransitionFragment();

static Transition::SharedPointer& CurrentTransition()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(Transition::SharedPointer, s_currentTransition);
    return s_currentTransition;
}

static Transition::SharedPointer s_initCurrentTransition = CurrentTransition();

void TransitionStrategy::BeginTransition()
{
    FEATSTD_DEBUG_ASSERT(CurrentTransition().PointsToNull());
    CurrentTransition() = Transition::SharedPointer(CANDERA_NEW(Transition)());
}

/**
 * Helper function the fetch a default fragment factory instance.
 */
DefaultTransitionFactory& GetDefaultFragmentFactory()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(DefaultTransitionFactory, s_defaultFragmentFactory);
    return s_defaultFragmentFactory;
}

static DefaultTransitionFactory& s_initDefaultFragmentFactory = GetDefaultFragmentFactory();

bool TransitionStrategy::AddDefaultTransitionFragment(Request& request, const RequestFragment::SharedPointer& requestFragment)
{
    if (!requestFragment.PointsToNull()) {
        RequestFragment* requestCopy = CANDERA_NEW(RequestFragment)(*requestFragment);
        if(requestCopy == 0) {
            FEATSTD_LOG_ERROR("Could not create a clone of the source request fragment when creating a transition fragment.");
            return false;
        }
        requestCopy->m_nextFragment.Release(); // next pointer is not needed anymore.

        if(AddTransitionFragmentInternal(RequestFragment::SharedPointer(requestCopy), Rule::SharedPointer(), GetDefaultFragmentFactory())) {
            //remove handled fragment
            if(!request.Remove(requestFragment)) {
                FEATSTD_LOG_WARN("Removal of the current request fragment failed, transition fragment was created successfully.");
            }
            return true;
        }
    }
    return false;
}

bool TransitionStrategy::AddTransitionFragment(Request& request, const RequestFragment::SharedPointer& requestFragment, const Rule::SharedPointer& rule)
{
    if ((!requestFragment.PointsToNull()) && (!rule.PointsToNull()) && (!(rule->m_factory).PointsToNull())) {

        RequestFragment* requestCopy = CANDERA_NEW(RequestFragment)(*requestFragment);
        if(requestCopy == 0) {
            FEATSTD_LOG_ERROR("Could not create a clone of the source request fragment when creating a transition fragment.");
            return false;
        }
        requestCopy->m_nextFragment.Release(); // Next pointer is not needed anymore.

        if( AddTransitionFragmentInternal(RequestFragment::SharedPointer(requestCopy), rule, *(rule->m_factory)) ) {
            //remove handled fragment
            if(!request.Remove(requestFragment)) {
                FEATSTD_LOG_WARN("Removal of the current request fragment failed, transition fragment was created successfully.");
            }
            return true;
        }
    }
    return false;
}

bool TransitionStrategy::AddTransitionFragment(Request& request, const RequestFragment::SharedPointer& sourceRequestFragment, const RequestFragment::SharedPointer& destinationRequestFragment, const Rule::SharedPointer& rule)
{
    if ((!sourceRequestFragment.PointsToNull()) && (!destinationRequestFragment.PointsToNull()) && (!rule.PointsToNull()) && (!(rule->m_factory).PointsToNull())) {

        RequestFragment* sourceCopy = CANDERA_NEW(RequestFragment)(*sourceRequestFragment);
        if(sourceCopy == 0) {
            FEATSTD_LOG_ERROR("Could not create a clone of the source request fragment when creating a transition fragment.");
            return false;
        }

        RequestFragment* destinationCopy = CANDERA_NEW(RequestFragment)(*destinationRequestFragment);
        if(destinationCopy == 0) {
            FEATSTD_LOG_ERROR("Could not create a clone of the destination request fragment when creating a transition fragment.");
            FEATSTD_DEBUG_ASSERT(sourceCopy != 0); //ensured by above if.
            CANDERA_DELETE(sourceCopy);
            return false;
        }

        //source fragment should point to the destination fragment
        destinationCopy->m_nextFragment.Release();
        sourceCopy->m_nextFragment = RequestFragment::SharedPointer(destinationCopy);

        if(AddTransitionFragmentInternal(RequestFragment::SharedPointer(sourceCopy), rule, *(rule->m_factory))) {
            //remove handled fragments
            bool result = request.Remove(destinationRequestFragment);
            result = result && request.Remove(sourceRequestFragment);

            if(!result) {
                FEATSTD_LOG_WARN("Removal of one or more request fragments failed, transition fragment was created successfully.");
            }
            return true;
        }
    }
    return false;
}

bool TransitionStrategy::AddTransitionFragmentInternal(const RequestFragment::SharedPointer& requestFragment, const Rule::SharedPointer& rule, TransitionFragmentFactory& factory)
{
    FEATSTD_DEBUG_ASSERT(!CurrentTransition().PointsToNull());

    Trigger* trigger = CANDERA_NEW(Trigger)(requestFragment, rule);
    if (trigger == 0) {
        return false;
    }

    TransitionFragment::SharedPointer fragment = factory.Create(Trigger::SharedPointer(trigger));
    if (fragment.PointsToNull()) {
        //Trigger will be deleted by shared pointer in fragment.
        return false;
    }

    if (!LastTransitionFragment().PointsToNull()) {
        LastTransitionFragment()->m_nextFragment = fragment;
    }
    else {
        CurrentTransition()->m_firstFragment = fragment;
    }
    LastTransitionFragment() = fragment;
    return true;
}

Transition::SharedPointer TransitionStrategy::EndTransition()
{
    FEATSTD_DEBUG_ASSERT(!CurrentTransition().PointsToNull());
    Transition::SharedPointer result = CurrentTransition();
    CurrentTransition() = Transition::SharedPointer();
    LastTransitionFragment() = TransitionFragment::SharedPointer();
    return result;
}

/**
*  Helper function to match request fragments to rules. Will only match Activation and Deactivation requests. Anything else will always return false.
*/
bool Matches(const RequestFragment& requestFragment, const Rule& rule) 
{
    if (requestFragment.GetHint().GetName() == rule.GetHint().GetName()) {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, CANDERA_LINT_REASON_RETURNINCASE);
        switch (static_cast<RequestFragment::Type>(requestFragment.GetType())) {
        case RequestFragment::Activate:
            return requestFragment.GetIdentifier().Matches(rule.GetDestination());
        case RequestFragment::Deactivate:
            return requestFragment.GetIdentifier().Matches(rule.GetSource());
        case RequestFragment::Finish:
            return (requestFragment.GetIdentifier().Matches(rule.GetSource())) || (requestFragment.GetIdentifier().Matches(rule.GetDestination()));
        default:
            return false;
        }
    }

    return false;
}

/**
*  Helper function to match any type selectors
*/
bool MatchesAnySelector(Identifier::Type identifier)
{
    switch (identifier) {
        FEATSTD_LINT_CURRENT_SCOPE(1960, "Violates MISRA C++ 3008 Required Rule 6-4-5: All paths end in return")
        case Identifier::AnyNode2DBuiltInIdentifier:
        case Identifier::AnyNode3DBuiltInIdentifier:
        case Identifier::AnyScene2DBuiltInIdentifier:
        case Identifier::AnyScene3DBuiltInIdentifier:
            return true;
        default:
            return false;
    }
}

/**
 *  Return a RequestFragment matching to the given hint name, identifier and type.
 */
RequestFragment::SharedPointer FindRequestFragment(const Request& request, const Hint& hint, const Identifier& identifier, RequestFragment::Type type)
{
    RequestFragment::SharedPointer nextFragment = request.GetFirstFragment();
    while (!nextFragment.PointsToNull()) {
        if ((type == nextFragment->GetType()) && (hint.GetName() == nextFragment->GetHint().GetName()) && (nextFragment->GetIdentifier().Matches(identifier))) {
            return nextFragment;
        }
        nextFragment = nextFragment->GetNext();
    }
    return nextFragment;
}

/**
*  Identify reversible fragments and handle them if reverse is allowed
*/
bool TransitionStrategy::FindReversibleFragments(Request& request, const RequestFragment::SharedPointer& requestFragment, RequestFragment::SharedPointer& nextFragment, const Rule& rule)
{
    bool isMatched = false;
    Transition::SharedPointer transition = m_activeTransitions.GetFirstTransition();
    while (!transition.PointsToNull()) {

        TransitionFragment::SharedPointer transitionFragment = transition->GetFirstFragment();
        while (!transitionFragment.PointsToNull()) {

            // match only specific Scene/Node Identifiers
            if ((!MatchesAnySelector(rule.GetSource().GetType())) && (!MatchesAnySelector(rule.GetDestination().GetType()))) {
                if ((!transitionFragment->GetTrigger().PointsToNull()) && (!transitionFragment->GetTrigger()->GetRule().PointsToNull())) {
                    Rule::SharedPointer transitionRule = transitionFragment->GetTrigger()->GetRule();
                    
                    if ((transitionRule->GetHint().GetName() == rule.GetHint().GetName()) &&
                        (transitionRule->GetSource().Matches(rule.GetDestination())) && 
                        (transitionRule->GetDestination().Matches(rule.GetSource())))
                    {
                        transitionFragment->Reverse();
                        Notify(OnReverseTransitionFragment(TransitionFragment::SharedPointer(transitionFragment)));
                        static_cast<void>(request.Remove(requestFragment));

                        if ((rule.GetSource().GetType() != Identifier::NoneBuiltInIdentifier) || (rule.GetDestination().GetType() != Identifier::NoneBuiltInIdentifier)) {
                            // remove second requestFragment
                            RequestFragment::SharedPointer secondFragment;

                            switch (static_cast<RequestFragment::Type>(requestFragment->GetType())) {
                            case RequestFragment::Activate:
                                secondFragment = FindRequestFragment(request, rule.GetHint(), rule.GetSource(), RequestFragment::Deactivate);
                                break;
                            case RequestFragment::Deactivate:
                            default:
                                FEATSTD_DEBUG_ASSERT(requestFragment->GetType() == RequestFragment::Deactivate);
                                secondFragment = FindRequestFragment(request, rule.GetHint(), rule.GetDestination(), RequestFragment::Activate);
                                break;
                            }

                            if (!secondFragment.PointsToNull()) {
                                if (nextFragment == secondFragment) {
                                    nextFragment = secondFragment->GetNext();
                                }
                                static_cast<void>(request.Remove(secondFragment));
                            }
                        }
                        isMatched = true;
                    }
                }
            }
            transitionFragment = transitionFragment->GetNext();
        }
        transition = transition->GetNext();
    }

    return isMatched;
}

void TransitionStrategy::GetMaximumEarlyFinishTimes(const TransitionFragment* transitionFragment, Float& maxEarlyActivationFinishTime,
    Float& maxEarlyDeactivationFinishTime) const
{
    while (0 != transitionFragment) {
        if (RequestFragment::Activate == transitionFragment->m_requestFragmentType) {
            if (Hint::Early == transitionFragment->m_activationStrategy) {
                maxEarlyActivationFinishTime = Math::Maximum(maxEarlyActivationFinishTime, transitionFragment->m_finishTime);
            }
        }
        else if (RequestFragment::Deactivate == transitionFragment->m_requestFragmentType) {
            if (Hint::Early == transitionFragment->m_deactivationStrategy) {
                maxEarlyDeactivationFinishTime = Math::Maximum(maxEarlyDeactivationFinishTime, transitionFragment->m_finishTime);
            }
        }
        else {
            // satisfy LINT
        }

        TransitionFragment* additionalFragment = transitionFragment->GetAdditionalFragment().GetPointerToSharedInstance();
        if (0 != additionalFragment) {
            GetMaximumEarlyFinishTimes(additionalFragment, maxEarlyActivationFinishTime, maxEarlyDeactivationFinishTime);
        }

        transitionFragment = transitionFragment->GetNext().GetPointerToSharedInstance();
    }
}

void TransitionStrategy::AddEarlyFinishTimesToLateTimelines(TransitionFragment* transitionFragment, Float maxEarlyActivationFinishTime,
    Float maxEarlyDeactivationFinishTime) const
{
    while (0 != transitionFragment) {
        if (RequestFragment::Activate == transitionFragment->m_requestFragmentType) {
            if (Hint::Late == transitionFragment->m_activationStrategy) {
                transitionFragment->ApplyLateDelay(maxEarlyActivationFinishTime);
            }
        }
        else if (RequestFragment::Deactivate == transitionFragment->m_requestFragmentType) {
            if (Hint::Late == transitionFragment->m_deactivationStrategy) {
                transitionFragment->ApplyLateDelay(maxEarlyDeactivationFinishTime);
            }
        }
        else {
            // satisfy LINT
        }

        TransitionFragment* additionalFragment = transitionFragment->GetAdditionalFragment().GetPointerToSharedInstance();
        if (0 != additionalFragment) {
            AddEarlyFinishTimesToLateTimelines(additionalFragment, maxEarlyActivationFinishTime, maxEarlyDeactivationFinishTime);
        }

        transitionFragment = transitionFragment->GetNext().GetPointerToSharedInstance();
    }
}

void TransitionStrategy::ApplyEarlyLateDelay(const Transition::Set& transitionSet) const
{
    Transition* transition = transitionSet.GetFirstTransition().GetPointerToSharedInstance();

    while (0 != transition) {
        Float maxEarlyActivationFinishTime = 0.0F;
        Float maxEarlyDeactivationFinishTime = 0.0F;

        GetMaximumEarlyFinishTimes(transition->GetFirstFragment().GetPointerToSharedInstance(),
            maxEarlyActivationFinishTime, maxEarlyDeactivationFinishTime);

        Float maxEarlyFinishTime = Math::Maximum(maxEarlyActivationFinishTime, maxEarlyDeactivationFinishTime);

        FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
        if (0.0F != maxEarlyFinishTime) {
            AddEarlyFinishTimesToLateTimelines(transition->GetFirstFragment().GetPointerToSharedInstance(),
                maxEarlyFinishTime, maxEarlyFinishTime);
        }

        transition = transition->GetNext().GetPointerToSharedInstance();
    }
}

bool TransitionStrategy::InjectBidirectionalReversedRules(Rule::Set& rules) const
{
    Rule::SharedPointer rule = rules.GetFirstRule();
    while ((!rule.PointsToNull())) {
        if (rule->IsBidirectional()) {
            Rule* nextRule = rule->GetNext().GetPointerToSharedInstance();
            bool isNextRuleInjected = false;
            if (0 != nextRule) {
                if ((rule->GetSource() == nextRule->GetDestination()) &&
                    (rule->GetDestination() == nextRule->GetSource()) &&
                    (rule->GetHint() == nextRule->GetHint()) &&
                    (rule->m_factory == nextRule->m_factory) &&
                    (rule->IsReverseAllowed() == nextRule->IsReverseAllowed()) &&
                    (!nextRule->IsBidirectional())) {
                    isNextRuleInjected = true;
                }
            }

            if (!isNextRuleInjected) {
                Rule::SharedPointer ruleReversed(CANDERA_NEW(Rule)(rule->GetDestination(), rule->GetSource(), rule->GetHint(),
                    rule->m_factory, rule->IsReverseAllowed(), false));
                if (ruleReversed == 0) {
                    return false;
                }

                ruleReversed->m_next = rule->GetNext();
                rule->m_next = ruleReversed;
                rule = rule->GetNext();
            }
        }

        rule = rule->GetNext();
    }

    return true;
}

Transition::SharedPointer TransitionStrategy::Start(Request& request, Rule::Set& rules, Transition::Set& activeTransitions)
{
    if (!InjectBidirectionalReversedRules(rules)) {
        return Transition::SharedPointer();
    }

    // Initialize / reset the CurrentTransition static variable
    BeginTransition();

    // Iterate through all request fragments
    RequestFragment::SharedPointer requestFragment = request.GetFirstFragment();
    while (!requestFragment.PointsToNull()) {

        // Cache nextFragment shared pointer as next pointer in requestFragment may change when request fragments are removed from the request and added to TransitionFragments
        RequestFragment::SharedPointer nextFragment = requestFragment->GetNext();

        if (requestFragment->GetType() == RequestFragment::Finish) {
            //Handle finish request
            Transition::SharedPointer transition = m_activeTransitions.GetFirstTransition();
            while (!transition.PointsToNull()) {

                TransitionFragment::SharedPointer transitionFragment = transition->GetFirstFragment();
                while (!transitionFragment.PointsToNull()) {

                    if ((!transitionFragment->GetTrigger().PointsToNull()) && (!transitionFragment->GetTrigger()->GetRule().PointsToNull())) {
                        if (Matches(*requestFragment, *(transitionFragment->GetTrigger()->GetRule()))) {
                            transitionFragment->Finish();
                            if (transitionFragment->IsFinished()) {
                                //It is not guaranteed that the virtual function Finish actually sets the transition fragment to finished.
                                Notify(OnFinishTransitionFragment(transitionFragment));
                            }
                        }
                    }
                    transitionFragment = transitionFragment->GetNext();

                }
                transition = transition->GetNext();
            }
            static_cast<void>(request.Remove(requestFragment));
        }
        else {
            // Iterate through the rules until one matches with the current request fragment
            Rule::SharedPointer rule = rules.GetFirstRule();
            bool fragmentCreated = false;
            bool isReversibleFragmentFound = false;
            while ((!fragmentCreated) && (!rule.PointsToNull())) {

                if (Matches(*requestFragment, *rule)) {

                    // Identify reversible fragments
                    if (rule->IsReverseAllowed())
                    {
                        isReversibleFragmentFound = FindReversibleFragments(request, requestFragment, nextFragment, *rule);
                    }

                    if (!isReversibleFragmentFound) {
                        // Distinction between rules that either have one single fragment (activation or deactivation) or two fragments.
                        // If the rule has both activation and deactivation fragments, a second, complementing request fragment needs to be found.
                        if ((rule->GetSource().GetType() == Identifier::NoneBuiltInIdentifier) || (rule->GetDestination().GetType() == Identifier::NoneBuiltInIdentifier)) {
                            // No second request fragment required. Create transition fragment and remove request fragment from request set.
                            fragmentCreated = AddTransitionFragment(request, requestFragment, rule);
                        }
                        else {
                            // This rule expects a source and a destination; look for a complementing request fragment.
                            RequestFragment::SharedPointer secondFragment;

                            switch (static_cast<RequestFragment::Type>(requestFragment->GetType())) {
                            case RequestFragment::Activate:
                                secondFragment = FindRequestFragment(request, rule->GetHint(), rule->GetSource(), RequestFragment::Deactivate);
                                break;
                            case RequestFragment::Deactivate:
                            default:
                                FEATSTD_DEBUG_ASSERT(requestFragment->GetType() == RequestFragment::Deactivate); // Ensured by enclosing if statement. 
                                secondFragment = FindRequestFragment(request, rule->GetHint(), rule->GetDestination(), RequestFragment::Activate);
                                break;
                            }

                            if (!secondFragment.PointsToNull()) {

                                if (nextFragment == secondFragment) {
                                    // Second fragment will be removed - cache next pointer.
                                    nextFragment = secondFragment->GetNext();
                                }

                                // If all components are set correctly, new transitions fragments (activation and deactivation fragment) are created and added to the current transition
                                fragmentCreated = AddTransitionFragment(request, requestFragment, secondFragment, rule);

                                if (!fragmentCreated) {
                                    // Second fragment was not removed, no change.
                                    nextFragment = secondFragment;
                                }
                            }
                        }
                    }
                }

                rule = rule->GetNext();
            }

            if ((!fragmentCreated) && (!isReversibleFragmentFound)) {
                // No rule has matched this request. Create a default transition.
                if (!AddDefaultTransitionFragment(request, requestFragment)) {
                    FEATSTD_LOG_INFO("Adding default transition fragment failed; request fragment will remain unhandled.");
                }
            }
        }

        requestFragment = nextFragment;
    }

    // Gather the current transition and add it to the active transitions, if the current one is not empty
    Transition::SharedPointer transition = EndTransition();
    if (!transition->GetFirstFragment().PointsToNull()) {
        Prepend(transition);
    }
    activeTransitions = m_activeTransitions;

    // Arrange the transition fragments according to the respective transition strategies and delay settings.
    // This is done by modifying the finish time of each fragment.
    ApplyEarlyLateDelay(m_activeTransitions);

    return transition;
}

bool TransitionStrategy::AddListener(EventListener& listener)
{
    if (!m_eventListeners.Contains(&listener)) {
        return m_eventListeners.Append(&listener);
    }
    return false;
}

bool TransitionStrategy::RemoveListener(EventListener& listener)
{
    return m_eventListeners.Remove(&listener);
}

void TransitionStrategy::SetAnimationTimeDispatcher(Animation::AnimationTimeDispatcher::SharedPointer& timeDispatcher) const
{
    GetDefaultFragmentFactory().SetAnimationTimeDispatcher(timeDispatcher);
}

Animation::AnimationTimeDispatcher::SharedPointer TransitionStrategy::GetAnimationTimeDispatcher() const
{
    return GetDefaultFragmentFactory().GetAnimationTimeDispatcher();
}

void TransitionStrategy::Update(TimeType deltaTime)
{
    UpdateInternal(deltaTime);
}

void TransitionStrategy::Prepend(const Transition::SharedPointer& transition)
{
    if (!transition.PointsToNull()) {
        transition->m_nextTransition = m_activeTransitions.m_firstTransition;
        m_activeTransitions.m_firstTransition = transition;
    }
}

void TransitionStrategy::Notify(const Event& event)
{
    for (EventListenerContainer::Iterator it = m_eventListeners.Begin(); it != m_eventListeners.End(); ++it) {
        EventListener* listener = *it;
        FEATSTD_DEBUG_ASSERT(listener != 0); // Assured by AddListener function. m_eventListeners is private.
        static_cast<void>(listener->OnEvent(event));
    }
}

void TransitionStrategy::UpdateInternal(TimeType deltaTime) 
{
    Transition* transition = m_activeTransitions.GetFirstTransition().GetPointerToSharedInstance();
    Transition* previous = 0;

    // Iterate through all active transitions and update their respective transition fragments
    while (transition != 0) {

        // Change the transition's status that its execution has started if it's not being executed yet
        bool isPendingExecution = transition->IsPendingExecution();
        if (isPendingExecution) {
            Notify(OnStartTransition(Transition::SharedPointer(transition)));
            transition->m_isPendingExecution = false;
        }

        // Iterate through all transition fragments within the current transition
        TransitionFragment* transitionFragment = transition->GetFirstFragment().GetPointerToSharedInstance();
        bool isFinished = true; // Set to false if a fragment has not completed yet.
        while (transitionFragment != 0) {
            if (!transitionFragment->IsFinished()) {

                if (isPendingExecution) {
                    // First execution of this fragment
                    Notify(OnStartTransitionFragment(TransitionFragment::SharedPointer(transitionFragment)));
                }

                transitionFragment->Update(deltaTime);

                if(transitionFragment->IsFinished()) {
                    // Transition fragment has completed
                    Notify(OnFinishTransitionFragment(TransitionFragment::SharedPointer(transitionFragment)));
                }
                // Transition will only be removed when all fragments have completed
                isFinished = isFinished && transitionFragment->IsFinished();
            }
            transitionFragment = transitionFragment->GetNext().GetPointerToSharedInstance();
        }

        if (!isFinished) {
            previous = transition;
            transition = transition->GetNext().GetPointerToSharedInstance();
        } 
        else {
            // Transition has completed
            Notify(OnFinishTransition(Transition::SharedPointer(transition)));

            // remove transition from currently active transitions.
            Transition::SharedPointer nextTransition = transition->GetNext();
            if (previous != 0) {
                // remove element.
                FEATSTD_DEBUG_ASSERT(previous->m_nextTransition == transition);
                previous->m_nextTransition = nextTransition;
            }
            else {
                // remove head of the list.
                FEATSTD_DEBUG_ASSERT(m_activeTransitions.GetFirstTransition().GetPointerToSharedInstance() == transition); // current element is head of the list
                m_activeTransitions.m_firstTransition = nextTransition;
             }
            transition = nextTransition.GetPointerToSharedInstance();
         }
    }
}

}   // namespace Candera
