//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TRANSITION_FRAGMENT_FACTORY_H)
#define CANDERA_TRANSITION_FRAGMENT_FACTORY_H

#include <Candera/EngineBase/Animation/AnimationTimeDispatcher.h>

#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <CanderaTransitions/Identifier.h>

namespace Candera {
namespace Transitions {
/// @addtogroup Transition
/// @{
    class Trigger;
    class TransitionFragment;

    /**
    * @brief Base class for concrete transition fragment factories.
    * When implementing custom transition fragments also implement a factory by deriving from this class.
    */
    class TransitionFragmentFactory
    {
    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(TransitionFragmentFactory);

        /**
        *  Create a TransitionFragment with the passed Trigger holding information about from which Rule, RequestFragments and Hints this transition fragment was generated.
        *  A transition fragment's behaviour can be parameterized based on the data included in the Trigger.
        *
        *  @param trigger A shared pointer to a Trigger instance.
        *  @return If creation was successful a shared pointer to the TransitionFragment. If creation failed the shared pointer will point to null.
        */
        virtual MemoryManagement::SharedPointer<TransitionFragment> Create(const MemoryManagement::SharedPointer<Trigger>& trigger);

    protected:

        /**
         *  Can hold a pointer to either a Node, Scene, Node2D, Scene2D or a custom pointer. Which one is set is specified by m_type, the other members are invalid.
         */
        struct Artifact {

            Artifact() :
                m_type(Identifier::NoneBuiltInIdentifier),
                m_custom(0)
            {}

            Identifier::Type m_type; ///< The identifier type from which this artifact was generated. Should be set to NoneBuiltInIdentifier if invalid.
            union {
#ifdef CANDERA_3D_ENABLED
                Node* m_node;        ///< For Node3D identifiers.
                Scene* m_scene;      ///< For Scene3D identifiers.
#endif
#ifdef CANDERA_2D_ENABLED
                Node2D* m_node2D;    ///< For Node2D identifiers.
                Scene2D* m_scene2D;  ///< For Scene2D identifiers.
#endif
                void* m_custom;      ///< For custom identifiers.
            };
        };

        /**
        *  Resolve a given identifier. Override this function if custom asset loading/resolve logic is required.
        *  @param identifier the identifier to resolve.
        *  @return The resolved artifact in form of a Identifier::Type and a pointer. If the pointer is invalid the Artifact type will be set to NoneBuiltInIdentifier.
        */
        virtual Artifact Resolve(const Identifier& identifier) const;

    private:
        CANDERA_SHARED_POINTER_DECLARATION();

        /// AnimationTimeDispatcher
        Animation::AnimationTimeDispatcher::SharedPointer m_animationTimeDispatcher;
    };

/// @}
}   // namespace Transitions
}   // namespace Candera

#endif // CANDERA_TRANSITION_FRAGMENT_FACTORY_H
