//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TransitionFragmentFactory.h"

#include <CanderaTransitions/TransitionFragment.h>
#include <CanderaTransitions/TransitionTraceMapper.h>

#include <FeatStd/Util/StaticObject.h>
#include <CanderaAssetLoader/AssetLoaderBase/ContentLoader.h>
#ifdef CANDERA_3D_ENABLED
#include <CanderaAssetLoader/AssetLoader3D/SceneContext.h>
#endif
#ifdef CANDERA_2D_ENABLED
#include <CanderaAssetLoader/AssetLoader2D/Scene2DContext.h>
#endif
#include <Candera/System/MemoryManagement/CanderaHeap.h>

#ifdef FEATSTD_LOG_ENABLED
#include <Candera/System/Diagnostics/Log.h>
#endif

namespace Candera {

FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaTransitions);

using namespace Transitions;

TransitionFragment::SharedPointer TransitionFragmentFactory::Create(const Trigger::SharedPointer& trigger)
{
    FEATSTD_UNUSED(trigger);
    return TransitionFragment::SharedPointer();
}

TransitionFragmentFactory::Artifact TransitionFragmentFactory::Resolve(const Identifier& identifier) const
{
    TransitionFragmentFactory::Artifact artifact;
    artifact.m_type = Identifier::NoneBuiltInIdentifier; // Invalid by default. Only set if target was fetched successfully.

    AssetProvider* assetProvider = ContentLoader::GetInstance().GetAssetProvider();
    if (assetProvider == 0) {
        FEATSTD_LOG_WARN("No asset provider available.");
        return artifact; //Invalid artifact
    }

    switch (static_cast<Identifier::Type>(identifier.GetType())) {

#ifdef CANDERA_3D_ENABLED
        case Identifier::Node3DBuiltInIdentifier:
        {
            Node* node = assetProvider->GetNodeById(identifier.GetSceneId(), identifier.GetNodeId());
            if (node != 0) {
                artifact.m_node = node;
                artifact.m_type = identifier.GetType();
            }
            break;
        }
        case Identifier::Scene3DBuiltInIdentifier:
        {
            SceneContext* sceneContext = assetProvider->GetSceneById(identifier.GetSceneId());
            Scene* scene = (sceneContext != 0) ? sceneContext->GetScene() : 0;
            if (scene != 0) {
                artifact.m_scene = scene;
                artifact.m_type = identifier.GetType();
            }
            break;
        }
#endif

#ifdef CANDERA_2D_ENABLED
        case Identifier::Node2DBuiltInIdentifier:
        {
            Node2D* node = assetProvider->GetNode2DById(identifier.GetSceneId(), identifier.GetNodeId());
            if (node != 0) {
                artifact.m_node2D = node;
                artifact.m_type = identifier.GetType();
            }
            break;
        }
        case Identifier::Scene2DBuiltInIdentifier:
        {
            Scene2DContext* sceneContext = assetProvider->GetScene2DById(identifier.GetSceneId());
            Scene2D* scene = (sceneContext != 0) ? sceneContext->GetScene() : 0;
            if (scene != 0) {
                artifact.m_scene2D = scene;
                artifact.m_type = identifier.GetType();
            }
            break;
        }
#endif

        default:
        {
            FEATSTD_LOG_WARN("Unsupported identifier type, cannot resolve: %s.", TransitionTraceMapper::MapIdentifierTypeToString(identifier.GetType()));
            break;
        }
    }

    return artifact;
}

}   // namespace Candera
