//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TRANSITION_RULE_H)
#define CANDERA_TRANSITION_RULE_H

#include <Candera/Environment.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <FeatStd/Util/Optional.h>
#include <CanderaTransitions/TransitionFragmentFactory.h>
#include <CanderaTransitions/Identifier.h>
#include <CanderaTransitions/Hint.h>
#include <CanderaTransitions/RequestFragment.h>

namespace Candera {

namespace Internal {
    class RuleAdder;
} // namespace Internal

namespace Transitions {
/// @addtogroup Transition
/// @{

    /**
    * @brief Class to store transition rules. 
    * The TransitionStrategy matches RequestFragments to Rules. Each rule can define a TransitionFragmentFactory to generate a TransitionFragment when the rule is applied. 
    */
    class Rule
    {
    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(Rule);

        /**
         *  Constructor
         *  @param source
         *  @param destination
         *  @param hint
         *  @param factory
         *  @param allowReverse
         */
        Rule(Identifier source, Identifier destination, Hint hint, const TransitionFragmentFactory::SharedPointer& factory, bool allowReverse = false, bool isBidirectional = false);

        const Identifier& GetSource() const { return m_source; }
        const Identifier& GetDestination() const { return m_destination; }
        const Hint& GetHint() const { return m_hint; }
        bool IsReverseAllowed() const { return m_isReverseAllowed; }
        bool IsBidirectional() const { return m_isBidirectional; }
        /**
         *  Get the next rule in the set.
         *  @return A shared pointer to the next rule in the set; will point to null if this is the last rule.
         */
        const Rule::SharedPointer& GetNext() const { return m_next; }

        /**
         *  @brief Rule set, holds a shared pointer to the first rule.
         */
        class Set {
        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(Set);

            /**
             *  Get the first rule of the set.
             *  @return A shared pointer to the first rule in the set; will point to null if the set is empty.
             */
            const Rule::SharedPointer& GetFirstRule() const { return m_firstRule; }

        private:
            friend class Internal::RuleAdder;
            Set(); //Set is not instantiable by user.

            Rule::SharedPointer m_firstRule;

            CANDERA_SHARED_POINTER_DECLARATION();
        };


    private:
        friend class Internal::RuleAdder;
        friend class TransitionStrategy;

        Identifier m_source;
        Identifier m_destination;
        Hint m_hint;
        TransitionFragmentFactory::SharedPointer m_factory;

        Rule::SharedPointer m_next;

        bool m_isReverseAllowed;
        bool m_isBidirectional;

        CANDERA_SHARED_POINTER_DECLARATION();
    };

/// @}
}   // namespace Transitions
}   // namespace Candera

#endif // CANDERA_TRANSITION_RULE_H
