//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TRANSITION_REQUEST_FRAGMENT_H)
#define CANDERA_TRANSITION_REQUEST_FRAGMENT_H

#include <Candera/Environment.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

#include <CanderaTransitions/Identifier.h>
#include <CanderaTransitions/Hint.h>

namespace Candera {
namespace Transitions {
/// @addtogroup Transition
/// @{

/**
 * @brief Holds a transition request for one specific element, identified by an Identifier.
 *
 * A request fragment can have be one of three types:
 *  - Deactivation: Deactivate the identified element.
 *  - Activation: Activate the identified element.
 *  - Finish: End all currently active fragments acting on the identified element.
 */
class RequestFragment
{
public:
    FEATSTD_TYPEDEF_SHARED_POINTER(RequestFragment);

    typedef enum {
        Deactivate = 0, ///< Deactivate the identified element.
        Activate,       ///< Activate the identified element.
        Finish          ///< End all currently active fragments acting on the identified element.
    } Type;

    /**
     * Get the identifier for this request fragment.
     * @return a reference to the identifier for which this request fragment was created.
     */
    const Identifier& GetIdentifier() const { return m_identifier; }

    /**
     * Get the Hint associated with this request fragment.
     * @return a reference to the hint.
     */
    const Hint& GetHint() const { return m_hint; }
    
    /**
     * Get the type of this request fragment.
     * @return a reference to the type of this request fragment.
     */
    const FeatStd::Initialized<Type>& GetType() const { return m_type; }

    /**
     * Get the next request fragment in the set.
     * @return a shared pointer to the next request fragment.
     */
    const SharedPointer& GetNext() const { return m_nextFragment; }

private:
    friend class Request;
    friend class TransitionStrategy;

    RequestFragment(const Identifier& identifier, const Hint& hint, Type type);
    RequestFragment(const RequestFragment&);

    Identifier m_identifier;
    Hint m_hint;
    FeatStd::Initialized<Type> m_type;
    SharedPointer m_nextFragment;

    CANDERA_SHARED_POINTER_DECLARATION();
};

/// @}
}   // namespace Transitions
}   // namespace Candera

#endif // CANDERA_TRANSITION_REQUEST_FRAGMENT_H
