//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TRANSITION_REQUEST_H)
#define CANDERA_TRANSITION_REQUEST_H

#include <Candera/Environment.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

#include <CanderaTransitions/RequestFragment.h>

namespace Candera {
namespace Transitions {
/// @addtogroup Transition
/// @{

/**
* A set of request fragments.
* @brief
*/
    class Request
    {
    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(Request);
        
        /**
         * Handle for transition requests. Use the Handle if the transition request may be interrupted
         * by other unpredictable/independent transition requests.
         */
        class Handle {
        public:
            Handle() {}

            bool IsRequestStarted() const { return !m_request.PointsToNull(); }

        private:
            friend class Request;

            Handle(const Handle& handle);

            Handle(const Request::SharedPointer& request, const RequestFragment::SharedPointer& lastRequestFragment) :
                m_request(request),
                m_lastRequestFragment(lastRequestFragment)
            {
            }

            mutable Request::SharedPointer m_request;
            mutable RequestFragment::SharedPointer m_lastRequestFragment;
        };

        /**
        *  Called to begin composing a request. Call Activate, Deactivate and Finish functions to add 
        *  corresponding RequestFragments to the Request. EndRequest will end composition and return the composed Request.
        */
        static void Begin();
        static void Begin(Handle& handle);

        /**
         * Returns true if a request has already started (between the calls of the method Begin the method End).
         */
        static bool IsRequestStarted();

        /**
        *  Adds an activation RequestFragment for an element referenced by identifier with the given hint.
        *  
        *  Will have no effect if no request has been begun, see BeginRequest.
        *  @param identifier
        *  @param hint
        *  @param handle
        */
        static void Activate(const Identifier& identifier, const Hint& hint, const Request::Handle& handle = Handle());

        /**
        *  Adds an activation RequestFragment for an element referenced by identifier with the given hint.
        *
        *  Will have no effect if no request has been begun, see BeginRequest.
        *  @param identifier
        *  @param hint
        *  @param handle
        */
        static void Deactivate(const Identifier& identifier, const Hint& hint, const Handle& handle = Handle());

        /**
        *  Adds a finish RequestFragment for an identifier and hint combination. 
        *  Finish is used to end any TransitionFragments currently executing for the matching identifier and hint. 
        *  The exact behaviour is up to the fragment, typically it will set the target state and mark itself as finished.
        *
        *  Will have no effect if no request has been begun, see BeginRequest.
        *  @param identifier
        *  @param hint
        *  @param handle
        */
        static void Finish(const Identifier& identifier, const Hint& hint, const Handle& handle = Handle());

        /**
        *  Completes the current request and returns a shared pointer to the generated Request instance.
        *  @return A shared pointer to the compiled Request, which will point to null if no request was begun, see BeginRequest.
        */
        static Request::SharedPointer End(const Handle& handle = Handle());

        /**
         *  Fetches the first RequestFragment in the Request.
         *  @return A shared pointer to the first RequestFragment, if the Request is empty this will point to null.
         */
        const RequestFragment::SharedPointer& GetFirstFragment() const { return m_firstFragment; }

        /**
         *  Removes fragment from request set.
         *  @param fragment The request fragment to be removed.
         *  @return true if the fragment was found and removed, false if the fragment is not contained in the request or points to null.
         */
        bool Remove(const RequestFragment::SharedPointer& fragment);

    private:

        RequestFragment::SharedPointer m_firstFragment;

        static void AddRequestFragment(const RequestFragment::SharedPointer& requestFragment, const Request::Handle& handle = Handle());

        CANDERA_SHARED_POINTER_DECLARATION();
    };

/// @}
}   // namespace Transitions
}   // namespace Candera

#endif // CANDERA_TRANSITION_REQUEST_H
