//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Request.h"
#include <FeatStd/Util/StaticObject.h>
#include <Candera/System/MemoryManagement/CanderaHeap.h>

namespace Candera {

using namespace Transitions;

static Request::Handle& CurrentTransitionRequestHandle()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(Request::Handle, s_lastTransitionHandle);
    return s_lastTransitionHandle;
}

static const Request::Handle& GetCurrentTransitionRequestHandle(const Request::Handle& handle)
{
    return (handle.IsRequestStarted()) ? handle : CurrentTransitionRequestHandle();
}

static Request::Handle& s_initLastTransitionRequestHandle = CurrentTransitionRequestHandle();

void Request::Begin(Request::Handle& handle)
{
#if defined(FEATSTD_DEBUG)
    const Request::Handle& currentHandle = GetCurrentTransitionRequestHandle(handle);
    FEATSTD_DEBUG_ASSERT(!currentHandle.IsRequestStarted());
#endif
    FEATSTD_DEBUG_ASSERT(handle.m_lastRequestFragment.PointsToNull());
    handle.m_request = Request::SharedPointer(CANDERA_NEW(Request)());
}

void Request::Begin()
{
    Begin(CurrentTransitionRequestHandle());
}

bool Request::IsRequestStarted()
{
    return GetCurrentTransitionRequestHandle(CurrentTransitionRequestHandle()).IsRequestStarted();
}

void Request::Activate(const Identifier& identifier, const Hint& hint, const Request::Handle& handle)
{
    AddRequestFragment(RequestFragment::SharedPointer(CANDERA_NEW(RequestFragment)(identifier, hint, RequestFragment::Activate)), handle);
}

void Request::Deactivate(const Identifier& identifier, const Hint& hint, const Request::Handle& handle)
{
    AddRequestFragment(RequestFragment::SharedPointer(CANDERA_NEW(RequestFragment)(identifier, hint, RequestFragment::Deactivate)), handle);
}

void Request::Finish(const Identifier& identifier, const Hint& hint, const Request::Handle& handle)
{
    AddRequestFragment(RequestFragment::SharedPointer(CANDERA_NEW(RequestFragment)(identifier, hint, RequestFragment::Finish)), handle);
}


Request::SharedPointer Request::End(const Request::Handle& handle)
{
    const Request::Handle& currentHandle = GetCurrentTransitionRequestHandle(handle);
    FEATSTD_DEBUG_ASSERT(currentHandle.IsRequestStarted());
    Request::SharedPointer result = currentHandle.m_request;
    currentHandle.m_request = Request::SharedPointer();
    currentHandle.m_lastRequestFragment = RequestFragment::SharedPointer();
    return result;
}

void Request::AddRequestFragment(const RequestFragment::SharedPointer& requestFragment, const Request::Handle& handle)
{
    const Request::Handle& currentHandle = GetCurrentTransitionRequestHandle(handle);
    FEATSTD_DEBUG_ASSERT(currentHandle.IsRequestStarted());
    if (currentHandle.IsRequestStarted()) {
        if (!currentHandle.m_lastRequestFragment.PointsToNull()) {
            currentHandle.m_lastRequestFragment->m_nextFragment = requestFragment;
        }
        else {
            currentHandle.m_request->m_firstFragment = requestFragment;
        }
        currentHandle.m_lastRequestFragment = requestFragment;
    }
}

bool Request::Remove(const RequestFragment::SharedPointer& fragment)
{
    if (m_firstFragment.PointsToNull() || fragment.PointsToNull()) {
        // Cannot remove from empty set and cannot remove null pointer.
        return false;
    }

    if (fragment == m_firstFragment) {
        // Remove first element;
        RequestFragment::SharedPointer nextFragment = fragment->GetNext();
        m_firstFragment = nextFragment;
        return true;
    }

    RequestFragment* fragmentPointer = fragment.GetPointerToSharedInstance();
    RequestFragment* previous = m_firstFragment.GetPointerToSharedInstance();
    RequestFragment* current = previous->GetNext().GetPointerToSharedInstance();

    while (current != 0) {
        if (current == fragmentPointer) {
            previous->m_nextFragment = fragmentPointer->GetNext();
            return true;
        }
        previous = current;
        current = current->GetNext().GetPointerToSharedInstance();
    }
    return false;
}

}   // namespace Candera
