//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Identifier.h"

namespace Candera {

using namespace Transitions;

Identifier::Identifier() :
    m_type(NoneBuiltInIdentifier),
    m_data()
{
}

Identifier::Type MapAnySelector(Identifier::AnySelector anySelector)
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, CANDERA_LINT_REASON_RETURNINCASE);
    switch (anySelector) {
    case Identifier::AnyScene2D:
        return Identifier::AnyScene2DBuiltInIdentifier;
    case Identifier::AnyScene3D:
        return Identifier::AnyScene3DBuiltInIdentifier;
    case Identifier::AnyNode2D:
        return Identifier::AnyNode2DBuiltInIdentifier;
    case Identifier::AnyNode3D:
        return Identifier::AnyNode3DBuiltInIdentifier;
    default:
        return Identifier::NoneBuiltInIdentifier;
    }
}

Identifier::Identifier(Identifier::AnySelector anySelector) :
    m_type(MapAnySelector(anySelector))
{
}

Identifier::Identifier(SceneType sceneType, Id sceneId) :
    m_type((sceneType == Scene2DType) ? Scene2DBuiltInIdentifier : Scene3DBuiltInIdentifier),
    m_data(sceneId, 1)
{
}

Identifier::Identifier(Identifier::SceneType sceneType, Id sceneId, Id nodeId) :
    m_type((sceneType == Scene2DType) ?
        Type(((nodeId == Id(0)) || (nodeId == Id(1))) ? Scene2DBuiltInIdentifier : Node2DBuiltInIdentifier) :
        Type(((nodeId == Id(0)) || (nodeId == Id(1))) ? Scene3DBuiltInIdentifier : Node3DBuiltInIdentifier)),
    m_data(sceneId, (nodeId == Id(0)) ? Id(1) : nodeId)
{
}

Identifier::Identifier(const CustomIdentifier::SharedPointer& customIdentifier) :
    m_type(ExternCustomIdentifier),
    m_data(customIdentifier.GetPointerToSharedInstance())
{
    Reference();
}

Identifier::Identifier(const Identifier& identifier) :
    m_type(identifier.m_type)
{
    if (identifier.m_type == ExternCustomIdentifier) {
        m_data.m_customIdentifier = identifier.m_data.m_customIdentifier;
        Reference();
    }
    //static cast required for certain gcc versions due to a compiler bug. 
    switch (static_cast<Identifier::Type>(identifier.GetType())) {
    case Scene2DBuiltInIdentifier:
    case Scene3DBuiltInIdentifier:
    case Node2DBuiltInIdentifier:
    case Node3DBuiltInIdentifier:
        m_data.m_id.m_sceneId = identifier.m_data.m_id.m_sceneId;
        m_data.m_id.m_nodeId = identifier.m_data.m_id.m_nodeId;
        break;
    default:
        break;
    }
}

Identifier::~Identifier()
{
    Dereference();
}

bool Identifier::operator==(const Identifier& rhs) const
{
    return ((m_type == rhs.m_type) && (m_data == rhs.m_data));
}


static bool IsObjectIdentifier(Identifier::Type type)
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, CANDERA_LINT_REASON_RETURNINCASE);
    switch (type) {
    case Identifier::Scene2DBuiltInIdentifier:
    case Identifier::Scene3DBuiltInIdentifier:
    case Identifier::Node2DBuiltInIdentifier:
    case Identifier::Node3DBuiltInIdentifier:
        return true;
    default:
        return false;
    }
}

static bool IsNodeIdentifier(Identifier::Type type)
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, CANDERA_LINT_REASON_RETURNINCASE);
    switch (type) {
    case Identifier::Node2DBuiltInIdentifier:
    case Identifier::Node3DBuiltInIdentifier:
    case Identifier::AnyNode2DBuiltInIdentifier:
    case Identifier::AnyNode3DBuiltInIdentifier:
        return true;
    default:
        return false;
    }
}

static Identifier::SceneType GetIdentifierSceneType(Identifier::Type type)
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, CANDERA_LINT_REASON_RETURNINCASE);
    switch (type) {
    case Identifier::Scene2DBuiltInIdentifier:
    case Identifier::Node2DBuiltInIdentifier:
    case Identifier::AnyScene2DBuiltInIdentifier:
    case Identifier::AnyNode2DBuiltInIdentifier:
        return Identifier::Scene2DType;
    default:
        return Identifier::Scene3DType;
    }
}

/**
 *  Helper function matching two Identifier types, where anyType must be one of the AnyScene or AnyNode identifier types.
 */
static bool MatchInternal(Identifier::Type anyType, Identifier::Type objectType)
{
    FEATSTD_DEBUG_ASSERT(!IsObjectIdentifier(anyType));

    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, CANDERA_LINT_REASON_RETURNINCASE);
    switch (anyType) {
    case Identifier::AnyScene2DBuiltInIdentifier:
        return (Identifier::Scene2DType == GetIdentifierSceneType(objectType)) && (!IsNodeIdentifier(objectType));
    case Identifier::AnyScene3DBuiltInIdentifier:
        return (Identifier::Scene3DType == GetIdentifierSceneType(objectType)) && (!IsNodeIdentifier(objectType));
    case Identifier::AnyNode2DBuiltInIdentifier:
        return (Identifier::Scene2DType == GetIdentifierSceneType(objectType)) && IsNodeIdentifier(objectType);
    case Identifier::AnyNode3DBuiltInIdentifier:
        return (Identifier::Scene3DType == GetIdentifierSceneType(objectType)) && IsNodeIdentifier(objectType);
    default:
        return false;
    }
}

bool Identifier::Matches(const Identifier& identifier) const
{
    if (this == &identifier) {
        return true;
    }
    if ((ExternCustomIdentifier == m_type) || (ExternCustomIdentifier == identifier.m_type)){
        if (identifier.m_type != m_type) {
            return false;
        }
        if (identifier.m_data.m_customIdentifier == m_data.m_customIdentifier) {
            return true;
        }
        if (0 == m_data.m_customIdentifier) {
            return false;
        }
        if (0 == identifier.m_data.m_customIdentifier) {
            return false;
        }
        return m_data.m_customIdentifier->Matches(*identifier.m_data.m_customIdentifier);
    }
    if (IsObjectIdentifier(m_type) && IsObjectIdentifier(identifier.m_type)) {
        return (m_type == identifier.m_type) && (m_data.m_id.m_sceneId == identifier.m_data.m_id.m_sceneId) && (m_data.m_id.m_nodeId == identifier.m_data.m_id.m_nodeId);
    }
    if (GetIdentifierSceneType(m_type) != GetIdentifierSceneType(identifier.m_type)) {
        return false;
    }
    if (IsObjectIdentifier(m_type)) {
        return MatchInternal(identifier.m_type, m_type);
    }
    if (IsObjectIdentifier(identifier.m_type)) {
        return MatchInternal(m_type, identifier.m_type);
    }
    if (m_type == identifier.m_type) {
        return true;
    }

    return false;
}

Id Identifier::GetSceneId() const
{
    FEATSTD_DEBUG_ASSERT(GetType() != ExternCustomIdentifier);
    return m_data.m_id.m_sceneId;
}

Id Identifier::GetNodeId() const
{
    FEATSTD_DEBUG_ASSERT(GetType() != ExternCustomIdentifier);
    return m_data.m_id.m_nodeId;
}

void Identifier::Reference()
{
    if ((ExternCustomIdentifier == m_type) && (0 != m_data.m_customIdentifier)) {
        m_data.m_customIdentifier->Retain();
    }
}

void Identifier::Dereference()
{
    if ((ExternCustomIdentifier == m_type) && (0 != m_data.m_customIdentifier)) {
        m_data.m_customIdentifier->Release();
        m_data.m_customIdentifier = 0;
    }
}

Identifier::Data::Data(Id sceneId, Id nodeId)
{
    m_id.m_sceneId = sceneId;
    m_id.m_nodeId = nodeId;
}

Identifier::Data::Data(CustomIdentifier* customIdentifier)
{
    m_id.m_sceneId = 0;
    m_id.m_nodeId = 0;
    m_customIdentifier = customIdentifier;
}

Identifier::Data::Data()
{
    m_id.m_sceneId = 0;
    m_id.m_nodeId = 0;
}

}   // namespace Candera
