//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TRANSITIONS_HINT_H)
#define CANDERA_TRANSITIONS_HINT_H

#include <FeatStd/Util/String.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/System/Mathematics/Vector3.h>

#include <FeatStd/Util/Optional.h>
#include <CanderaTransitions/CustomHint.h>


namespace Candera {

namespace Transitions {
/// @addtogroup Transition
/// @{

/**
* @brief A Hint can be set on a Rule and a transition RequestFragment. The TransitionStrategy will match transition 
* requests to rules based on criteria such as the Hint and create TransitionFragments accordingly. 
* A TransitionFragment's behaviour can be parameterized based on the data included in the Hint.
*/
class Hint
{
public:

    /**
    *  @brief Used to describe the activation / deactivation strategy of a transition.
    */
    enum FragmentStrategy {
        Normal = 0,
        Early,
        Late
    };

    /**
     *  @brief Holds additional parameters to be passed with the Hint.
     *  Some built-in parameters are included, custom parameters can be added by adding an instance of CustomHint 
     */
    class Data
    {
    public:
        const FeatStd::Optional<Float>& Fade() const { return m_fade; }
        const FeatStd::Optional<Vector2>& FadeTimeline() const { return m_fadeTimeline; }

        const FeatStd::Optional<Vector3>& Slide() const { return m_slide; }
        const FeatStd::Optional<Vector2>& SlideTimeline() const { return m_slideTimeline; }
        bool IsSlideRelative() const { return m_isSlideRelative; }

        const FeatStd::Optional<Vector3>& Scale() const { return m_scale; }
        const FeatStd::Optional<Vector2>& ScaleTimeline() const { return m_scaleTimeline; }
        bool IsScaleRelative() const { return m_isScaleRelative; }

        const FeatStd::Optional<Candera::Id>& Animation() const { return m_animationId; }
        const FeatStd::Optional<Vector2>& AnimationTimeline() const { return m_animationTimeline; }
        bool IsAnimationTimeRelative() const { return m_isAnimationTimeRelative; }

        const FeatStd::Optional<FragmentStrategy>& ActivationStrategy() const { return m_activationStrategy; }
        const FeatStd::Optional<FragmentStrategy>& DeactivationStrategy() const { return m_deactivationStrategy; }

        const FeatStd::Optional<Float>& ActivationDelay() const { return m_activationDelay; }
        const FeatStd::Optional<Float>& DeactivationDelay() const { return m_deactivationDelay; }


        const CustomHint::SharedPointer& GetCustomHint() const { return m_customHint; }

    private:
        template <typename T> friend class FeatStd::Initialized;
        template <typename T> friend class FeatStd::Optional;
        friend class Hint;

        Data(const FeatStd::Optional<Float>& fade, const FeatStd::Optional<Vector2>& fadeTimeline, const FeatStd::Optional<Vector3>& slide, const FeatStd::Optional<Vector2>& slideTimeline, bool isSlideRelative, const FeatStd::Optional<Vector3>& scale, const FeatStd::Optional<Vector2>& scaleTimeline, bool isScaleRelative, 
            const FeatStd::Optional<Candera::Id>& animationId, const FeatStd::Optional<Vector2>& animationTimeline, bool isAnimationTimeRelative, const FeatStd::Optional<FragmentStrategy>& activationStrategy, const FeatStd::Optional<FragmentStrategy>& deactivationStrategy,
            const FeatStd::Optional<Float>& activationDelay, const FeatStd::Optional<Float>& deactivationDelay, const CustomHint::SharedPointer& customHint);
        Data(const Data& data);
        Data();

        //builtin transition types
        FeatStd::Optional<Float> m_fade;
        FeatStd::Optional<Vector2> m_fadeTimeline;

        FeatStd::Optional<Vector3> m_slide;
        FeatStd::Optional<Vector2> m_slideTimeline;
        bool m_isSlideRelative;

        FeatStd::Optional<Vector3> m_scale;
        FeatStd::Optional<Vector2> m_scaleTimeline;
        bool m_isScaleRelative;

        FeatStd::Optional<Candera::Id> m_animationId;
        FeatStd::Optional<Vector2> m_animationTimeline;
        bool m_isAnimationTimeRelative;

        FeatStd::Optional<FragmentStrategy> m_activationStrategy;
        FeatStd::Optional<FragmentStrategy> m_deactivationStrategy;

        FeatStd::Optional<Float> m_activationDelay;
        FeatStd::Optional<Float> m_deactivationDelay;

        // custom data
        CustomHint::SharedPointer m_customHint;

        bool operator==(const Data& rhs) const;
    };

    /**
     *  Default constructor for a simple hint used when composing transition requests.
     *  @param name The string to match Rules to RequestFragments.
     */
    Hint(const FeatStd::String& name = FeatStd::String(""));

    /**
    *  This constructor is usually used when composing Rules as the DefaultTransitionStrategy will ignore optional parameters set in the RequestFragment.
    *  There are two parameters for each builtin transition type, the builtin types being fade, slide and scale.
    *  The first parameter defines a target value for the builtin transition, the second parameter defines a timeline for when the transition should start and end.
    *  The DefaultTransitionFragment interprets the timeline values as being in milliseconds.
    *
    *  @param name The string to match Rules to RequestFragments.
    *  @param fade Optional parameter for the fade target value.
    *  @param fadeTimeline Optional parameter for the fade timeline.
    *  @param slide Optional parameter for the target Vector3 position.
    *  @param slideTimeline Optional parameter for the slide timeline.
    *  @param scale Optional parameter for the target Vector3 scale.
    *  @param scaleTimeline Optional parameter for the scale timeline.
    *  @param animationId Optional parameter for the Animation Id.
    *  @param animationTimeline Optional parameter for the animation timeline.
    *  @param customHint CustomHint instance.
    *  @param isSlideRelative default parameter specifies whether or not the given slide parameter is in relative or absolute coordinates. The default value is true.
    *  @param isScaleRelative default parameter specifies whether or not the given scale parameter is in relative or absolute coordinates. The default value is true.
    *  @param isAnimationTimeRelative default parameter specifies whether or not the given animation parameter is relative to the given animation timeline. If this is true,
    *         the animation speed is adapted such that transition timeline end time matches animation end. The default value is true.
    */
    CANDERA_DEPRECATED_3_6_0("The Hint constructor has been replaced with an updated constructor that includes strategy and delay options for the activation and deactivation fragments.",
    Hint(const FeatStd::String& name,
        const FeatStd::Optional<Float>& fade, const FeatStd::Optional<Vector2>& fadeTimeline,
        const FeatStd::Optional<Vector3>& slide, const FeatStd::Optional<Vector2>& slideTimeline,
        const FeatStd::Optional<Vector3>& scale, const FeatStd::Optional<Vector2>& scaleTimeline,
        const FeatStd::Optional<Candera::Id>& animationId, const FeatStd::Optional<Vector2>& animationTimeline,
        const CustomHint::SharedPointer& customHint,
        bool isSlideRelative = true, bool isScaleRelative = true, bool isAnimationTimeRelative = true));

    /**
     *  This constructor is usually used when composing Rules as the DefaultTransitionStrategy will ignore optional parameters set in the RequestFragment.
     *  There are two parameters for each builtin transition type, the builtin types being fade, slide and scale.
     *  The first parameter defines a target value for the builtin transition, the second parameter defines a timeline for when the transition should start and end.
     *  The DefaultTransitionFragment interprets the timeline values as being in milliseconds.
     *
     *  @param name The string to match Rules to RequestFragments.
     *  @param fade Optional parameter for the fade target value.
     *  @param fadeTimeline Optional parameter for the fade timeline.
     *  @param slide Optional parameter for the target Vector3 position.
     *  @param slideTimeline Optional parameter for the slide timeline.
     *  @param scale Optional parameter for the target Vector3 scale.
     *  @param scaleTimeline Optional parameter for the scale timeline.
     *  @param animationId Optional parameter for the Animation Id.
     *  @param animationTimeline Optional parameter for the animation timeline.
     *  @param activationStrategy Optional parameter to describe the RequestFragment's activation strategy.
     *  @param deactivationStrategy Optional parameter to describe the RequestFragment's deactivation strategy.
     *  @param activationDelay Optional parameter to describe the delay in milliseconds before the activation fragment is executed.
     *  @param deactivationDelay Optional parameter to describe the delay in milliseconds before the deactivation fragment is executed.
     *  @param customHint CustomHint instance.
     *  @param isSlideRelative default parameter specifies whether or not the given slide parameter is in relative or absolute coordinates. The default value is true.
     *  @param isScaleRelative default parameter specifies whether or not the given scale parameter is in relative or absolute coordinates. The default value is true.
     *  @param isAnimationTimeRelative default parameter specifies whether or not the given animation parameter is relative to the given animation timeline. If this is true,
     *         the animation speed is adapted such that transition timeline end time matches animation end. The default value is true.
     */
     Hint(const FeatStd::String& name, 
         const FeatStd::Optional<Float>& fade, const FeatStd::Optional<Vector2>& fadeTimeline,
         const FeatStd::Optional<Vector3>& slide, const FeatStd::Optional<Vector2>& slideTimeline,
         const FeatStd::Optional<Vector3>& scale, const FeatStd::Optional<Vector2>& scaleTimeline,
         const FeatStd::Optional<Candera::Id>& animationId, const FeatStd::Optional<Vector2>& animationTimeline,
         const FeatStd::Optional<FragmentStrategy>& activationStrategy,
         const FeatStd::Optional<FragmentStrategy>& deactivationStrategy,
         const FeatStd::Optional<Float>& activationDelay,
         const FeatStd::Optional<Float>& deactivationDelay,
         const CustomHint::SharedPointer& customHint, 
         bool isSlideRelative = true, bool isScaleRelative = true, bool isAnimationTimeRelative = true);
    
    /**
     *  Copy constructor
     */
    Hint(const Hint& hint);

    /**
     *  Fetch the name of this Hint instance.
     */
    const FeatStd::String& GetName() const { return m_name; }

    /**
     *  data included in with the hint.
     */
    const FeatStd::Optional<Data>& GetDetails() const;

    bool operator==(const Hint& rhs) const {
        return ((m_name == rhs.m_name) && (m_details == rhs.m_details));
    }

private:
    FeatStd::String m_name;
    FeatStd::Optional<Data> m_details;
};

/// @}
}   // namespace Transitions
}   // namespace Candera

#endif // CANDERA_TRANSITIONS_HINT_H
