//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_SCRIPTING_SCRIPTCOMPONENT_H
#define CANDERA_SCRIPTING_SCRIPTCOMPONENT_H

#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/System/EntityComponentSystem/Component.h>
#include <Candera/System/EntityComponentSystem/ComponentSystem.h>
#include <CanderaPlatform/OS/CanderaTypes.h>
#include <CanderaScripting/Script.h>

namespace Candera {

namespace Scripting {

/** @addtogroup Scripting
 *  @{
 */

typedef CanderaObject ScriptEntity;

/**
 * @brief  ScriptComponent can be attached to any entity. If the component is attached, its script is executed by
 *         the associated script system. If it gets detached, the script is removed from the system.
 */

class ScriptComponent : public EntityComponentSystem::Component<ScriptEntity>
{

public:
    FEATSTD_RTTI_DECLARATION();

    enum ParameterType {
        Number,
        Integer,
        Boolean,
        String,
        ObjectReference,
        None
    };

    /**
     *  Destructor
     */
    virtual ~ScriptComponent();

    /**
     *  Get the script of this component. Note: A script is set using ScriptSystem::SetComponentScript.
     *  @return  The script of this component.
     */
    MemoryManagement::SharedPointer<Script> GetScript() const { return m_script; }

    /**
     *  Get the priority of this component. Note: The priority is set using ScriptSystem::SetComponentPriority.
     *  @return  The priority of this component.
     */
    Float GetPriority() const { return m_priority; }

    /**
     *  Query if this component is enabled. Note: The enabled flag is set using ScriptSystem::SetComponentEnabled.
     *  @return  True, if this component is enabled, false otherwise.
     */
    bool IsEnabled() const { return m_isEnabled; }

private:
    friend class ScriptSystem;
    friend class ScriptParameters;
    friend class EntityComponentSystem::ComponentSystem<ScriptComponent, ScriptEntity>;

    /**
     *  Constructor
     */
    ScriptComponent();

    MemoryManagement::SharedPointer<Script> m_script;
    Float m_priority;

    struct ParameterInfo {
        ParameterInfo(const Char* name, ParameterType type, TypeId typeId) : m_parameter(type, typeId), m_name(name) {}

        struct Parameter {
            Parameter(ParameterType type, TypeId typeId = TypeId()) : m_typeId(typeId), m_type(type) { m_value.m_string = 0; }

            union Value {
                Double m_double;
                Int64 m_integer;
                Int32 m_handle;
                bool m_boolean;
                const Char* m_string;
            } m_value;

            TypeId m_typeId;
            ParameterType m_type;
        };

        Parameter m_parameter;
        const Char* m_name;
    };

    typedef Candera::Internal::Vector<ParameterInfo, FeatStd::Internal::LinearIncreasePolicy<1> > Parameters;
    Parameters m_parameters;

    bool m_isEnabled;
};

/** @} */ // end of Scripting

} // namespace Scripting

} // namespace Candera

#endif
