//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_SCRIPTING_LUASCRIPTSYSTEM_H
#define CANDERA_SCRIPTING_LUASCRIPTSYSTEM_H

#include <CanderaScripting/ScriptSystem.h>
#include <lua.hpp>

namespace Candera {

namespace EntityComponentSystem { class EntitySystem; }

namespace Scripting {

namespace Internal { class LuaBinding; }

/** @addtogroup LuaScripting
 *  @{
 */

/**
 * @brief  LuaScriptSystem manages a Lua virtual machine running scripts from script components.
 */

class LuaScriptSystem : public ScriptSystem
{
    FEATSTD_TYPEDEF_BASE(ScriptSystem);

public:
    /**
     *  Destructor
     */
    virtual ~LuaScriptSystem() override;

    /**
     *  Initialize the Lua script system.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    virtual bool Init() override;

    /**
     *  Start executing the script system's Update function.
     */
    virtual void Start();

    /**
     *  Stop executing the script system's Update function, and reset the system.
     */
    virtual void Stop();

    /**
     *  Shut the script system down.
     */
    virtual void ShutDown() override;

    /**
     *  Reset the script system
     */
    virtual void Reset() override;

    /**
     *  Notify the system that the contents of an existing script have been changed.
     *  @param script  The script that has been changed.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    virtual bool OnScriptChanged(const Script* script) override;

    /**
     *  Get the type of the parameter identified by its name and the given component
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle         The handle of the component to get the parameter type from.
     *  @param parameterName  The name of the parameter to get the type for.
     *  @return  The ScriptComponent::ParameterType of the parameter identified by the name, or ScriptComponent::None if
     *           the handle could not be resolved, or no parameter for the given name and component exists.
     */
    virtual ScriptComponent::ParameterType GetComponentLiveParameterType(Handle handle, const Char* parameterName) const override;

    /**
     *  Get the double (i.e. ScriptComponent::Number) value of the parameter identified by its name and the
     *  given component from Lua.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to get the parameter from.
     *  @param parameterName   The name of the parameter to get the value for.
     *  @param parameterValue  A reference to where the value of the parameter will be stored.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool GetComponentLiveParameterNumber(Handle handle, const Char* parameterName, Double& parameterValue) const override;

    /**
     *  Get the int (i.e. ScriptComponent::Integer) value of the parameter identified by its name and the
     *  given component from Lua.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to get the parameter from.
     *  @param parameterName   The name of the parameter to get the value for.
     *  @param parameterValue  A reference to where the value of the parameter will be stored.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool GetComponentLiveParameterInteger(Handle handle, const Char* parameterName, Int64& parameterValue) const override;

    /**
     *  Get the bool (i.e. ScriptComponent::Boolean) value of the parameter identified by its name and the
     *  given component from Lua.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to get the parameter from.
     *  @param parameterName   The name of the parameter to get the value for.
     *  @param parameterValue  A reference to where the value of the parameter will be stored.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool GetComponentLiveParameterBoolean(Handle handle, const Char* parameterName, bool& parameterValue) const override;

    /**
     *  Get the string (i.e. ScriptComponent::String) value of the parameter identified by its name and the
     *  given component from Lua.
     *  ATTENTION: The returned string is a copy, and owned by the caller. Thus, the caller is responsible for
     *  deleting it after use.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to get the parameter from.
     *  @param parameterName   The name of the parameter to get the value for.
     *  @param parameterValue  A reference to where the value of the parameter will be stored.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool GetComponentLiveParameterString(Handle handle, const Char* parameterName, Char*& parameterValue) const override;

    /**
     *  Set the Double (i.e. ScriptComponent::Number) value of the parameter identified by its name and the
     *  given component in Lua.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to set the parameter for.
     *  @param parameterName   The name of the parameter to set the value for.
     *  @param parameterValue  The value of the parameter to be set.
     *  @return  True, if the value of the parameter identified by the name could be set in the script system's
     *           virtual machine.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool SetComponentLiveParameterNumber(Handle handle, const Char* parameterName, Double parameterValue) const override;

    /**
     *  Set the int (i.e. ScriptComponent::Integer) value of the parameter identified by its name and the
     *  given component in Lua.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to set the parameter for.
     *  @param parameterName   The name of the parameter to set the value for.
     *  @param parameterValue  The value of the parameter to be set.
     *  @return  True, if the value of the parameter identified by the name could be set in the script system's
     *           virtual machine.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool SetComponentLiveParameterInteger(Handle handle, const Char* parameterName, Int64 parameterValue) const override;

    /**
     *  Set the bool (i.e. ScriptComponent::Boolean) value of the parameter identified by its name and the
     *  given component in Lua.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to set the parameter for.
     *  @param parameterName   The name of the parameter to set the value for.
     *  @param parameterValue  The value of the parameter to be set.
     *  @return  True, if the value of the parameter identified by the name could be set in the script system's
     *           virtual machine.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool SetComponentLiveParameterBoolean(Handle handle, const Char* parameterName, bool parameterValue) const override;

    /**
     *  Set the string (i.e. ScriptComponent::String) value of the parameter identified by its name and the
     *  given component in Lua.
     *  ATTENTION: The given string is not stored, but a copy of it is created and stored instead.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to set the parameter for.
     *  @param parameterName   The name of the parameter to set the value for.
     *  @param parameterValue  The value of the parameter to be set.
     *  @return  True, if the value of the parameter identified by the name could be set in the script system's
     *           virtual machine.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool SetComponentLiveParameterString(Handle handle, const Char* parameterName, const Char* parameterValue) const override;

    /**
     *  Call a method of a script component using the given parameters in Lua.
     *  @param handle      The handle of the component for which to call the method.
     *  @param methodName  The name of the method to call.
     *  @param in          The parameters that are passed to the method.
     *  @param out         The expected return values of the method.
     *  @return  True, if the method was successfully called using the 'in' parameters, and the return values were
     *           successfully stored in the 'out' parameters.
     *           False, if the method does not exist,
     *           or threw a runtime error,
     *           or the given 'in'/'out' parameters did not match the actual parameters/return values of the method in the script,
     *           or the script system is stopped.
     */
    virtual bool CallComponentMethod(Handle handle, const Char* methodName, const ScriptParameters& in, ScriptParameters& out) const override;

    /**
     *  Dispatch the given script event to all script event listeners of the script system.
     *  Script event listeners can be native code, or script instances in the script system's virtual machine.
     *  @param scriptEvent  The script event to be dispatched.
     *  @return  True, if the scriptEvent was dispatched.
     *           False, if the script system is stopped, or the scriptEvent's parameters are empty.
     */
    virtual bool DispatchScriptEvent(const ScriptEvent& scriptEvent) override;

    FEATSTD_RTTI_DECLARATION();

    /**
     *  Get the name of Candera in Lua
     *  @return  The name of Candera in Lua.
     */
    static const Char* GetLuaCanderaName() { return s_canderaName; }

    /**
     *  Get the Lua state.
     *  This is intended for derived systems so they can expose additional Lua features.
     *  @return  The Lua state
     */
    lua_State* GetLuaState() const { return m_luaState; }

protected:
    /**
     *  Constructor
     */
    LuaScriptSystem();

    /**
     *  FixedUpdate callback, which is called 1/FixedTimeStep times per second by the UpdateSystem.
     */
    virtual void FixedUpdate() override;

    /**
     *  Update callback, which is called once per frame by the UpdateSystem.
     *  @param applicationTime  The time in seconds that has passed since the start of the application.
     *  @param deltaTime        The time in seconds that has passed since the last frame.
     */
    virtual void UpdateWithSeconds(Double applicationTime, Double deltaTime) override;

    /**
     *  LateUpdate callback, which is called by the UpdateSystem after FixedUpdate and Update.
     */
    virtual void LateUpdate() override;

    /**
     *  Callback after a component was attached.
     *  @param handle  The handle of the component that was attached.
     *  @param entity  The entity the component was attached to.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    virtual bool OnAttachComponent(Handle handle, ScriptEntity* const entity) override;

    /**
     *  Callback after a component was detached.
     *  @param handle  The handle of the component that was detached.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    virtual bool OnDetachComponent(Handle handle) override;

    /**
     *  Callback from the base ScriptSystem when the priority of a component changed.
     *  @param handle  The handle of the component to set the priority for.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    virtual bool OnSetComponentPriority(Handle handle) override;

    /**
     *  Callback for derived classes to handle enabled flag changes of components.
     *  @param handle     The handle of the component to set the enabled flag for.
     *  @param isEnabled  Enable the Lua component (true), or disabled it (false).
     *  @return  True, if the operation was successful. False, otherwise.
     */
    virtual bool OnSetComponentEnabled(Handle handle, bool isEnabled) override;

    /**
     *  Callback for derived classes to handle enabled flag changes of components.
     *  @param handle     The handle of the component to set the priority for.
     *  @param isEnabled  A reference to return the enabled flag in.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    virtual bool OnGetComponentEnabled(Handle handle, bool& isEnabled) const override;

    /**
     *  Callback from the base ScriptSystem when the script of a component changed.
     *  @param handle  The handle of the component to set the script for.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    virtual bool OnSetComponentScript(Handle handle) override;

    /**
     *  Called by Update when it is being executed for the first time.
     */
    virtual void CallScriptInit() override;

    /**
     *  Get the ObjectReferenceSystem::Handle (i.e. ScriptComponent::ObjectReference) value of the parameter identified by
     *  its name and the given component from the script system's virtual machine.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle           The handle of the component to get the parameter from.
     *  @param parameterName    The name of the parameter to get the value for.
     *  @param parameterValue   A reference to where the value of the parameter will be stored.
     *  @param parameterTypeId  A reference to where the TypeId of the parameterValue will be stored.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool GetComponentLiveParameterObjectReferenceHandle(Handle handle, const Char* parameterName,
        Internal::ObjectReferenceSystem::Handle& parameterValue, TypeId& parameterTypeId) const;

    /**
     *  Set the ObjectReferenceSystem::Handle (i.e. ScriptComponent::ObjectReference) value of the parameter identified by
     *  its name and the given component in the script system's virtual machine.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle           The handle of the component to set the parameter for.
     *  @param parameterName    The name of the parameter to set the value for.
     *  @param parameterValue   The value of the parameter to be set.
     *  @param parameterTypeId  The TypeId of the parameterValue to be set.
     *  @return  True, if the value of the parameter identified by the name could be set in the script system's
     *           virtual machine.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool SetComponentLiveParameterObjectReferenceHandle(Handle handle, const Char* parameterName,
        Internal::ObjectReferenceSystem::Handle parameterValue, TypeId parameterTypeId) const;

private:
    bool SetScriptInVM(const Script* script, bool forceSet = false) const;
    bool AttachComponentInVM(const Handle handle) const;
    bool DetachComponentInVM(const Handle handle) const;

    void PrepareCanderaInternalFunctionCallOnStack(const Char* functionName) const;
    void CallCanderaInternalFunction(const Char* functionName) const;
    Int PrepareComponentLiveOnStack(Handle handle, const Char* name) const;
    Int PrepareComponentLiveParameterOnStack(Handle handle) const;

    lua_State* m_luaState;

    static LuaScriptSystem* s_this;
    static const Char* s_canderaName;
    static const Char* s_canderaInternalName;
    static const Char* s_canderaInternalObjectReferenceMetaTable;
    static const Char* s_canderaInternalObjectReferenceTypeMetaTable;
    static const Char* s_masterScript;

    static const luaL_Reg s_canderaInternalLibFunctions[];
    static Int OpenCanderaInternalLib(lua_State* luaState);

    static const luaL_Reg s_luaLibs[];
    static void OpenLuaLibs(lua_State* luaState);

    /**
     *  Registered Candera_Internal functions in Lua
     */

    // Synchronize the variables (i.e. parameters) of the script instance in the virtual machine with the parameters
    // and their values of the corresponding script component and vice versa.
    static Int SynchronizeScriptParameters(lua_State* luaState);

    // Used by Lua to query the (default) enabled flag of the script component.
    static Int IsComponentDefaultEnabled(lua_State* luaState);

    // Used by Lua for the Candera.Time.realtime value.
    static Int GetRealtime(lua_State* luaState);

    // Used by Lua for setting Candera.Time.fixedDeltaTime as UpdateSystem's fixed time step.
    static Int SetFixedTimeStep(lua_State* luaState);

    struct ObjectReference {
        TypeId m_typeId;
        Internal::ObjectReferenceSystem::Handle m_handle;
    };

    // Used by LuaScriptSystem and LuaBinding to create ObjectReference Lua userdata on the Lua stack for further use.
    static bool CreateLuaObjectReference(lua_State* luaState, TypeId typeId, Internal::ObjectReferenceSystem::Handle objectReference);

    // Used by LuaScriptSystem and LuaBinding to get a Lua ObjectReference userdatum from the Lua stack.
    static ObjectReference* GetLuaObjectReference(lua_State* luaState, Int stackIndex, bool isMandatory = false);

    // Used by LuaScriptSystem and LuaBinding to get a CanderaObject pointer from a Lua ObjectReference userdatum from the Lua stack.
    static CanderaObject* GetPointerFromLuaObjectReference(lua_State* luaState);

    friend class Internal::LuaBinding;
    friend class EntityComponentSystem::EntitySystem;
};

/** @} */ // end of LuaScripting

} // namespace Scripting

} // namespace Candera

#endif
