//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "LuaScriptSystem.h"

#include <CanderaScripting/Lua/LuaBinding.h>
#include <CanderaScripting/Script.h>
#include <CanderaScripting/ScriptEvent.h>
#include <CanderaScripting/ScriptParameters.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/EntityComponentSystem/EntitySystem.h>
#include <Candera/System/UpdateSystem/UpdateSystem.h>

namespace Candera {

FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaScripting);

namespace Scripting {

FEATSTD_RTTI_DEFINITION(LuaScriptSystem, ScriptSystem)

LuaScriptSystem* LuaScriptSystem::s_this = 0;
const Char* LuaScriptSystem::s_canderaName = "Candera";
const Char* LuaScriptSystem::s_canderaInternalName = "Candera_Internal";
const Char* LuaScriptSystem::s_canderaInternalObjectReferenceMetaTable = "Candera_Internal.ObjectReferenceMetaTable";
const Char* LuaScriptSystem::s_canderaInternalObjectReferenceTypeMetaTable = "Candera_Internal.ObjectReferenceTypeMetaTable";


const Char* LuaScriptSystem::s_masterScript = {
    "-- Setup the global Candera table and functions\n"
    "Candera_Internal.Scripts = {}\n"
    "Candera_Internal.ScriptComponents = {}    -- linear table of script components sorted by script priority\n"
    "Candera_Internal.ScriptComponentIdToScriptComponent = {}    -- maps script component id to script component\n"
    "Candera_Internal.IsStopped = true\n"
    "Candera.IdToScriptComponent = {}    -- maps script component id to script component VTable for the user\n"
    "Candera.ReferenceType = {}\n" // stores ScriptComponent::ParameterType enums
    "Candera.Shader = {}" // stores Shader::UniformType enums
    "Candera.Time = {}\n"
    "local _time = { time = 0.0, deltaTime = 0.0, fixedTime = 0.0, fixedDeltaTime = 0.0 }\n"
    "local timeMetaTable = { \n"
    "    __index = function(t, k)\n"
    "                  if (k == 'realtime') then return Candera_Internal.GetRealtime() end\n"
    "                  return _time[k]\n"
    "              end,\n"
    "    __newindex = function(t, k, v)\n"
    "                     if (k == 'fixedDeltaTime') then\n"
    "                         Candera_Internal.SetFixedTimeStep(v)\n"
    "                         _time.fixedDeltaTime = v\n"
    "                     end\n"
    "                 end }\n"
    "setmetatable(Candera.Time, timeMetaTable)\n"
    "\n"
    "function Candera_Internal:SetTime(time, deltaTime, fixedTime, fixedDeltaTime, smoothDeltaTime)\n"
    "    _time.time = time\n"
    "    _time.deltaTime = deltaTime\n"
    "    _time.fixedTime = fixedTime\n"
    "    _time.fixedDeltaTime = fixedDeltaTime\n"
    "    _time.smoothDeltaTime = smoothDeltaTime\n"
    "end\n"
    "\n"
    "local function SetTableToReadOnly(t)\n"
    "    local proxy = {}\n"
    "    local mt = { __index = t, __newindex = function(t, k, v) Candera.LogError('Attempt to update read-only table at ' .. k) end }\n"
    "    setmetatable(proxy, mt)\n"
    "    return proxy\n"
    "end\n"
    "\n"
    "function Candera_Internal:SealEnumTables()\n"
    "    Candera.ReferenceType = SetTableToReadOnly(Candera.ReferenceType)\n"
    "    Candera.Shader = SetTableToReadOnly(Candera.Shader)\n"
    "end\n"
    "\n"
    "local function GetScriptComponent(id)\n"
    "    local scriptComponent = Candera_Internal.ScriptComponentIdToScriptComponent[id]\n"
    "    if not scriptComponent then\n"
    "        Candera.LogError(\"Component id '\" .. id .. \"' is not valid. No action was performed.\")\n"
    "        do return end\n"
    "    end\n"
    "    return scriptComponent\n"
    "end\n"
    "\n"
    "function Candera.SetEnabled(id, isEnabled)\n"
    "    local scriptComponent = GetScriptComponent(id)\n"
    "    if scriptComponent and scriptComponent.IsEnabled ~= isEnabled then"
    "        scriptComponent.IsEnabled = isEnabled\n"
    "        local vtable = scriptComponent.VTable\n"
    "        if vtable then\n"
    "            if not scriptComponent.IsInitialized then\n"
    "                scriptComponent.IsInitialized = true\n"
    "                if vtable.Init then vtable.Init(vtable, id) end\n"
    "            end\n"
    "            if isEnabled and vtable.OnEnable then vtable.OnEnable(vtable, id)\n"
    "            elseif not isEnabled and vtable.OnDisable then vtable.OnDisable(vtable, id) end\n"
    "        end\n"
    "    end\n"
    "end\n"
    "\n"
    "function Candera.IsEnabled(id)\n"
    "    local scriptComponent = GetScriptComponent(id)\n"
    "    return scriptComponent and scriptComponent.IsEnabled\n"
    "end\n"
    "\n"
    "-- Start function invoked by Candera\n"
    "function Candera_Internal:Start()\n"
    "    self.IsStopped = false\n"
    "    for index, scriptComponent in ipairs(self.ScriptComponents) do\n"
    "        local vtable = scriptComponent.VTable\n"
    "        local scriptComponentId = scriptComponent.Id\n"
    "        local awake = vtable and vtable.Awake\n"
    "        if awake then\n"
    "            awake(vtable, scriptComponentId)\n"
    "        end\n"
    "        local onEnable = vtable and vtable.OnEnable\n"
    "        if (onEnable and scriptComponent.IsEnabled) then\n"
    "            onEnable(vtable, scriptComponentId)\n"
    "        end\n"
    "    end\n"
    "end\n"
    "\n"
    "-- Stop function invoked by Candera\n"
    "function Candera_Internal:Stop()\n"
    "    for index, scriptComponent in ipairs(self.ScriptComponents) do\n"
    "        local vtable = scriptComponent.VTable\n"
    "        local scriptComponentId = scriptComponent.Id\n"
    "        local onDisable = vtable and vtable.OnDisable\n"
    "        if (onDisable and scriptComponent.IsEnabled) then\n"
    "            onDisable(vtable, scriptComponentId)\n"
    "        end\n"
    "        local onDestroy = vtable and vtable.OnDestroy\n"
    "        if onDestroy then\n"
    "            onDestroy(vtable, scriptComponentId)\n"
    "        end\n"
    "    end\n"
    "    self.IsStopped = true\n"
    "end\n"
    "\n"
    "-- Update function invoked by Candera\n"
    "function Candera_Internal:FixedUpdate()\n"
    "    -- Update callback\n"
    "    for index, scriptComponent in ipairs(self.ScriptComponents) do\n"
    "        local vtable = scriptComponent.VTable\n"
    "        local fixedUpdate = vtable and vtable.FixedUpdate\n"
    "        if (fixedUpdate and scriptComponent.IsEnabled) then\n"
    "            fixedUpdate(vtable, scriptComponent.Id)\n"
    "        end\n"
    "    end\n"
    "end\n"
    "\n"
    "-- Update function invoked by Candera\n"
    "function Candera_Internal:Update()\n"
    "    -- Update callback\n"
    "    for index, scriptComponent in ipairs(self.ScriptComponents) do\n"
    "        local vtable = scriptComponent.VTable\n"
    "        local update = vtable and vtable.Update\n"
    "        if (update and scriptComponent.IsEnabled) then\n"
    "            update(vtable, scriptComponent.Id)\n"
    "        end\n"
    "    end\n"
    "end\n"
    "\n"
    "-- LateUpdate function invoked by Candera\n"
    "function Candera_Internal:LateUpdate()\n"
    "    -- LateUpdate callback\n"
    "    for index, scriptComponent in ipairs(self.ScriptComponents) do\n"
    "        local vtable = scriptComponent.VTable\n"
    "        local lateUpdate = vtable and vtable.LateUpdate\n"
    "        if (lateUpdate and scriptComponent.IsEnabled) then\n"
    "            lateUpdate(vtable, scriptComponent.Id)\n"
    "        end\n"
    "    end\n"
    "end\n"
    "\n"
    "-- Init function invoked by Candera after all scripts have been added\n"
    "function Candera_Internal:Init()\n"
    "    for index, scriptComponent in ipairs(self.ScriptComponents) do\n"
    "        local vtable = scriptComponent.VTable\n"
    "        local init = vtable and vtable.Init\n"
    "        if init then\n"
    "            scriptComponent.IsInitialized = true\n"
    "            init(vtable, scriptComponent.Id)\n"
    "        end\n"
    "    end\n"
    "end\n"
    "\n"
    "-- Add a script to lua\n"
    "function Candera_Internal:SetScript(id, code)\n"
    "    if (self.Scripts[id] ~= nil) then\n"
    "        self.Scripts[id].Script = nil\n"
    "    end\n"
    "\n"
    "    if (code == nil) then\n"
    "        Candera.LogError(\"No code was given for script '\" .. id .. \"'.\")\n"
    "        return false\n"
    "    end\n"
    "\n"
    "    local script, errormessage = load(code)\n"
    "    if (script == nil) then\n"
    "        return false, errormessage\n"
    "    end\n"
    "\n"
    "    if (self.Scripts[id] == nil) then\n"
    "        self.Scripts[id] = { Script = script, ComponentIds = {} }\n"
    "    else\n"
    "        self.Scripts[id].Script = script\n"
    "        local scriptComponentIds = self.Scripts[id].ComponentIds\n"
    "        for key, scriptComponentId in pairs(scriptComponentIds) do\n"
    "            local scriptComponent = self.ScriptComponentIdToScriptComponent[scriptComponentId]\n"
    "            local vtable = script and script()\n"
    "            scriptComponent.VTable = vtable\n"
    "            Candera.IdToScriptComponent[scriptComponentId] = vtable\n"
    "            if vtable then self.SynchronizeScriptParameters(scriptComponentId, vtable) end\n"
    "        end\n"
    "    end\n"
    "\n"
    "    return true\n"
    "end\n"
    "\n"
    "local function SortScriptComponents(scriptComponents)\n"
    "    table.sort(scriptComponents, function(a, b) return (a.Priority > b.Priority) end)\n"
    "end\n"
    "\n"
    "-- Set the priority of a script component\n"
    "function Candera_Internal:SetScriptComponentPriority(id, priority)\n"
    "    local scriptComponentTable = self.ScriptComponentIdToScriptComponent[id]\n"
    "    scriptComponentTable.Priority = priority\n"
    "    SortScriptComponents(self.ScriptComponents)\n"
    "end\n"
    "\n"
    "-- Add a script component to lua\n"
    "function Candera_Internal:AddScriptComponent(id, scriptId, priority, isEnabled)\n"
    "    if (self.ScriptComponentIdToScriptComponent[id] ~= nil) then\n"
    "        Candera.LogError(\"(Internal Error) Script component with id '\" .. id .. \"' already exists.\")\n"
    "        return\n"
    "    end\n"
    "\n"
    "    local scriptInfo = self.Scripts[scriptId]\n"
    "    local scriptComponentIds = scriptInfo and scriptInfo.ComponentIds\n"
    "    if (scriptComponentIds == nil) then\n"
    "        scriptComponentIds = {}\n"
    "        scriptInfo.ComponentIds = scriptComponentIds\n"
    "    end\n"
    "    scriptComponentIds[id] = id\n"
    "\n"
    "    local script = scriptInfo and scriptInfo.Script\n"
    "    local vtable = script and script()\n"
    "    local scriptComponent = { Id = id, IsEnabled = isEnabled, Priority = priority, VTable = vtable, ScriptId = scriptId, IsInitialized = false }\n"
    "    table.insert(self.ScriptComponents, scriptComponent)\n"
    "    self.ScriptComponentIdToScriptComponent[id] = scriptComponent\n"
    "    Candera.IdToScriptComponent[id] = vtable\n"
    "    SortScriptComponents(self.ScriptComponents)\n"
    "    if vtable then\n"
    "        self.SynchronizeScriptParameters(id, vtable)\n"
    "        if not self.IsStopped then\n"
    "            local awake = vtable.Awake\n"
    "            if awake then awake(vtable, id) end\n"
    "            if isEnabled then\n"
    "                local onEnable = vtable.OnEnable\n"
    "                if onEnable then onEnable(vtable, id) end\n"
    "                local init = vtable.Init\n"
    "                if init then\n"
    "                    scriptComponent.IsInitialized = true\n"
    "                    init(vtable, id)\n"
    "                end\n"
    "            end\n"
    "        end\n"
    "    end\n"
    "end\n"
    "\n"
    "-- Remove script component from lua, call script component's OnDestroy()\n"
    "function Candera_Internal:RemoveScriptComponent(scriptComponentId)\n"
    "    for index, scriptComponent in ipairs(self.ScriptComponents) do\n"
    "        if (scriptComponentId == scriptComponent.Id) then\n"
    "            if not self.IsStopped then\n"
    "                local vtable = scriptComponent.VTable\n"
    "                local onDisable = vtable and vtable.OnDisable\n"
    "                if onDisable then\n"
    "                    onDisable(vtable, scriptComponentId)\n"
    "                end\n"
    "                local onDestroy = vtable and vtable.OnDestroy\n"
    "                if onDestroy then\n"
    "                    onDestroy(vtable, scriptComponentId)\n"
    "                end\n"
    "            end\n"
    "\n"
    "            -- remove script if it is no longer used by any component\n"
    "            local scriptComponentIds = self.Scripts[scriptComponent.ScriptId].ComponentIds\n"
    "            scriptComponentIds[scriptComponentId] = nil\n"
    "            local next = next\n"
    "            if (next(scriptComponentIds)) == nil then\n"
    "                self.Scripts[scriptComponent.ScriptId] = nil\n"
    "            end\n"
    "\n"
    "            -- remove script component\n"
    "            self.ScriptComponentIdToScriptComponent[scriptComponentId] = nil\n"
    "            Candera.IdToScriptComponent[scriptComponentId] = nil\n"
    "            table.remove(self.ScriptComponents, index)\n"
    "\n"
    "            return\n"
    "        end\n"
    "    end\n"
    "end\n"
    "\n"
    "-- Reset the state of all script components\n"
    "function Candera_Internal:ResetScriptComponents()\n"
    "    for key, scriptComponent in ipairs(self.ScriptComponents) do\n"
    "        local script = self.Scripts[scriptComponent.ScriptId]\n"
    "        local scriptComponentId = scriptComponent.Id\n"
    "        scriptComponent.IsEnabled = self.IsComponentDefaultEnabled(scriptComponentId)\n"
    "        if (script ~= nil) then\n"
    "            local vtable = script.Script and script.Script()\n"
    "            scriptComponent.VTable = vtable\n"
    "            Candera.IdToScriptComponent[scriptComponentId] = vtable\n"
    "            if vtable then self.SynchronizeScriptParameters(scriptComponentId, vtable) end\n"
    "        end\n"
    "    end\n"
    "end\n"
};

LuaScriptSystem::LuaScriptSystem()
    :
    m_luaState(0)
{
    // Since you can create one and only one LuaScriptSystem via EntitySystem::Create<LuaScriptSystem>(), this static variable is exclusively used by 
    // static functions called from Lua that want to access the system.
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1938, "accesses global data [MISRA C++ Rule 12-8-1] because Lua calls functions from this singleton class")
    s_this = this;
}

LuaScriptSystem::~LuaScriptSystem()
{
    if (0 != m_luaState) {
        LuaScriptSystem::ShutDown();
    }

    s_this = 0;
}

bool LuaScriptSystem::Init()
{
    FEATSTD_DEBUG_ASSERT(0 == m_luaState);
    if (0 != m_luaState) {
        // Init has already been called.

        // Note: If you really want to recreate the VM (chances are you don't), you need to call ShutDown() before calling Init().
        return false;
    }

    if (!Base::Init()) {
        return false;
    }

    m_luaState = luaL_newstate();

    OpenLuaLibs(m_luaState);

    // Open Candera_Internal Lua library
    luaL_requiref(m_luaState, s_canderaInternalName, OpenCanderaInternalLib, 1);
    lua_pop(m_luaState, 1);

    // Open Candera Lua library
    luaL_requiref(m_luaState, s_canderaName, Internal::LuaBinding::OpenCanderaLib, 1);
    lua_pop(m_luaState, 1);

    // Load master script
    Int status = luaL_loadstring(m_luaState, s_masterScript);
    if (0 == status) {
        status = lua_pcall(m_luaState, 0, 0, 0);
    }

    if (0 != status) {
        FEATSTD_LOG_ERROR("LuaScriptSystem::Init failed: %s", lua_tostring(m_luaState, -1));
        lua_pop(m_luaState, 1);
        return false;
    }

    Internal::LuaBinding::Init(m_luaState);
    CallCanderaInternalFunction("SealEnumTables"); // set enum tables to read-only in Lua.

    bool success = true;
    for (SizeType i = 0; i < m_components.Size(); ++i) {
        const ScriptComponent* component = m_components[i];
        FEATSTD_DEBUG_ASSERT(0 != component);
        ScriptEntity* entity = component->GetEntity();
        if (0 != entity) {
            success = OnAttachComponent(GetHandle(component), entity) && success;
        }
    }

    return success;
}

void LuaScriptSystem::Start()
{
    if (IsEnabled()) {
        return;
    }

    if (IsPaused()) {
        Pause();
        return;
    }

    Base::Start();
    CallCanderaInternalFunction("Start");
}

void LuaScriptSystem::Stop()
{
    CallCanderaInternalFunction("Stop");
    Base::Stop();
}

void LuaScriptSystem::ShutDown()
{
    if (0 != m_luaState) {
        lua_close(m_luaState);
        m_luaState = 0;
    }

    Base::ShutDown();
}

void LuaScriptSystem::Reset()
{
    CallCanderaInternalFunction("ResetScriptComponents");
    Base::Reset();
}

void LuaScriptSystem::FixedUpdate()
{
    if (IsEnabled()) {
        Base::FixedUpdate();

        UpdateSystem* updateSystem = EntityComponentSystem::EntitySystem::Get<UpdateSystem>();
        if (0 != updateSystem) {
            PrepareCanderaInternalFunctionCallOnStack("SetTime");
            lua_pushnumber(m_luaState, static_cast<lua_Number>(updateSystem->GetApplicationTime()));
            lua_pushnumber(m_luaState, static_cast<lua_Number>(updateSystem->GetDeltaTime()));
            lua_pushnumber(m_luaState, static_cast<lua_Number>(updateSystem->GetFixedApplicationTime()));
            lua_pushnumber(m_luaState, static_cast<lua_Number>(updateSystem->GetFixedDeltaTime()));
            lua_pushnumber(m_luaState, static_cast<lua_Number>(updateSystem->GetSmoothedDeltaTime()));
            if (lua_pcall(m_luaState, 6, 0, 0) != 0) {
                FEATSTD_LOG_ERROR("%s::SetTime failed: %s", s_canderaInternalName, lua_tostring(m_luaState, -1));
                lua_pop(m_luaState, 1);
            }
        }

        CallCanderaInternalFunction("FixedUpdate");
    }
}

void LuaScriptSystem::UpdateWithSeconds(Double applicationTime, Double deltaTime)
{
    if (IsEnabled()) {
        Base::UpdateWithSeconds(applicationTime, deltaTime);
        CallCanderaInternalFunction("Update");
    }
}

void LuaScriptSystem::LateUpdate()
{
    if (IsEnabled()) {
        Base::LateUpdate();
        CallCanderaInternalFunction("LateUpdate");
    }
}

bool LuaScriptSystem::OnAttachComponent(Handle handle, ScriptEntity* const entity)
{
    FEATSTD_UNUSED(entity);
    const ScriptComponent* component = GetPointer(handle);
    if (0 == component) {
        return false;
    }

    Script::SharedPointer script = component->GetScript();
    if (script.PointsToNull()) {
        return true;  // It's OK if there is no script, user can add one (e.g. in Scene Composer) later
    }

    const Char* scriptString = script->GetScript();
    if (0 == scriptString) {
        return true;  // It's OK if there is no script code, user can add one (e.g. in Scene Composer) later
    }

    return AttachComponentInVM(handle);
}

bool LuaScriptSystem::OnDetachComponent(Handle handle)
{
    bool result = Base::OnDetachComponent(handle);
    return DetachComponentInVM(handle) && result;
}

bool LuaScriptSystem::OnScriptChanged(const Script* script)
{
    if (0 == m_luaState) {
        return false;
    }

    return SetScriptInVM(script, /* forceSet */ true);
}

ScriptComponent::ParameterType LuaScriptSystem::GetComponentLiveParameterType(Handle handle, const Char* parameterName) const
{
    Int stackElementsToPop = PrepareComponentLiveParameterOnStack(handle);
    if (0 > stackElementsToPop) {
        return ScriptComponent::None;
    }

    ScriptComponent::ParameterType parameterType = ScriptComponent::None;

    lua_State* luaState = m_luaState;

    Int luaType = lua_getfield(luaState, -1, parameterName);
    switch (luaType) {
    case LUA_TNUMBER:
        parameterType = (lua_isinteger(luaState, -1) != 0) ? ScriptComponent::Integer : ScriptComponent::Number;
        break;

    case LUA_TBOOLEAN:
        parameterType = ScriptComponent::Boolean;
        break;

    case LUA_TSTRING:
        parameterType = ScriptComponent::String;
        break;

    case LUA_TUSERDATA: {
            LuaScriptSystem::ObjectReference* objectReference = LuaScriptSystem::GetLuaObjectReference(luaState, -1);
            if (0 != objectReference) {
                parameterType = ScriptComponent::ObjectReference;
            }
        }
        break;

    default:
        break;
    }

    lua_pop(m_luaState, (stackElementsToPop + 1));
    return parameterType;
}

bool LuaScriptSystem::GetComponentLiveParameterNumber(Handle handle, const Char* parameterName, Double& parameterValue) const
{
    Int stackElementsToPop = PrepareComponentLiveParameterOnStack(handle);
    if (0 > stackElementsToPop) {
        return false;
    }

    bool success = false;
    Int luaType = lua_getfield(m_luaState, -1, parameterName);
    if (LUA_TNUMBER == luaType) {
        parameterValue = lua_tonumber(m_luaState, -1);
        success = true;
    }

    lua_pop(m_luaState, (stackElementsToPop + 1));
    return success;
}

bool LuaScriptSystem::GetComponentLiveParameterInteger(Handle handle, const Char* parameterName, Int64& parameterValue) const
{
    Int stackElementsToPop = PrepareComponentLiveParameterOnStack(handle);
    if (0 > stackElementsToPop) {
        return false;
    }

    bool success = false;
    Int luaType = lua_getfield(m_luaState, -1, parameterName);
    if (LUA_TNUMBER == luaType) {
        bool isInteger = (lua_isinteger(m_luaState, -1) != 0);
        if (isInteger) {
            parameterValue = lua_tointeger(m_luaState, -1);
            success = true;
        }
    }

    lua_pop(m_luaState, (stackElementsToPop + 1));
    return success;
}

bool LuaScriptSystem::GetComponentLiveParameterBoolean(Handle handle, const Char* parameterName, bool& parameterValue) const
{
    Int stackElementsToPop = PrepareComponentLiveParameterOnStack(handle);
    if (0 > stackElementsToPop) {
        return false;
    }

    bool success = false;
    Int luaType = lua_getfield(m_luaState, -1, parameterName);
    if (LUA_TBOOLEAN == luaType) {
        parameterValue = (lua_toboolean(m_luaState, -1) != 0);
        success = true;
    }

    lua_pop(m_luaState, (stackElementsToPop + 1));
    return success;
}

bool LuaScriptSystem::GetComponentLiveParameterString(Handle handle, const Char* parameterName, Char*& parameterValue) const
{
    Int stackElementsToPop = PrepareComponentLiveParameterOnStack(handle);
    if (0 > stackElementsToPop) {
        return false;
    }

    bool success = false;
    parameterValue = 0;
    Int luaType = lua_getfield(m_luaState, -1, parameterName);
    if (LUA_TSTRING == luaType) {
        const Char* luaString = lua_tostring(m_luaState, -1);
        parameterValue = FEATSTD_NEW_ARRAY(Char, StringPlatform::Length(luaString) + 1);
        if (0 != parameterValue) {
            StringPlatform::Copy(parameterValue, luaString);
            success = true;
        }
    }

    lua_pop(m_luaState, (stackElementsToPop + 1));
    return success;
}

bool LuaScriptSystem::GetComponentLiveParameterObjectReferenceHandle(Handle handle, const Char* parameterName,
    Internal::ObjectReferenceSystem::Handle& parameterValue, TypeId& parameterTypeId) const
{
    Int stackElementsToPop = PrepareComponentLiveParameterOnStack(handle);
    if (0 > stackElementsToPop) {
        return false;
    }

    bool success = false;
    lua_State* luaState = m_luaState;
    Int luaType = lua_getfield(luaState, -1, parameterName);
    if (LUA_TUSERDATA == luaType) {
        ObjectReference* referencePointer = GetLuaObjectReference(luaState, -1);
        if (0 != referencePointer) {
            parameterTypeId = referencePointer->m_typeId;
            parameterValue = referencePointer->m_handle;
            success = true;
        }
    }

    lua_pop(m_luaState, (stackElementsToPop + 1));
    return success;
}

bool LuaScriptSystem::SetComponentLiveParameterNumber(Handle handle, const Char* parameterName, Double parameterValue) const
{
    Int stackElementsToPop = PrepareComponentLiveParameterOnStack(handle);
    if (0 > stackElementsToPop) {
        return false;
    }

    bool success = false;
    Int luaType = lua_getfield(m_luaState, -1, parameterName);
    if (LUA_TNUMBER == luaType) {
        lua_pushnumber(m_luaState, parameterValue);
        lua_setfield(m_luaState, -3, parameterName);   // Set the variable in Lua.
        success = true;
    }

    lua_pop(m_luaState, (stackElementsToPop + 1));
    return success;
}

bool LuaScriptSystem::SetComponentLiveParameterInteger(Handle handle, const Char* parameterName, Int64 parameterValue) const
{
    Int stackElementsToPop = PrepareComponentLiveParameterOnStack(handle);
    if (0 > stackElementsToPop) {
        return false;
    }

    bool success = false;
    Int luaType = lua_getfield(m_luaState, -1, parameterName);
    if (LUA_TNUMBER == luaType) {
        lua_pushinteger(m_luaState, parameterValue);
        lua_setfield(m_luaState, -3, parameterName);   // Set the variable in Lua.
        success = true;
    }

    lua_pop(m_luaState, (stackElementsToPop + 1));
    return success;
}

bool LuaScriptSystem::SetComponentLiveParameterBoolean(Handle handle, const Char* parameterName, bool parameterValue) const
{
    Int stackElementsToPop = PrepareComponentLiveParameterOnStack(handle);
    if (0 > stackElementsToPop) {
        return false;
    }

    bool success = false;
    Int luaType = lua_getfield(m_luaState, -1, parameterName);
    if (LUA_TBOOLEAN == luaType) {
        lua_pushboolean(m_luaState, parameterValue ? 1 : 0);
        lua_setfield(m_luaState, -3, parameterName);   // Set the variable in Lua.
        success = true;
    }

    lua_pop(m_luaState, (stackElementsToPop + 1));
    return success;
}

bool LuaScriptSystem::SetComponentLiveParameterString(Handle handle, const Char* parameterName, const Char* parameterValue) const
{
    Int stackElementsToPop = PrepareComponentLiveParameterOnStack(handle);
    if (0 > stackElementsToPop) {
        return false;
    }

    bool success = false;
    Int luaType = lua_getfield(m_luaState, -1, parameterName);
    if (LUA_TSTRING == luaType) {
        static_cast<void>(lua_pushstring(m_luaState, parameterValue));
        lua_setfield(m_luaState, -3, parameterName);   // Set the variable in Lua.
        success = true;
    }

    lua_pop(m_luaState, (stackElementsToPop + 1));
    return success;
}

bool LuaScriptSystem::SetComponentLiveParameterObjectReferenceHandle(Handle handle, const Char* parameterName,
    Internal::ObjectReferenceSystem::Handle parameterValue, TypeId parameterTypeId) const
{
    Int stackElementsToPop = PrepareComponentLiveParameterOnStack(handle);
    if (0 > stackElementsToPop) {
        return false;
    }

    lua_State* luaState = m_luaState;
    bool success = false;
    Int luaType = lua_getfield(luaState, -1, parameterName);
    if (LUA_TUSERDATA == luaType) {
        ObjectReference* currentReferencePointer = GetLuaObjectReference(luaState, -1, false);
        if (0 != currentReferencePointer) {
            CanderaObject* object = 0;
            Internal::ObjectReferenceComponent* objectReferenceComponent = m_objectReferenceSystem->GetPointer(parameterValue);
            if (0 != objectReferenceComponent) {
                object = objectReferenceComponent->GetEntity();
            }

            if ((parameterValue.IsNullHandle() || (currentReferencePointer->m_typeId == parameterTypeId)) ||
                ((0 != object) && (object->IsTypeOf(currentReferencePointer->m_typeId)))) {
                if (CreateLuaObjectReference(luaState, currentReferencePointer->m_typeId, parameterValue)) {
                    lua_setfield(luaState, -3, parameterName);   // Set the variable in Lua.
                    success = true;
                }
            }
        }
    }

    lua_pop(m_luaState, (stackElementsToPop + 1));
    return success;
}

static void CallComponentMethodLogOutParameterError(const Char* methodName, const Char* luaType, const Char* expectedType, SizeType returnValueNumber)
{
    FEATSTD_LOG_ERROR("'%s' failed: lua type '%s' at return value index %d does not match ScriptParameter::%s type.", methodName, luaType, returnValueNumber, expectedType);
}

bool LuaScriptSystem::CallComponentMethod(Handle handle, const Char* methodName, const ScriptParameters& in, ScriptParameters& out) const
{
    if (IsStopped()) {
        return false;
    }

    Int stackElementsToPop = PrepareComponentLiveParameterOnStack(handle);
    if (0 > stackElementsToPop) {
        return false;
    }

    lua_State* luaState = m_luaState;
    bool success = false;
    Int argumentCount = 0;
    Int luaType = lua_getfield(luaState, -1, methodName);
    if (LUA_TFUNCTION == luaType) {
        lua_pushvalue(luaState, -2); // Push the table ('VTable' from the Lua script, called 'self' in lua callbacks).
        lua_pushinteger(luaState, static_cast<lua_Integer>(GetHandleRaw(handle))); // Push handle, called 'id' in lua callbacks.

        for (SizeType i = 0; i < in.Size(); ++i) {
            ScriptComponent::ParameterType parameterType = in.GetType(i);
            switch (parameterType) {
            case ScriptComponent::Number: {
                Double value = 0.0;
                if (in.Get(value, i)) {
                    lua_pushnumber(luaState, static_cast<lua_Number>(value));
                    ++argumentCount;
                }
                else {
                    // This cannot happen unless the ScriptParameter class was changed, and broken.
                    FEATSTD_DEBUG_FAIL();
                }
                break;
            }
            case ScriptComponent::Integer: {
                Int value = 0;
                if (in.Get(value, i)) {
                    lua_pushinteger(luaState, static_cast<lua_Integer>(value));
                    ++argumentCount;
                }
                else {
                    // This cannot happen unless the ScriptParameter class was changed, and broken.
                    FEATSTD_DEBUG_FAIL();
                }
                break;
            }
            case ScriptComponent::Boolean: {
                bool value = true;
                if (in.Get(value, i)) {
                    lua_pushboolean(luaState, value ? 1 : 0);
                    ++argumentCount;
                }
                else {
                    // This cannot happen unless the ScriptParameter class was changed, and broken.
                    FEATSTD_DEBUG_FAIL();
                }
                break;
            }
            case ScriptComponent::String:{
                const Char* value = 0;
                if (in.Get(value, i)) {
                    static_cast<void>(lua_pushstring(luaState, value));
                    ++argumentCount;
                }
                else {
                    // This cannot happen unless the ScriptParameter class was changed, and broken.
                    FEATSTD_DEBUG_FAIL();
                }
                break;
            }
            case ScriptComponent::ObjectReference: {
                Internal::ObjectReferenceSystem::Handle objectReference;
                if (GetObjectReference(in, objectReference, i)) {
                    CanderaObject* value = 0;
                    if (in.Get(value, i)) {
                        if (!CreateLuaObjectReference(luaState,
                            (0 == value) ? CanderaObject::GetTypeId() : value->GetDynamicTypeId(), objectReference)) {
                            lua_pushnil(luaState);
                        }

                        ++argumentCount;
                    }
                    else {
                        // This cannot happen unless the ScriptParameter class was changed, and broken.
                        FEATSTD_DEBUG_FAIL();
                    }
                }
                else {
                    // This cannot happen unless the ScriptParameter class was changed, and broken.
                    FEATSTD_DEBUG_FAIL();
                }
                break;
            }
            default:
                FEATSTD_LOG_ERROR("ScriptSystem::CallComponentMethod: ScriptParameters 'in' uses an unsupported type at index %d", i);
                break;
            }
        }

        const SizeType outParameterCounter = out.Size();
        if (lua_pcall(luaState, argumentCount+2, static_cast<Int>(outParameterCounter), 0) != 0) {
            FEATSTD_LOG_ERROR("'%s' failed: %s", methodName, lua_tostring(luaState, -1));
            lua_pop(luaState, 1);
        }
        else {
            --stackElementsToPop;
            success = true;

            for (SizeType i = 0; i < outParameterCounter; ++i) {
                ScriptComponent::ParameterType parameterType = out.GetType(i);
                Int idx = static_cast<Int>(i - outParameterCounter);

                switch (parameterType) {
                case ScriptComponent::Number: {
                    if (LUA_TNUMBER == lua_type(luaState, idx)) {
                        Double value = lua_tonumber(luaState, idx);
                        if (!out.Set(value, i)) {
                            success = false;
                            FEATSTD_DEBUG_FAIL();
                        }
                    }
                    else {
                        success = false;
                        CallComponentMethodLogOutParameterError(methodName, lua_tostring(luaState, idx), "Number", i);
                    }
                    break;
                }
                case ScriptComponent::Integer: {
                    Int returnedType = lua_type(luaState, idx);
                    if (LUA_TNUMBER == returnedType) {
                        Int64 value = lua_tointeger(luaState, idx);
                        if (!out.Set(static_cast<Int>(value), i)) {
                            success = false;
                            FEATSTD_DEBUG_FAIL();
                        }
                    }
                    else {
                        success = false;
                        CallComponentMethodLogOutParameterError(methodName, lua_tostring(luaState, idx), "Integer", i);
                    }
                    break;
                }
                case ScriptComponent::Boolean: {
                    if (LUA_TBOOLEAN == lua_type(luaState, idx)) {
                        Int flag = lua_toboolean(luaState, idx);
                        if (!out.Set((flag == 0) ? false : true, i)) {
                            success = false;
                            FEATSTD_DEBUG_FAIL();
                        }
                    }
                    else {
                        success = false;
                        CallComponentMethodLogOutParameterError(methodName, lua_tostring(luaState, idx), "Boolean", i);
                    }
                    break;
                }
                case ScriptComponent::String: {
                    if (LUA_TSTRING == lua_type(luaState, idx)) {
                        const Char* string = lua_tostring(luaState, idx);
                        if (!out.Set(string, i)) {
                            success = false;
                            FEATSTD_DEBUG_FAIL();
                        }
                    }
                    else {
                        success = false;
                        CallComponentMethodLogOutParameterError(methodName, lua_tostring(luaState, idx), "String", i);
                    }
                    break;
                }
                case ScriptComponent::ObjectReference: {
                    if (LUA_TUSERDATA == lua_type(luaState, idx)) {
                        ObjectReference* referencePointer = GetLuaObjectReference(luaState, -1);
                        if (0 != referencePointer) {
                            if (!SetObjectReference(out, referencePointer->m_handle, referencePointer->m_typeId, i)) {
                                success = false;
                                FEATSTD_DEBUG_FAIL();
                            }
                        }
                    }
                    else {
                        success = false;
                        CallComponentMethodLogOutParameterError(methodName, lua_tostring(luaState, idx), "ObjectReference", i);
                    }
                    break;
                }
                default:
                    FEATSTD_DEBUG_FAIL();
                }
            }
        }
    }
    else {
        FEATSTD_LOG_ERROR("Calling '%s' failed. It does not exist, or it is not a function.", methodName);
    }

    lua_pop(luaState, (stackElementsToPop + 1));
    return success;
}

bool LuaScriptSystem::DispatchScriptEvent(const ScriptEvent& scriptEvent)
{
    if (!Base::DispatchScriptEvent(scriptEvent)) {
        return false;
    }

    lua_State* luaState = m_luaState;

    if ((0 == luaState) || (scriptEvent.GetParameters().Size() == 0)) {
        return false;
    }

    for (SizeType i = 0; i < m_scriptEventListeners.Size(); ++i) {
        Handle handle = m_scriptEventListeners[i];
        const ScriptComponent* component = GetPointer(handle);
        if ((0 != component) && (0 != component->GetEntity())) {
            Int stackElementsToPop = PrepareComponentLiveParameterOnStack(handle);
            if (0 > stackElementsToPop) {
                continue;
            }

            Int luaType = lua_getfield(luaState, -1, "OnEvent");
            if (LUA_TFUNCTION == luaType) {
                lua_pushvalue(luaState, -2); // Push the table ('VTable' from the Lua script, called 'self' in lua callbacks).
                lua_pushinteger(luaState, static_cast<lua_Integer>(GetHandleRaw(handle))); // Push handle, called 'id' in lua callbacks.
                lua_newtable(luaState); // table to hold the ScriptParameters from the ScriptEvent

                const ScriptParameters& scriptParameters = scriptEvent.GetParameters();
                for (SizeType j = 0; j < scriptParameters.Size(); ++j) {
                    bool isValid = true;
                    Int luaIndex = static_cast<Int>(j + 1); // By convention, lua arrays start at index 1, not 0.
                    switch (scriptParameters.GetType(j)) {
                    case ScriptComponent::Integer: {
                        Int value = 0;
                        isValid = scriptParameters.Get(value, j);
                        lua_pushinteger(luaState, value);
                        lua_seti(luaState, -2, luaIndex);
                        break;
                    }
                    case ScriptComponent::Number: {
                        Double value = 0.0;
                        isValid = scriptParameters.Get(value, j);
                        lua_pushnumber(luaState, value);
                        lua_seti(luaState, -2, luaIndex);
                        break;
                    }
                    case ScriptComponent::Boolean: {
                        bool value = false;
                        isValid = scriptParameters.Get(value, j);
                        lua_pushboolean(luaState, value ? 1 : 0);
                        lua_seti(luaState, -2, luaIndex);
                        break;
                    }
                    case ScriptComponent::String: {
                        const Char* value = 0;
                        isValid = scriptParameters.Get(value, j);
                        static_cast<void>((lua_pushstring(luaState, value))); // cast to void, we have no use for Lua's internal copy.
                        lua_seti(luaState, -2, luaIndex);
                        break;
                    }
                    case ScriptComponent::ObjectReference: {
                        Internal::ObjectReferenceSystem::Handle objectReference;
                        if (GetObjectReference(scriptParameters, objectReference, j)) {
                            CanderaObject* value = 0;
                            if (scriptParameters.Get(value, j)) {
                                if (!CreateLuaObjectReference(luaState,
                                    (0 == value) ? CanderaObject::GetTypeId() : value->GetDynamicTypeId(), objectReference)) {
                                    lua_pushnil(luaState);
                                }
                                lua_seti(luaState, -2, luaIndex);
                            }
                        }
                        break;
                    }
                    default:
                        break;
                    }

                    FEATSTD_DEBUG_ASSERT(isValid);
                    FEATSTD_UNUSED(isValid);
                }

                if (lua_pcall(luaState, 3, 0, 0) != 0) {
                    FEATSTD_LOG_ERROR("'OnEvent' failed: %s", lua_tostring(luaState, -1));
                    lua_pop(luaState, 1);
                }
                else {
                    --stackElementsToPop;
                }
            }

            lua_pop(luaState, (stackElementsToPop + 1));
        }
    }

    return true;
}

bool LuaScriptSystem::OnSetComponentPriority(Handle handle)
{
    const ScriptComponent* component = GetPointer(handle);
    if ((0 == m_luaState) || (0 == component)) {
        return false;
    }

    // Only notify Lua if this component is attached.
    if (0 != component->GetEntity())  {
        PrepareCanderaInternalFunctionCallOnStack("SetScriptComponentPriority");

        lua_pushinteger(m_luaState, static_cast<lua_Integer>(GetHandleRaw(handle)));
        lua_pushnumber(m_luaState, static_cast<lua_Number>(component->GetPriority()));
        if (lua_pcall(m_luaState, 3, 0, 0) != 0) {
            FEATSTD_LOG_ERROR("%s:SetScriptComponentPriority failed: %s", s_canderaInternalName, lua_tostring(m_luaState, -1));
            lua_pop(m_luaState, 1);
            return false;
        }
    }

    return true;
}

bool LuaScriptSystem::OnSetComponentEnabled(Handle handle, bool isEnabled)
{
    if (0 == m_luaState) {
        return false;
    }

    Int luaType = lua_getglobal(m_luaState, s_canderaName);
    FEATSTD_DEBUG_ASSERT(LUA_TTABLE == luaType);
    FEATSTD_UNUSED(luaType);

    luaType = lua_getfield(m_luaState, -1, "SetEnabled");
    FEATSTD_DEBUG_ASSERT(LUA_TFUNCTION == luaType);
    FEATSTD_UNUSED(luaType);

    lua_rotate(m_luaState, -2, -1);
    lua_pop(m_luaState, 1);

    lua_pushinteger(m_luaState, static_cast<lua_Integer>(GetHandleRaw(handle)));
    lua_pushboolean(m_luaState, isEnabled ? 1 : 0);
    if (lua_pcall(m_luaState, 2, 0, 0) != 0) {
        FEATSTD_LOG_ERROR("%s:SetEnabled failed: %s", s_canderaInternalName, lua_tostring(m_luaState, -1));
        lua_pop(m_luaState, 1);
        return false;
    }

    return true;
}

bool LuaScriptSystem::OnGetComponentEnabled(Handle handle, bool& isEnabled) const
{
    if (0 == m_luaState) {
        return false;
    }

    Int stackElementsToPop = PrepareComponentLiveOnStack(handle, "IsEnabled");
    if (0 > stackElementsToPop) {
        return false;
    }

    FEATSTD_DEBUG_ASSERT((0 != lua_isboolean(m_luaState, -1)));
    isEnabled = (0 != lua_toboolean(m_luaState, -1));
    lua_pop(m_luaState, stackElementsToPop);
    return true;
}

bool LuaScriptSystem::OnSetComponentScript(Handle handle)
{
    const ScriptComponent* component = GetPointer(handle);
    if ((0 == m_luaState) || (0 == component)) {
        return false;
    }

    if (0 == component->GetEntity()) {
        return true; // Component is not attached, nothing to do.
    }

    if (!DetachComponentInVM(handle)) {
        return false;
    }

    return AttachComponentInVM(handle);
}

void LuaScriptSystem::CallScriptInit()
{
    Base::CallScriptInit();
    CallCanderaInternalFunction("Init");
}

bool LuaScriptSystem::SetScriptInVM(const Script* script, bool forceSet) const
{
    if (0 == m_luaState) {
        return false;
    }

    if (0 == script) {
        return true; // No script performs no action, but that's not an error.
    }

    if (!forceSet) {
        // If a script in Lua at Candera_Internal.Scripts[script->GetInstanceId()] already exists, return.
        Int luaType = lua_getglobal(m_luaState, s_canderaInternalName);
        FEATSTD_DEBUG_ASSERT(LUA_TTABLE == luaType); FEATSTD_UNUSED(luaType);

        luaType = lua_getfield(m_luaState, -1, "Scripts");
        FEATSTD_DEBUG_ASSERT(LUA_TTABLE == luaType); FEATSTD_UNUSED(luaType);

        luaType = lua_geti(m_luaState, -1, static_cast<lua_Integer>(script->GetInstanceId()));
        FEATSTD_DEBUG_ASSERT((LUA_TTABLE == luaType) || (LUA_TNIL == luaType)); FEATSTD_UNUSED(luaType);

        if (!lua_isnil(m_luaState, -1)) {
            lua_pop(m_luaState, 3);
            return true;
        }

        lua_pop(m_luaState, 3);
    }

    // Call Candera_Internal.SetScript(Candera_Internal, script->GetInstanceId(), script->GetScript()) in Lua.
    PrepareCanderaInternalFunctionCallOnStack("SetScript");

    lua_pushinteger(m_luaState, static_cast<lua_Integer>(script->GetInstanceId()));
    const Char* scriptCopy = lua_pushstring(m_luaState, script->GetScript());
    FEATSTD_DEBUG_ASSERT(0 == StringPlatform::CompareStrings(scriptCopy, script->GetScript())); FEATSTD_UNUSED(scriptCopy);

    if (lua_pcall(m_luaState, 3, 2, 0) != 0) {
        FEATSTD_LOG_ERROR("%s:SetScript failed: %s", s_canderaInternalName, lua_tostring(m_luaState, -1));
        lua_pop(m_luaState, 2);
        return false;
    }

    if (LUA_TBOOLEAN != lua_type(m_luaState, -2)) {
        return false;
    }

    Int hasCompilingSucceeded = lua_toboolean(m_luaState, -2);
    const Char* compileErrorMessage = lua_tostring(m_luaState, -1);
    if (0 != compileErrorMessage) {
        if (0 != m_lastCompileErrorMessage) {
            FEATSTD_DELETE_ARRAY(m_lastCompileErrorMessage);
            m_lastCompileErrorMessage = 0;
        }

        m_lastCompileErrorMessage = FEATSTD_NEW_ARRAY(Char, StringPlatform::Length(compileErrorMessage) + 1);
        if (0 == m_lastCompileErrorMessage) {
            lua_pop(m_luaState, 2);
            return false;
        }

        StringPlatform::Copy(m_lastCompileErrorMessage, compileErrorMessage);
    }

    lua_pop(m_luaState, 2);
    return (hasCompilingSucceeded != 0);
}

bool LuaScriptSystem::AttachComponentInVM(const Handle handle) const
{
    const ScriptComponent* component = GetPointer(handle);
    if ((0 == m_luaState) || (0 == component)) {
        return false;
    }

    Script::SharedPointer script = component->GetScript();
    if (script.PointsToNull()) {
        return true; // No script means there is nothing to attach in Lua.
    }

    if (!SetScriptInVM(script.GetPointerToSharedInstance())) {
        return false;
    }

    PrepareCanderaInternalFunctionCallOnStack("AddScriptComponent");

    lua_pushinteger(m_luaState, static_cast<lua_Integer>(GetHandleRaw(handle)));
    lua_pushinteger(m_luaState, static_cast<lua_Integer>(script->GetInstanceId()));
    lua_pushnumber(m_luaState, static_cast<lua_Number>(component->GetPriority()));
    lua_pushboolean(m_luaState, component->IsEnabled() ? 1 : 0);
    if (lua_pcall(m_luaState, 5, 0, 0) != 0) {
        FEATSTD_LOG_ERROR("Candera::AddScriptComponent failed: %s", lua_tostring(m_luaState, -1));
        lua_pop(m_luaState, 1);
        return false;
    }

    return true;
}

bool LuaScriptSystem::DetachComponentInVM(const Handle handle) const
{
    const ScriptComponent* component = GetPointer(handle);
    if ((0 == m_luaState) || (0 == component)) {
        return false;
    }

    PrepareCanderaInternalFunctionCallOnStack("RemoveScriptComponent");

    lua_pushinteger(m_luaState, static_cast<lua_Integer>(GetHandleRaw(handle)));
    if (lua_pcall(m_luaState, 2, 0, 0) != 0) {
        FEATSTD_LOG_ERROR("%s:RemoveScriptComponent failed: %s", s_canderaInternalName, lua_tostring(m_luaState, -1));
        lua_pop(m_luaState, 1);
        return false;
    }

    return true;
}

void LuaScriptSystem::PrepareCanderaInternalFunctionCallOnStack(const Char* functionName) const
{
    if (0 == m_luaState) {
        return;
    }

    Int luaType = lua_getglobal(m_luaState, s_canderaInternalName);
    FEATSTD_DEBUG_ASSERT(LUA_TTABLE == luaType);
    FEATSTD_UNUSED(luaType);

    luaType = lua_getfield(m_luaState, -1, functionName);
    FEATSTD_DEBUG_ASSERT(LUA_TFUNCTION == luaType);
    FEATSTD_UNUSED(luaType);

    lua_rotate(m_luaState, -2, -1);
}

void LuaScriptSystem::CallCanderaInternalFunction(const Char* functionName) const
{
    if (0 == m_luaState) {
        return;
    }

    PrepareCanderaInternalFunctionCallOnStack(functionName);

    if (lua_pcall(m_luaState, 1, 0, 0) != 0) {
        FEATSTD_LOG_ERROR("%s:%s failed: %s", s_canderaInternalName, functionName, lua_tostring(m_luaState, -1));
        lua_pop(m_luaState, 1);
    }
}

Int LuaScriptSystem::PrepareComponentLiveOnStack(Handle handle, const Char* name) const
{
    const ScriptComponent* component = GetPointer(handle);
    if ((0 == m_luaState) || (0 == component)) {
        return -1;
    }

    Int luaType = lua_getglobal(m_luaState, s_canderaInternalName);
    FEATSTD_DEBUG_ASSERT(LUA_TTABLE == luaType); FEATSTD_UNUSED(luaType);
    luaType = lua_getfield(m_luaState, -1, "ScriptComponentIdToScriptComponent");
    FEATSTD_DEBUG_ASSERT(LUA_TTABLE == luaType); FEATSTD_UNUSED(luaType);
    luaType = lua_geti(m_luaState, -1, static_cast<lua_Integer>(GetHandleRaw(handle)));
    FEATSTD_DEBUG_ASSERT((LUA_TTABLE == luaType) || (LUA_TNIL == luaType)); FEATSTD_UNUSED(luaType);
    if (lua_isnil(m_luaState, -1)) {
        lua_pop(m_luaState, 3);
        return -1;
    }

    luaType = lua_getfield(m_luaState, -1, name);
    FEATSTD_UNUSED(luaType);
    if (lua_isnil(m_luaState, -1)) {
        lua_pop(m_luaState, 4);
        return -1;
    }

    return 4;
}

Int LuaScriptSystem::PrepareComponentLiveParameterOnStack(Handle handle) const
{
    return PrepareComponentLiveOnStack(handle, "VTable");
}

Int LuaScriptSystem::SynchronizeScriptParameters(lua_State* luaState)
{
    using namespace Internal;
    if ((0 == s_this) || (0 == luaState)) {
        return 0;
    }

    UInt32 handleRaw = static_cast<UInt32>(luaL_checkinteger(luaState, 1));
    Handle handle = s_this->CastToHandle(handleRaw);

    // If the following asserts hit, then the master script has been changed in a way
    // that it does not fit the Lua script system anymore.
    FEATSTD_DEBUG_ASSERT(!lua_isnil(luaState, 2)); // vtable must not be nil
    FEATSTD_DEBUG_ASSERT(!handle.IsNullHandle()); // the component must be resolveable

    // Remove parameters from the component which do no longer exist in the Lua script.
    ComponentDefaultParameterIterator it(s_this, handle);
    const Char* componentParameterName = 0;
    ScriptComponent::ParameterType componentParameterType = ScriptComponent::None;
    while (it.Get(componentParameterName, componentParameterType))
    {
        bool foundParameter = false;

        lua_pushvalue(luaState, 2);
        lua_pushnil(luaState);
        while (0 != lua_next(luaState, -2))
        {
            lua_pushvalue(luaState, -2);
            const Char* parameterName = lua_tostring(luaState, -1);
            const Int luaType = lua_type(luaState, -2);
            const bool isInteger = (lua_isinteger(luaState, -2) != 0);
            const bool isReferenceType = (LUA_TUSERDATA == luaType) && (ScriptComponent::ObjectReference == componentParameterType);
            if (0 == StringPlatform::CompareStrings(parameterName, componentParameterName)) {
                if (((ScriptComponent::Number == componentParameterType) && (LUA_TNUMBER == luaType) && (!isInteger)) ||
                    ((ScriptComponent::Integer == componentParameterType) && (LUA_TNUMBER == luaType) && (isInteger)) ||
                    ((ScriptComponent::Boolean == componentParameterType) && (LUA_TBOOLEAN == luaType)) ||
                    ((ScriptComponent::String == componentParameterType) && (LUA_TSTRING == luaType)) ||
                    isReferenceType)
                {
                    if (isReferenceType) {
                        ObjectReference* referencePointer = GetLuaObjectReference(luaState, -2);
                        if (0 != referencePointer) {
                            foundParameter = true;
                            break;
                        }
                    }
                    else {
                        foundParameter = true;
                    }
                }
            }

            lua_pop(luaState, 2);
        }

        if (foundParameter) {
            it.Next();
        }
        else {
            bool success = s_this->RemoveComponentParameter(handle, componentParameterName);
            FEATSTD_DEBUG_ASSERT(success);
            FEATSTD_UNUSED(success);
        }

        lua_pop(luaState, 1);
    }

    // Add parameters from script that do not exist in the component yet.
    // Update parameters in script that already exist in the component.
    lua_pushvalue(luaState, 2); // Push the table ('VTable' from the Lua script).
    lua_pushnil(luaState);
    while (0 != lua_next(luaState, -2))
    {
        lua_pushvalue(luaState, -2);
        const Char* parameterName = lua_tostring(luaState, -1);
        const Int luaType = lua_type(luaState, -2);

        switch (luaType) {
        case LUA_TNUMBER: {
            bool isInteger = (lua_isinteger(luaState, -2) != 0);
            if (isInteger) {
                Int64 integer = 0;
                if (s_this->GetComponentDefaultParameterInteger(handle, parameterName, integer)) {
                    lua_pushvalue(luaState, 2);        // Push the table.
                    lua_pushinteger(luaState, integer);
                    lua_setfield(luaState, -2, parameterName);   // Set the variable in Lua.
                    lua_pop(luaState, 1);              // Pop the table.
                }
                else {
                    integer = lua_tointeger(luaState, -2);
                    bool success = s_this->AddComponentParameterInteger(handle, parameterName, integer);
                    FEATSTD_DEBUG_ASSERT(success);
                    FEATSTD_UNUSED(success);
                }
            }
            else {
                Double number = 0.0;
                if (s_this->GetComponentDefaultParameterNumber(handle, parameterName, number)) {
                    lua_pushvalue(luaState, 2);        // Push the table.
                    lua_pushnumber(luaState, number);
                    lua_setfield(luaState, -2, parameterName);   // Set the variable in Lua.
                    lua_pop(luaState, 1);              // Pop the table.
                }
                else {
                    number = lua_tonumber(luaState, -2);
                    bool success = s_this->AddComponentParameterNumber(handle, parameterName, number);
                    FEATSTD_DEBUG_ASSERT(success);
                    FEATSTD_UNUSED(success);
                }
            }
            break;
        }

        case LUA_TBOOLEAN: {
            bool boolean = false;
            if (s_this->GetComponentDefaultParameterBoolean(handle, parameterName, boolean)) {
                lua_pushvalue(luaState, 2);
                lua_pushboolean(luaState, boolean ? 1 : 0);
                lua_setfield(luaState, -2, parameterName);
                lua_pop(luaState, 1);
            }
            else {
                boolean = (lua_toboolean(luaState, -2) != 0);
                bool success = s_this->AddComponentParameterBoolean(handle, parameterName, boolean);
                FEATSTD_DEBUG_ASSERT(success);
                FEATSTD_UNUSED(success);
            }
            break;
        }

        case LUA_TSTRING: {
            const Char* string = 0;
            if (s_this->GetComponentDefaultParameterString(handle, parameterName, string)) {
                lua_pushvalue(luaState, 2);
                static_cast<void>(lua_pushstring(luaState, string)); // Returns pointer to internal copy, which we don't need.
                lua_setfield(luaState, -2, parameterName);
                lua_pop(luaState, 1);
            }
            else {
                string = lua_tostring(luaState, -2);
                bool success = s_this->AddComponentParameterString(handle, parameterName, string);
                FEATSTD_DEBUG_ASSERT(success);
                FEATSTD_UNUSED(success);
            }
            break;
        }

        case LUA_TUSERDATA: {
            ObjectReference* referencePointer = GetLuaObjectReference(luaState, -2);
            if (0 != referencePointer) {
                if (!s_this->GetComponentDefaultParameterObjectReference(handle, parameterName, referencePointer->m_handle,
                    referencePointer->m_typeId)) {
                    bool success = s_this->AddComponentParameterObjectReference(handle, parameterName, referencePointer->m_handle,
                        referencePointer->m_typeId);
                    FEATSTD_DEBUG_ASSERT(success);
                    FEATSTD_UNUSED(success);
                }
            }
            break;
        }

        default:
            // Lua types that we do not support/handle.
            break;
        }

        lua_pop(luaState, 2);
    }

    lua_pop(luaState, 1);

    return 0;
}

Int LuaScriptSystem::IsComponentDefaultEnabled(lua_State* luaState)
{
    if ((0 == s_this) || (0 == luaState)) {
        return 0;
    }

    UInt32 handleRaw = static_cast<UInt32>(luaL_checkinteger(luaState, 1));
    Handle handle = s_this->CastToHandle(handleRaw);
    const ScriptComponent* component = s_this->GetPointer(handle);
    if (0 == component) {
        // This function is only used by the masterscript, so this should never happen unless the masterscript
        // has been changed and does no longer fit the requirements of the LuaScriptSystem.
        FEATSTD_DEBUG_FAIL();
        return 0;
    }

    lua_pushboolean(luaState, (component->IsEnabled() ? 1 : 0));
    return 1;
}

Int LuaScriptSystem::GetRealtime(lua_State* luaState)
{
    if ((0 == s_this) || (0 == luaState)) {
        return 0;
    }

    UpdateSystem* updateSystem = EntityComponentSystem::EntitySystem::Get<UpdateSystem>();
    if (0 != updateSystem) {
        lua_pushnumber(luaState, updateSystem->GetRealtime());
        return 1;
    }

    return 0;
}

Int LuaScriptSystem::SetFixedTimeStep(lua_State* luaState)
{
    if ((0 == s_this) || (0 == luaState)) {
        return 0;
    }

    UpdateSystem* updateSystem = EntityComponentSystem::EntitySystem::Get<UpdateSystem>();
    if (0 != updateSystem) {
        Float fixedTimeStep = static_cast<Float>(luaL_checknumber(luaState, 1));
        updateSystem->SetFixedTimeStep(fixedTimeStep);
    }

    return 0;
}

bool LuaScriptSystem::CreateLuaObjectReference(lua_State* luaState, TypeId typeId, Internal::ObjectReferenceSystem::Handle objectReference)
{
    const SizeType sizeOfUserData = sizeof(ObjectReference);
    ObjectReference* newReferencePointer = FeatStd::Internal::PointerToPointer<ObjectReference*>(lua_newuserdata(luaState, sizeOfUserData));
    if (0 != newReferencePointer) {
        FEATSTD_DEBUG_ASSERT(0 != typeId);
        Int tableType = luaL_getmetatable(luaState, typeId);
        FEATSTD_DEBUG_ASSERT(LUA_TTABLE == tableType);
        FEATSTD_UNUSED(tableType);
        static_cast<void>(lua_setmetatable(luaState, -2));
        newReferencePointer->m_typeId = typeId;
        newReferencePointer->m_handle = objectReference;
        return true;
    }

    return false;
}

LuaScriptSystem::ObjectReference* LuaScriptSystem::GetLuaObjectReference(lua_State* luaState, Int stackIndex, bool isMandatory)
{
    Int stackTop = lua_gettop(luaState);
    if (stackIndex < 0) {
        stackIndex = stackTop + stackIndex + 1;
    }

    const Char* classMetaTableName = 0;
    if (0 != lua_getmetatable(luaState, stackIndex)) {
        if (lua_getfield(luaState, -1, "__name") == LUA_TSTRING) {
            classMetaTableName = lua_tostring(luaState, -1);
            if (0 == classMetaTableName) {
                lua_pop(luaState, lua_gettop(luaState) - stackTop);
                return 0;
            }
        }
        else {
            FEATSTD_DEBUG_FAIL(); // A metatable without a '__name' is not possible.
        }

        if (lua_getfield(luaState, -2, "__metatable") == LUA_TTABLE) {
            if (lua_getfield(luaState, -1, "__name") == LUA_TSTRING) {
                const Char* baseClassMetaTableName = lua_tostring(luaState, -1);
                if (0 != StringPlatform::CompareStrings(baseClassMetaTableName, s_canderaInternalObjectReferenceTypeMetaTable)) {
                    lua_pop(luaState, lua_gettop(luaState) - stackTop);
                    return 0;
                }
            }
        }
    }

    ObjectReference* objectReference = 0;
    if (0 != classMetaTableName) {
        if (isMandatory) {
            objectReference = FeatStd::Internal::PointerToPointer<ObjectReference*>(luaL_checkudata(luaState, stackIndex, classMetaTableName));
        }
        else {
            objectReference = FeatStd::Internal::PointerToPointer<ObjectReference*>(luaL_testudata(luaState, stackIndex, classMetaTableName));
        }
    }

    lua_pop(luaState, lua_gettop(luaState) - stackTop);
    return objectReference;
}

CanderaObject* LuaScriptSystem::GetPointerFromLuaObjectReference(lua_State* luaState)
{
    ObjectReference* objectReference = GetLuaObjectReference(luaState, 1, /* isMandatory = */ true);
    if (0 != objectReference) {
        CanderaObject* object = LuaScriptSystem::s_this->GetObjectPointer(objectReference->m_handle);
        if (0 != object) {
            if ((object->IsTypeOf(objectReference->m_typeId))) {
                return object;
            }
            else {
                FEATSTD_LOG_ERROR("Cannot resolve object reference of type '%s' to requested type '%s'.", object->GetDynamicTypeId(),
                    objectReference->m_typeId);
            }
        }
    }

    return 0;
}


const luaL_Reg LuaScriptSystem::s_canderaInternalLibFunctions[] = {
    { "SynchronizeScriptParameters", LuaScriptSystem::SynchronizeScriptParameters },
    { "IsComponentDefaultEnabled", LuaScriptSystem::IsComponentDefaultEnabled },
    { "GetRealtime", GetRealtime },
    { "SetFixedTimeStep", SetFixedTimeStep },
    { NULL, NULL } // sentinel
};

Int LuaScriptSystem::OpenCanderaInternalLib(lua_State* luaState) {
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1960, "Violates MISRA C++ 2008 Required Rule 5-18-1. Third party macro")
    luaL_newlib(luaState, s_canderaInternalLibFunctions);

    // Create the meta table that identifies native ObjectReference user data.
    Int success = static_cast<Int>(luaL_newmetatable(luaState, s_canderaInternalObjectReferenceMetaTable));
    FEATSTD_DEBUG_ASSERT(0 != success); // Assert that there is no meta table name collision.
    FEATSTD_UNUSED(success);
    lua_pop(luaState, 1);

    return 1;
}

const luaL_Reg LuaScriptSystem::s_luaLibs[] = {
    { "_G", luaopen_base },
    { LUA_LOADLIBNAME, luaopen_package },
    { LUA_COLIBNAME, luaopen_coroutine },
    { LUA_TABLIBNAME, luaopen_table },
//     { LUA_IOLIBNAME, luaopen_io },
//     { LUA_OSLIBNAME, luaopen_os },
    { LUA_STRLIBNAME, luaopen_string },
    { LUA_MATHLIBNAME, luaopen_math },
    { LUA_UTF8LIBNAME, luaopen_utf8 },
//    { LUA_DBLIBNAME, luaopen_debug },
#if defined(LUA_COMPAT_BITLIB)
    { LUA_BITLIBNAME, luaopen_bit32 },
#endif
    { NULL, NULL }
};

void LuaScriptSystem::OpenLuaLibs(lua_State* luaState)
{
    const luaL_Reg *lib;
    /* "require" functions from 's_luaLibs' and set results to global table */
    for (lib = s_luaLibs; (lib->func != 0); lib++) {
        luaL_requiref(luaState, lib->name, lib->func, 1);
        lua_pop(luaState, 1);  /* remove lib */
    }
}

} // namespace Scripting

} // namespace Candera
