//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_SCRIPTING_LUABINDING3D2D_H
#define CANDERA_SCRIPTING_LUABINDING3D2D_H

#include <CanderaScripting/LuaScriptSystem.h>
#include <CanderaScripting/Lua/LuaBinding.h>

namespace Candera {

namespace Scripting {

namespace Internal {

/** @addtogroup LuaScripting
 *  @{
 */

/**
 * @brief  LuaBinding3D2D exposes Candera functionality that exists in both 3D and 2D in Lua. See LuaBinding for further details.
 */

class LuaBinding3D2D : LuaBinding
{
public:
    static void AddFunctions(lua_State* luaState);

private:
    FEATSTD_MAKE_CLASS_STATIC(LuaBinding3D2D);

    static const luaL_Reg s_additionalFunctions[];

    FEATSTD_LINT_NEXT_EXPRESSION(1511, "Member does not hide non-virtual member, because they have different signatures.")
    template<typename T3D, typename T2D, typename F3D, typename F2D> static Int SetBool(lua_State* luaState, F3D f3d, F2D f2d);

    FEATSTD_LINT_NEXT_EXPRESSION(1511, "Member does not hide non-virtual member, because they have different signatures.")
    template<typename T3D, typename T2D, typename F3D, typename F2D> static Int GetBool(lua_State* luaState, F3D f3d, F2D f2d);

    FEATSTD_LINT_NEXT_EXPRESSION(1511, "Member does not hide non-virtual member, because they have different signatures.")
    template<typename T3D, typename T2D, typename F3D, typename F2D> static Int SetFloat(lua_State* luaState, F3D f3d, F2D f2d);

    FEATSTD_LINT_NEXT_EXPRESSION(1511, "Member does not hide non-virtual member, because they have different signatures.")
    template<typename T3D, typename T2D, typename F3D, typename F2D> static Int GetFloat(lua_State* luaState, F3D f3d, F2D f2d);

    FEATSTD_LINT_NEXT_EXPRESSION(1511, "Member does not hide non-virtual member, because they have different signatures.")
    template<typename T3D, typename T2D, typename F3D, typename F2D> static Int SetInt(lua_State* luaState, F3D f3d, F2D f2d);

    FEATSTD_LINT_NEXT_EXPRESSION(1511, "Member does not hide non-virtual member, because they have different signatures.")
    template<typename T3D, typename T2D, typename F3D, typename F2D> static Int GetInt(lua_State* luaState, F3D f3d, F2D f2d);

    FEATSTD_LINT_NEXT_EXPRESSION(1511, "Member does not hide non-virtual member, because they have different signatures.")
    template<typename T3D, typename T2D, typename F3D, typename F2D> static Int GetColor(lua_State* luaState, F3D f3d, F2D f2d);

    FEATSTD_LINT_NEXT_EXPRESSION(1511, "Member does not hide non-virtual member, because they have different signatures.")
    template<typename T3D, typename T2D, typename TT, typename F3D, typename F2D> static Int SetTuple4(lua_State* luaState, F3D f3d, F2D f2d);

    FEATSTD_LINT_NEXT_EXPRESSION(1511, "Member does not hide non-virtual member, because they have different signatures.")
    template<typename T3D, typename T2D, typename F3D, typename F2D> static Int GetRectangle(lua_State* luaState, F3D f3d, F2D f2d);

    template<
#ifdef CANDERA_3D_ENABLED
        typename F3D
#endif
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_ENABLED)
        ,
#endif
#ifdef CANDERA_2D_ENABLED
        typename F2D
#endif
    > static Int SetVector(lua_State* luaState, 
#ifdef CANDERA_3D_ENABLED
    F3D f3d
#endif
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_ENABLED)
    ,
#endif
#ifdef CANDERA_2D_ENABLED
    F2D f2d
#endif
    );

    template<
#ifdef CANDERA_3D_ENABLED
        typename F3D
#endif
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_ENABLED)
        ,
#endif
#ifdef CANDERA_2D_ENABLED
        typename F2D
#endif
    > static Int GetVector(lua_State* luaState,
#ifdef CANDERA_3D_ENABLED
    F3D f3d
#endif
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_ENABLED)
    ,
#endif
#ifdef CANDERA_2D_ENABLED
    F2D f2d
#endif
    );

    template<
#ifdef CANDERA_3D_ENABLED
        typename F3D
#endif
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_ENABLED)
        ,
#endif
#ifdef CANDERA_2D_ENABLED
        typename F2D
#endif
    > static Int SetRotation(lua_State* luaState,
#ifdef CANDERA_3D_ENABLED
    F3D f3d
#endif
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_ENABLED)
    ,
#endif
#ifdef CANDERA_2D_ENABLED
    F2D f2d
#endif
    );

    /** @addtogroup LuaBindingTransformable
     * @{
     *  <hr>\section CanderaLuaSetPosition SetPosition
     *  \if CGIDOC_ENABLE_3D
     *  3D:
     *  \code
     *  Candera.SetPosition(id, x, y, z)
     *  \endcode
     *  Sets the position defined by 'x','y','z', if the node of the script component identified by 'id' is a 3D node.
     *  \endif
     *  \if CGIDOC_ENABLE_2D
     *
     *  2D:
     *  \code
     *  Candera.SetPosition(id, x, y)
     *  \endcode
     *  Sets the position defined by 'x','y', if the node of the script component identified by 'id' is a 2D node.
     *  \endif
     ** @} */
    static Int SetPosition(lua_State* luaState);    // Transformable(2D)::SetPosition

    /** @addtogroup LuaBindingTransformable
     * @{
     *  <hr>\section CanderaLuaGetPosition GetPosition
     *  \if CGIDOC_ENABLE_3D
     *  3D:
     *  \code
     *  local x,y,z = Candera.GetPosition(id)
     *  \endcode
     *  Returns the position in 'x','y','z', if the node of the script component identified by 'id' is a 3D node.
     *  \endif
     *  \if CGIDOC_ENABLE_2D
     *
     *  2D:
     *  \code
     *  local x,y = Candera.GetPosition(id)
     *  \endcode
     *  Returns the position in 'x','y', if the node of the script component identified by 'id' is a 2D node.
     *  \endif
     ** @} */
    static Int GetPosition(lua_State* luaState);    // Transformable(2D)::GetPosition

    /** @addtogroup LuaBindingTransformable
     * @{
     *  <hr>\section CanderaLuaSetRotation SetRotation
     *  \if CGIDOC_ENABLE_3D
     *  3D:
     *  \code
     *  Candera.SetRotation(id, pitch, yaw, roll)
     *  \endcode
     *  Sets the rotation defined in degrees by 'pitch','yaw','roll', if the node of the script component identified by 'id' is a 3D node.
     *  \endif
     *  \if CGIDOC_ENABLE_2D
     *
     *  2D:
     *  \code
     *  Candera.SetRotation(id, rotation)
     *  \endcode
     *  Sets the position defined in degrees by 'rotation', if the node of the script component identified by 'id' is a 2D node.
     *  \endif
     ** @} */
    static Int SetRotation(lua_State* luaState);    // Transformable(2D)::SetRotation

    /** @addtogroup LuaBindingTransformable
     * @{
     *  <hr>\section CanderaLuaGetRotation GetRotation
     *  \if CGIDOC_ENABLE_3D
     *  3D:
     *  \code
     *  local pitch, yaw, roll = Candera.GetRotation(id)
     *  \endcode
     *  Returns the rotation in degrees in 'pitch','yaw','roll', if the node of the script component identified by 'id' is a 3D node.
     *  \endif
     *  \if CGIDOC_ENABLE_2D
     *
     *  2D:
     *  \code
     *  local rotation = Candera.GetRotation(id)
     *  \endcode
     *  Returns the rotation in degrees in 'rotation', if the node of the script component identified by 'id' is a 2D node.
     *  \endif
     ** @} */
    static Int GetRotation(lua_State* luaState);    // Transformable(2D)::GetRotation

    /** @addtogroup LuaBindingTransformable
     * @{
     *  <hr>\section CanderaLuaSetScale SetScale
     *  \if CGIDOC_ENABLE_3D
     *  3D:
     *  \code
     *  Candera.SetScale(id, x, y, z)
     *  \endcode
     *  Sets the scale defined by 'x','y','z', if the node of the script component identified by 'id' is a 3D node.
     *  \endif
     *  \if CGIDOC_ENABLE_2D
     *
     *  2D:
     *  \code
     *  Candera.SetScale(id, x, y)
     *  \endcode
     *  Sets the scale defined by 'x','y', if the node of the script component identified by 'id' is a 2D node.
     *  \endif
     ** @} */
    static Int SetScale(lua_State* luaState);       // Transformable(2D)::SetScale

    /** @addtogroup LuaBindingTransformable
     * @{
     *  <hr>\section CanderaLuaGetScale GetScale
     *  \if CGIDOC_ENABLE_3D
     *  3D:
     *  \code
     *  local x,y,z = Candera.GetScale(id)
     *  \endcode
     *  Returns the scale in 'x','y','z', if the node of the script component identified by 'id' is a 3D node.
     *  \endif
     *  \if CGIDOC_ENABLE_2D
     *
     *  2D:
     *  \code
     *  local x,y = Candera.GetScale(id)
     *  \endcode
     *  Returns the scale in 'x','y', if the node of the script component identified by 'id' is a 2D node.
     *  \endif
     ** @} */
    static Int GetScale(lua_State* luaState);       // Transformable(2D)::GetScale

    /** @addtogroup LuaBindingTransformable
     * @{
     *  <hr>\section CanderaLuaTranslate Translate
     *  \if CGIDOC_ENABLE_3D
     *  3D:
     *  \code
     *  Candera.Translate(id, x, y, z)
     *  \endcode
     *  Translates the position by 'x','y','z', if the node of the script component identified by 'id' is a 3D node.
     *  \endif
     *  \if CGIDOC_ENABLE_2D
     *
     *  2D:
     *  \code
     *  Candera.Translate(id, x, y)
     *  \endcode
     *  Translates the position by 'x','y', if the node of the script component identified by 'id' is a 2D node.
     *  \endif
     ** @} */
    static Int Translate(lua_State* luaState);      // Transformable(2D)::Translate

    /** @addtogroup LuaBindingTransformable
     * @{
     *  <hr>\section CanderaLuaRotate Rotate
     *  \if CGIDOC_ENABLE_3D
     *  3D:
     *  \code
     *  Candera.Rotate(id, pitch, yaw, roll)
     *  \endcode
     *  Rotates by 'pitch','yaw','roll' in degrees, if the node of the script component identified by 'id' is a 3D node.
     *  Pitch defines the rotation around the x-axis, yaw around the y-axis, and roll the around z-axis.
     *  \endif
     *  \if CGIDOC_ENABLE_2D
     *
     *  2D:
     *  \code
     *  Candera.Rotate(id, rotation)
     *  \endcode
     *  Translates the position by 'rotation' in degrees, if the node of the script component identified by 'id' is a 2D node.
     *  \endif
     ** @} */
    static Int Rotate(lua_State* luaState);         // Transformable(2D)::Rotate

    /** @addtogroup LuaBindingTransformable
     * @{
     *  <hr>\section CanderaLuaScale Scale
     *  \if CGIDOC_ENABLE_3D
     *  3D:
     *  \code
     *  Candera.Scale(id, x, y, z)
     *  \endcode
     *  Scales by 'x','y','z', if the node of the script component identified by 'id' is a 3D node.
     *  \endif
     *  \if CGIDOC_ENABLE_2D
     *
     *  2D:
     *  \code
     *  Candera.Scale(id, x, y)
     *  \endcode
     *  Scales by 'x','y', if the node of the script component identified by 'id' is a 2D node.
     *  \endif
     ** @} */
    static Int Scale(lua_State* luaState);          // Transformable(2D)::Scale


    /** @addtogroup LuaBindingNode
     * @{
     *  <hr>\section CanderaLuaGetWorldPosition GetWorldPosition
     *  \if CGIDOC_ENABLE_3D
     *  3D:
     *  \code
     *  local x,y,z = Candera.GetWorldPosition(id)
     *  \endcode
     *  Returns the world position in 'x','y','z', if the node of the script component identified by 'id' is a 3D node.
     *  \endif
     *  \if CGIDOC_ENABLE_2D
     *
     *  2D:
     *  \code
     *  local x,y = Candera.GetWorldPosition(id)
     *  \endcode
     *  Returns the world position in 'x','y', if the node of the script component identified by 'id' is a 2D node.
     *  \endif
     ** @} */
    static Int GetWorldPosition(lua_State* luaState);            // Node(2D)::GetWorldPosition

    /** @addtogroup LuaBindingNode
     * @{
     *  <hr>\section CanderaLuaSetRenderingEnabled SetRenderingEnabled
     *  \code
     *  Candera.SetRenderingEnabled(id, isRenderingEnabled)
     *  \endcode
     *  Defines the local setting if rendering shall be enabled (true) or not (false), for the node of the script component
     *  identified by 'id'. This boolean value influences the node's descendants. See function \ref CanderaLuaIsEffectiveRenderingEnabled 
     *  "IsEffectiveRenderingEnabled" for further details.
     ** @} */
    static Int SetRenderingEnabled(lua_State* luaState);         // Node(2D)::SetRenderingEnabled

    /** @addtogroup LuaBindingNode
     * @{
     *  <hr>\section CanderaLuaIsRenderingEnabled IsRenderingEnabled
     *  \code
     *  local isRenderingEnabled = Candera.IsRenderingEnabled(id)
     *  \endcode
     *  Retrieve the local setting if rendering is enabled (true) or disabled (nil), for the node of the script component
     *  identified by 'id'. To compute if effective rendering is enabled in dependency to the node's ancestors, see
     *  function \ref CanderaLuaIsEffectiveRenderingEnabled "IsEffectiveRenderingEnabled" for further details.
     ** @} */
    static Int IsRenderingEnabled(lua_State* luaState);          // Node(2D)::IsRenderingEnabled

    /** @addtogroup LuaBindingNode
     * @{
     *  <hr>\section CanderaLuaIsEffectiveRenderingEnabled IsEffectiveRenderingEnabled
     *  \code
     *  local isEffectiveRenderingEnabled = Candera.IsEffectiveRenderingEnabled(id)
     *  \endcode
     *  Retrieve whether rendering is enabled (true) or not (nil) in dependency to the ancestors settings of he node of the 
     *  script component identified by 'id'. Return value of true means that the node and all its ancestors have rendering set enabled.
     *  A return value of 'nil' means that either the node or any ancestor node has set rendering disabled.
     ** @} */
    static Int IsEffectiveRenderingEnabled(lua_State* luaState); // Node(2D)::IsEffectiveRenderingEnabled

    /** @addtogroup LuaBindingNode
     * @{
     *  <hr>\section CanderaLuaSetAlphaValue SetAlphaValue
     *  \code
     *  Candera.SetAlphaValue(id, alphaValue)
     *  \endcode
     *  Sets the local alpha value for the node of the script component identified by 'id'.
     *  The alpha value range is [0.0, 1.0], where 0.0 means fully transparent (invisible), 1.0 means fully opaque.
     *  See also \ref CanderaLuaGetEffectiveAlphaValue "GetEffectiveAlphaValue".
     ** @} */
    static Int SetAlphaValue(lua_State* luaState);               // Node(2D)::SetAlphaValue

    /** @addtogroup LuaBindingNode
     * @{
     *  <hr>\section CanderaLuaGetAlphaValue GetAlphaValue
     *  \code
     *  local alphaValue = Candera.GetAlphaValue(id)
     *  \endcode
     *  Returns the local alpha value for the node of the script component identified by 'id'.
     *  The alpha value range is [0.0, 1.0], where 0.0 means fully transparent (invisible), 1.0 means fully opaque.
     *  See also \ref CanderaLuaGetEffectiveAlphaValue "GetEffectiveAlphaValue".
     ** @} */
    static Int GetAlphaValue(lua_State* luaState);               // Node(2D)::GetAlphaValue

    /** @addtogroup LuaBindingNode
     * @{
     *  <hr>\section CanderaLuaGetEffectiveAlphaValue GetEffectiveAlphaValue
     *  \code
     *  local effectiveAlphaValue = Candera.GetEffectiveAlphaValue(id)
     *  \endcode
     *  Returns the product of the alpha values for the node of the script component identified by 'id' and all its ancestors.
     *  The alpha value range is [0.0, 1.0], where 0.0 means fully transparent (invisible), 1.0 means fully opaque.
     ** @} */
    static Int GetEffectiveAlphaValue(lua_State* luaState);      // Node(2D)::GetEffectiveAlphaValue

    /** @addtogroup LuaBindingNode
     * @{
     *  <hr>\section CanderaLuaSetRenderOrderRank SetRenderOrderRank
     *  \code
     *  Candera.SetRenderOrderRank(id, renderOrderRank)
     *  \endcode
     *  Set render order rank of the node of the script component identified by 'id'.
     *  This can be used to define an explicit render order for the node within the assigned render order bin.
     *  The rank specifies a comparison parameter that is evaluated, if the node is attached to a render order bin that is sorted by rank.
     *  Note: The node's rank is meaningless, if the node is attached to a render order bin that is not sorted
     *  by rank but by other parameters, like for instance the "opaque" and "transparent" render order bins per default.
     *  Default value for render order rank is 0.
     ** @} */
    static Int SetRenderOrderRank(lua_State* luaState);          // Node(2D)::SetRenderOrderRank

    /** @addtogroup LuaBindingNode
     * @{
     *  <hr>\section CanderaLuaGetRenderOrderRank GetRenderOrderRank
     *  \code
     *  local renderOrderRank = Candera.GetRenderOrderRank(id)
     *  \endcode
     *  Get the render order rank of the node of the script component identified by 'id'.
     *  The rank specifies a comparison parameter that is evaluated, if the node is attached to a render order bin that is sorted by rank.
     *  Note: The node's rank is meaningless, if the node is attached to a render order bin that is not sorted
     *  by rank but by other parameters, like for instance the "opaque" and "transparent" render order bins per default.
     *  Default value for render order rank is 0.
     ** @} */
    static Int GetRenderOrderRank(lua_State* luaState);          // Node(2D)::GetRenderOrderRank


    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaIsCamera IsCamera
     *  \code
     *  local isCamera = Candera.IsCamera(id)
     *  \endcode
     *  Returns 'true' if the node of the script component identified by 'id' is a camera, 'nil' otherwise.
     *  Note: \ref CanderaLuaIs3D "Is3D" and \ref CanderaLuaIs2D "Is2D" can be used in conjunction with IsCamera to determine
     *  if it is a 3D or 2D camera.
     ** @} */
    static Int IsCamera(lua_State* luaState);

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaSetCameraViewport SetCameraViewport
     *  \code
     *  Candera.SetCameraViewport(id, left, top, width, height)
     *  \endcode
     *  Set the viewport values of the camera of the script component identified by 'id'.
     *  
     *  \if CGIDOC_ENABLE_3D
     *  3D:
     *
     *  The viewport specifies a rectangular region within the camera's render target.
     *  The viewport settings transform the vertex positions from normalized device coordinates [-1..1] into
     *  normalized screen coordinates [0..1]. The upper left corner of the Render Target is addressed with left:0, top:0.
     *  The dimension of the Render Target is specified with width 1.0, height 1.0.
     *  The viewport's width and height must be greater than 0.
     *  A viewport that intersects the render target is allowed, for instance values like: left -1, top -1, width 2.0, height 2.0.
     *  If the viewport aspect ratio differs from the aspect ratio of the Camera's projection matrix, the final render result will
     *  appear stretched or compressed.
     *  The default value for viewport is: left 0, top 0, width 1, height 1.
     *  \endif
     *
     *  \if CGIDOC_ENABLE_2D
     *  2D:
     *
     *  It is possible to specify negative viewport dimensions, which define that the viewport dimensions from the
     *  render target. If additionally negative viewport coordinates are set, the viewport will not include the entire
     *  render target as the dimensions remain the same. If the render target should be included entirely, the dimensions
     *  need to be set manually.
     *
     *  The default value for viewport is: left 0, top 0, width -1, height -1. Negative values indicate that the entire
     *  render target is included.
     *  \endif
     ** @} */
    static Int SetCameraViewport(lua_State* luaState);          // Camera(2D)::SetViewport

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaGetCameraViewport GetCameraViewport
     *  \code
     *  local left, top, width, height = Candera.GetCameraViewport(id)
     *  \endcode
     *  Get the viewport values of the camera of the script component identified by 'id'.
     *  See \ref CanderaLuaSetCameraViewport "SetCameraViewport" for details.
     ** @} */
    static Int GetCameraViewport(lua_State* luaState);          // Camera(2D)::GetViewport

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaSetCameraScissorRectangle SetCameraScissorRectangle
     *  \code
     *  Candera.SetCameraScissorRectangle(id, left, top, width, height)
     *  \endcode
     *  Set the scissor rectangle values of the camera of the script component identified by 'id'.
     *  
     *  \if CGIDOC_ENABLE_3D
     *  3D:
     *
     *  The scissor rectangle specifies a rectangular region within the camera's render target in normalized screen coordinates [0..1].
     *  The upper left corner of the render target is addressed with left:0, top:0.
     *  The dimension of the render target is specified with width 1.0, height 1.0.
     *  The scissor rectangles width and height must be greater than 0.
     *  A scissor rectangle that intersects the render target is allowed, for instance values like: left -1, top -1, width 2.0, height 2.0.
     *  If scissoring is enabled (see \ref CanderaLuaIsCameraScissoringEnabled "IsScissoringEnabled") pixels outside of the intersection area of viewport and scissor rectangle are clipped.
     *  The entire area defined by scissor rectangle is cleared if scissoring is enabled, the viewport otherwise.
     *  The default value for scissor rectangle is: left 0, top 0, width 1, height 1.
     *  \endif
     *
     *  \if CGIDOC_ENABLE_2D
     *  2D:
     *
     *  The scissor rectangle specifies a rectangular region within the camera's render target.
     *  The scissor rectangles width and height must be greater than 0.
     *  A scissor rectangle that intersects the render target is allowed.
     *  If scissoring is enabled (see \ref CanderaLuaIsCameraScissoringEnabled "IsScissoringEnabled") pixels outside of the intersection area of viewport and scissor
     *  rectangle are clipped.
     *  \endif
     ** @} */
    static Int SetCameraScissorRectangle(lua_State* luaState);  // Camera(2D)::SetScissorRectangle

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaGetCameraScissorRectangle GetCameraScissorRectangle
     *  \code
     *  local left, top, width, height = Candera.GetCameraScissorRectangle(id)
     *  \endcode
     *  Get the scissor rectangle values of the camera of the script component identified by 'id'.
     *  See \ref CanderaLuaSetCameraScissorRectangle "SetCameraScissorRectangle" for details.
     ** @} */
    static Int GetCameraScissorRectangle(lua_State* luaState);  // Camera(2D)::GetScissorRectangle

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaSetCameraScissoringEnabled SetCameraScissoringEnabled
     *  \code
     *  Candera.SetCameraScissoringEnabled(id, isScissoringEnabled)
     *  \endcode
     *  Defines if scissoring is enabled (true) or not (false) for the camera  of the script component identified by 'id'.
     *  Per default, scissoring is disabled.
     *  If scissoring is enabled, the scissor rectangle (see \ref CanderaLuaSetCameraScissorRectangle "SetCameraScissorRectangle") is
     *  used by clearing and scissor test.
     ** @} */
    static Int SetCameraScissoringEnabled(lua_State* luaState); // Camera(2D)::SetScissoringEnabled

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaIsCameraScissoringEnabled IsCameraScissoringEnabled
     *  \code
     *  local isScissoringEnabled = Candera.IsCameraScissoringEnabled(id)
     *  \endcode
     *  Retrieves whether scissoring is enabled (true) or not (nil) for the camera of the script component identified by 'id'.     
     *  If scissoring is enabled, the scissor rectangle (see \ref CanderaLuaSetCameraScissorRectangle "SetCameraScissorRectangle") is
     *  used by clearing and scissor test.
     ** @} */
    static Int IsCameraScissoringEnabled(lua_State* luaState);  // Camera(2D)::IsScissoringEnabled

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaSetCameraSequenceNumber SetCameraSequenceNumber
     *  \code
     *  Candera.SetCameraSequenceNumber(id, sequenceNumber)
     *  \endcode
     *  Sets render sequence of the camera of the script component identified by 'id' in an ascending order.
     ** @} */
    static Int SetCameraSequenceNumber(lua_State* luaState);    // Camera(2D)::SetSequenceNumber

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaGetCameraSequenceNumber GetCameraSequenceNumber
     *  \code
     *  local sequenceNumber = Candera.GetCameraSequenceNumber(id)
     *  \endcode
     *  Retrieves render sequence of the camera of the script component identified by 'id'.
     ** @} */
    static Int GetCameraSequenceNumber(lua_State* luaState);    // Camera(2D)::GetSequenceNumber

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaSetCameraSwapEnabled SetCameraSwapEnabled
     *  \code
     *  Candera.SetCameraSwapEnabled(id, isSwapEnabled)
     *  \endcode
     *  Enables (true) or disables (false) a swap of the camera's render target of the script component identified by 'id'
     *  after rendering has been processed. Disabled per default.
     ** @} */
    static Int SetCameraSwapEnabled(lua_State* luaState);       // Camera(2D)::SetSwapEnabled

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaIsCameraSwapEnabled IsCameraSwapEnabled
     *  \code
     *  local isSwapEnabled = Candera.IsCameraSwapEnabled(id)
     *  \endcode
     *  Retrieves whether the camera's swap of the script component identified by 'id' is enabled (true) or not (nil).
     ** @} */
    static Int IsCameraSwapEnabled(lua_State* luaState);        // Camera(2D)::IsSwapEnabled

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaSetCameraColorClearEnabled SetCameraColorClearEnabled
     *  \code
     *  Candera.SetCameraColorClearEnabled(id, isColorClearEnabled)
     *  \endcode
     *  Enable (true) or disable (false) clearing on activation of the camera of the script component identified by 'id'.
     *  Enabled by default.
     ** @} */
    static Int SetCameraColorClearEnabled(lua_State* luaState); // Camera::GetClearMode().SetColorClearEnabled, Camera2D::SetColorClearEnabled                    

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaIsCameraColorClearEnabled IsCameraColorClearEnabled
     *  \code
     *  local isColorClearEnabled = Candera.IsCameraColorClearEnabled(id)
     *  \endcode
     *  Return if clearing in activation of the camera of the script component identified by 'id' is enabled (true) or disabled (nil).
     ** @} */
    static Int IsCameraColorClearEnabled(lua_State* luaState);  // Camera::GetClearMode().IsColorClearEnabled, Camera2D::IsColorClearEnabled

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaSetCameraClearColor SetCameraClearColor
     *  \code
     *  Candera.SetCameraClearColor(id, red, green, blue, alpha)
     *  \endcode
     *  Assigns the clear color to the camera of the script component identified by 'id'.
     ** @} */
    static Int SetCameraClearColor(lua_State* luaState);        // Camera::GetClearMode().SetClearColor, Camera2D::SetClearColor

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaGetCameraClearColor GetCameraClearColor
     *  \code
     *  local red, green, blue, alpha = Candera.GetCameraClearColor(id)
     *  \endcode
     *  Returns the clear color of the camera of the script component identified by 'id'.
     ** @} */
    static Int GetCameraClearColor(lua_State* luaState);        // Camera::GetClearMode().GetClearColor, Camera2D::GetClearColor
};

/** @} */ // end of LuaScripting

} // namespace Internal

} // namespace Scripting

} // namespace Candera

#endif

