//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "LuaBinding3D2D.h"

#if defined(CANDERA_3D_ENABLED)
#include <Candera/Engine3D/Core/Camera.h>
#endif

#if defined(CANDERA_2D_ENABLED)
#include <Candera/Engine2D/Core/Camera2D.h>
#endif

namespace Candera {

namespace Scripting {

namespace Internal {

const luaL_Reg LuaBinding3D2D::s_additionalFunctions[] = {
    // Transformable(2D)
    { "SetPosition", SetPosition },
    { "GetPosition", GetPosition },
    { "SetRotation", SetRotation },
    { "GetRotation", GetRotation },
    { "SetScale", SetScale },
    { "GetScale", GetScale },
    { "Translate", Translate },
    { "Rotate", Rotate },
    { "Scale", Scale },

    // Node(2D)
    { "GetWorldPosition", GetWorldPosition },
    { "SetRenderingEnabled", SetRenderingEnabled },
    { "IsRenderingEnabled", IsRenderingEnabled },
    { "IsEffectiveRenderingEnabled", IsEffectiveRenderingEnabled },
    { "SetAlphaValue", SetAlphaValue },
    { "GetAlphaValue", GetAlphaValue },
    { "GetEffectiveAlphaValue", GetEffectiveAlphaValue },
    { "SetRenderOrderRank", SetRenderOrderRank },
    { "GetRenderOrderRank", GetRenderOrderRank },

    // Camera(2D)
    { "IsCamera", IsCamera },
    { "SetCameraViewport", SetCameraViewport },
    { "GetCameraViewport", GetCameraViewport },
    { "SetCameraScissorRectangle", SetCameraScissorRectangle },
    { "GetCameraScissorRectangle", GetCameraScissorRectangle },
    { "SetCameraScissoringEnabled", SetCameraScissoringEnabled },
    { "IsCameraScissoringEnabled", IsCameraScissoringEnabled },
    { "SetCameraSequenceNumber", SetCameraSequenceNumber },
    { "GetCameraSequenceNumber", GetCameraSequenceNumber },
    { "SetCameraSwapEnabled", SetCameraSwapEnabled },
    { "IsCameraSwapEnabled", IsCameraSwapEnabled },
    { "SetCameraColorClearEnabled", SetCameraColorClearEnabled },
    { "IsCameraColorClearEnabled", IsCameraColorClearEnabled },
    { "SetCameraClearColor", SetCameraClearColor },
    { "GetCameraClearColor", GetCameraClearColor },

    { NULL, NULL } // sentinel
};

void LuaBinding3D2D::AddFunctions(lua_State* luaState)
{
    // expects the table to be on top of the stack to add the functions to
    luaL_setfuncs(luaState, s_additionalFunctions, 0);
}

// ----------------------------------------------------------------------------
// Template functors


#ifdef CANDERA_3D_ENABLED
#define CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_3D CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR
#define CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR_3D CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR
#else
#define CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_3D(entityType, dataType, functorName, functionName)
#define CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR_3D(entityType, dataType, functorName, functionName)
#endif

#ifdef CANDERA_2D_ENABLED
#define CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_2D CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR
#define CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR_2D CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR
#else
#define CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_2D(entityType, dataType, functorName, functionName)
#define CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR_2D(entityType, dataType, functorName, functionName)
#endif

#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_ENABLED)
#define CANDERA_LUASCRIPTSYSTEM_FUNCTION_3D_2D(luaFunctionName, functionName, functorName3D, functorName2D) \
Int LuaBinding3D2D::luaFunctionName(lua_State* luaState)                                                    \
{                                                                                                           \
    return functionName(luaState, functorName3D(), functorName2D());                                        \
}

#define CANDERA_LUASCRIPTSYSTEM_FUNCTION_T3D_T2D(entityType3D, entityType2D, luaFunctionName, functionName, functorName3D, functorName2D) \
Int LuaBinding3D2D::luaFunctionName(lua_State* luaState)                                                                                  \
{                                                                                                                                         \
    return functionName<entityType3D, entityType2D>(luaState, functorName3D(), functorName2D());                                          \
}
#elif defined(CANDERA_3D_ENABLED)
//only 3D enabled
#define CANDERA_LUASCRIPTSYSTEM_FUNCTION_3D_2D(luaFunctionName, functionName, functorName3D, functorName2D) \
Int LuaBinding3D2D::luaFunctionName(lua_State* luaState)                                                    \
{                                                                                                           \
    return functionName(luaState, functorName3D());                                                         \
}

#define CANDERA_LUASCRIPTSYSTEM_FUNCTION_T3D_T2D(entityType3D, entityType2D, luaFunctionName, functionName, functorName3D, functorName2D) \
Int LuaBinding3D2D::luaFunctionName(lua_State* luaState)                                                                                  \
{                                                                                                                                         \
    return LuaBinding::functionName<entityType3D>(luaState, functorName3D());                                                             \
}
#elif defined(CANDERA_2D_ENABLED)
//only 3D enabled
#define CANDERA_LUASCRIPTSYSTEM_FUNCTION_3D_2D(luaFunctionName, functionName, functorName3D, functorName2D) \
Int LuaBinding3D2D::luaFunctionName(lua_State* luaState)                                                    \
{                                                                                                           \
    return functionName(luaState, functorName2D());                                                         \
}

#define CANDERA_LUASCRIPTSYSTEM_FUNCTION_T3D_T2D(entityType3D, entityType2D, luaFunctionName, functionName, functorName3D, functorName2D) \
Int LuaBinding3D2D::luaFunctionName(lua_State* luaState)                                                                                  \
{                                                                                                                                         \
    return LuaBinding::functionName<entityType2D>(luaState, functorName2D());                                                             \
}
#endif

#define CANDERA_LUASCRIPTSYSTEM_IS_ENTITYTYPE_BOOL(entityType, functionName)                                                                          \
CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR(entityType, bool, FEATSTD_CONCAT3(FIs, entityType, functionName), FEATSTD_CONCAT2(Is, functionName))              \
CANDERA_LUASCRIPTSYSTEM_FUNCTION_T(LuaBinding3D2D, entityType, FEATSTD_CONCAT3(Is, entityType, functionName), GetBool, FEATSTD_CONCAT3(FIs, entityType, functionName))

template<typename T3D, typename T2D, typename TT, typename F3D, typename F2D>
Int LuaBinding3D2D::SetTuple4(lua_State* luaState, F3D f3d, F2D f2d)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Float value1 = static_cast<Float>(luaL_checknumber(luaState, 2));
    Float value2 = static_cast<Float>(luaL_checknumber(luaState, 3));
    Float value3 = static_cast<Float>(luaL_checknumber(luaState, 4));
    Float value4 = static_cast<Float>(luaL_checknumber(luaState, 5));

#ifdef CANDERA_3D_ENABLED
    if (entity->IsTypeOf(T3D::GetTypeId())) {
        T3D* t3d = static_cast<T3D*>(entity);
        f3d(t3d, TT(value1, value2, value3, value4));
        return 0;
    }
#endif // CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    if (entity->IsTypeOf(T2D::GetTypeId())) {
        T2D* t2d = static_cast<T2D*>(entity);
        f2d(t2d, TT(value1, value2, value3, value4));
    }
#endif // CANDERA_2D_ENABLED

    return 0;
}

template<typename T3D, typename T2D, typename F3D, typename F2D>
Int LuaBinding3D2D::SetBool(lua_State* luaState, F3D f3d, F2D f2d)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    bool boolean = (lua_toboolean(luaState, 2) != 0);

#ifdef CANDERA_3D_ENABLED
    if (entity->IsTypeOf(T3D::GetTypeId())) {
        T3D* t3d = static_cast<T3D*>(entity);
        f3d(t3d, boolean);
        return 0;
    }
#endif // CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    if (entity->IsTypeOf(T2D::GetTypeId())) {
        T2D* t2d = static_cast<T2D*>(entity);
        f2d(t2d, boolean);
    }
#endif // CANDERA_2D_ENABLED

    return 0;
}

template<typename T3D, typename T2D, typename F3D, typename F2D>
Int LuaBinding3D2D::GetBool(lua_State* luaState, F3D f3d, F2D f2d)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

#ifdef CANDERA_3D_ENABLED
    if (entity->IsTypeOf(T3D::GetTypeId())) {
        T3D* t3d = static_cast<T3D*>(entity);
        lua_pushboolean(luaState, f3d(t3d) ? 1 : 0);
        return 1;
    }
#endif // CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    if (entity->IsTypeOf(T2D::GetTypeId())) {
        T2D* t2d = static_cast<T2D*>(entity);
        lua_pushboolean(luaState, f2d(t2d) ? 1 : 0);
        return 1;
    }
#endif // CANDERA_2D_ENABLED

    return 0;
}


#define CANDERA_LUASCRIPTSYSTEM_SET_ENTITYTYPE3D_2D_BOOL(entityType3D, entityType2D, luaFunctionName, functionName)                                                        \
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_3D(entityType3D, bool, FEATSTD_CONCAT2(luaFunctionName, 3D), functionName)                                                             \
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_2D(entityType2D, bool, FEATSTD_CONCAT2(luaFunctionName, 2D), functionName)                                                             \
CANDERA_LUASCRIPTSYSTEM_FUNCTION_T3D_T2D(entityType3D, entityType2D, luaFunctionName, SetBool, FEATSTD_CONCAT2(luaFunctionName, 3D), FEATSTD_CONCAT2(luaFunctionName, 2D))

#define CANDERA_LUASCRIPTSYSTEM_IS_ENTITYTYPE3D_2D_BOOL(entityType3D, entityType2D, luaFunctionName, functionName)                                                         \
CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR_3D(entityType3D, bool, FEATSTD_CONCAT2(luaFunctionName, 3D), functionName)                                                             \
CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR_2D(entityType2D, bool, FEATSTD_CONCAT2(luaFunctionName, 2D), functionName)                                                             \
CANDERA_LUASCRIPTSYSTEM_FUNCTION_T3D_T2D(entityType3D, entityType2D, luaFunctionName, GetBool, FEATSTD_CONCAT2(luaFunctionName, 3D), FEATSTD_CONCAT2(luaFunctionName, 2D))

template<typename T3D, typename T2D, typename F3D, typename F2D>
Int LuaBinding3D2D::SetFloat(lua_State* luaState, F3D f3d, F2D f2d)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Float value = static_cast<Float>(luaL_checknumber(luaState, 2));

#ifdef CANDERA_3D_ENABLED
    if (entity->IsTypeOf(T3D::GetTypeId())) {
        T3D* t3d = static_cast<T3D*>(entity);
        f3d(t3d, value);
        return 0;
    }
#endif // CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    if (entity->IsTypeOf(T2D::GetTypeId())) {
        T2D* t2d = static_cast<T2D*>(entity);
        f2d(t2d, value);
    }
#endif // CANDERA_2D_ENABLED

    return 0;
}

template<typename T3D, typename T2D, typename F3D, typename F2D>
Int LuaBinding3D2D::GetFloat(lua_State* luaState, F3D f3d, F2D f2d)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

#ifdef CANDERA_3D_ENABLED
    if (entity->IsTypeOf(T3D::GetTypeId())) {
        T3D* t3d = static_cast<T3D*>(entity);
        Float value = f3d(t3d);
        lua_pushnumber(luaState, value);
        return 1;
    }
#endif // CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    if (entity->IsTypeOf(T2D::GetTypeId())) {
        T2D* t2d = static_cast<T2D*>(entity);
        Float value = f2d(t2d);
        lua_pushnumber(luaState, value);
        return 1;
    }
#endif // CANDERA_2D_ENABLED

    return 0;
}

#define CANDERA_LUASCRIPTSYSTEM_GET_ENTITYTYPE3D_2D_FLOAT(entityType3D, entityType2D, luaFunctionName, functionName)                                                        \
CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR_3D(entityType3D, Float, FEATSTD_CONCAT2(luaFunctionName, 3D), functionName)                                                             \
CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR_2D(entityType2D, Float, FEATSTD_CONCAT2(luaFunctionName, 2D), functionName)                                                             \
CANDERA_LUASCRIPTSYSTEM_FUNCTION_T3D_T2D(entityType3D, entityType2D, luaFunctionName, GetFloat, FEATSTD_CONCAT2(luaFunctionName, 3D), FEATSTD_CONCAT2(luaFunctionName, 2D))

template<typename T3D, typename T2D, typename F3D, typename F2D>
Int LuaBinding3D2D::SetInt(lua_State* luaState, F3D f3d, F2D f2d)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Int value = static_cast<Int>(luaL_checkinteger(luaState, 2));

#ifdef CANDERA_3D_ENABLED
    if (entity->IsTypeOf(T3D::GetTypeId())) {
        T3D* t3d = static_cast<T3D*>(entity);
        f3d(t3d, value);
        return 0;
    }
#endif // CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    if (entity->IsTypeOf(T2D::GetTypeId())) {
        T2D* t2d = static_cast<T2D*>(entity);
        f2d(t2d, value);
    }
#endif // CANDERA_2D_ENABLED

    return 0;
}

template<typename T3D, typename T2D, typename F3D, typename F2D>
Int LuaBinding3D2D::GetInt(lua_State* luaState, F3D f3d, F2D f2d)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

#ifdef CANDERA_3D_ENABLED
    if (entity->IsTypeOf(T3D::GetTypeId())) {
        T3D* t3d = static_cast<T3D*>(entity);
        Int value = f3d(t3d);
        lua_pushinteger(luaState, value);
        return 1;
    }
#endif // CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    if (entity->IsTypeOf(T2D::GetTypeId())) {
        T2D* t2d = static_cast<T2D*>(entity);
        Int value = f2d(t2d);
        lua_pushinteger(luaState, value);
        return 1;
    }
#endif // CANDERA_2D_ENABLED

    return 0;
}

#define CANDERA_LUASCRIPTSYSTEM_SET_ENTITYTYPE3D_2D_INT(entityType3D, entityType2D, luaFunctionName, functionName)                                                       \
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_3D(entityType3D, Int, FEATSTD_CONCAT2(luaFunctionName, 3D), functionName)                                                            \
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_2D(entityType2D, Int, FEATSTD_CONCAT2(luaFunctionName, 2D), functionName)                                                            \
CANDERA_LUASCRIPTSYSTEM_FUNCTION_T3D_T2D(entityType3D, entityType2D, luaFunctionName, SetInt, FEATSTD_CONCAT2(luaFunctionName, 3D), FEATSTD_CONCAT2(luaFunctionName, 2D))

#define CANDERA_LUASCRIPTSYSTEM_GET_ENTITYTYPE3D_2D_INT(entityType3D, entityType2D, luaFunctionName, functionName)                                                       \
CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR_3D(entityType3D, Int, FEATSTD_CONCAT2(luaFunctionName, 3D), functionName)                                                            \
CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR_2D(entityType2D, Int, FEATSTD_CONCAT2(luaFunctionName, 2D), functionName)                                                            \
CANDERA_LUASCRIPTSYSTEM_FUNCTION_T3D_T2D(entityType3D, entityType2D, luaFunctionName, GetInt, FEATSTD_CONCAT2(luaFunctionName, 3D), FEATSTD_CONCAT2(luaFunctionName, 2D))

template<typename T3D, typename T2D, typename F3D, typename F2D>
Int LuaBinding3D2D::GetColor(lua_State* luaState, F3D f3d, F2D f2d)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

#ifdef CANDERA_3D_ENABLED
    if (entity->IsTypeOf(T3D::GetTypeId())) {
        T3D* t3d = static_cast<T3D*>(entity);
        const Color& color = f3d(t3d);
        lua_pushnumber(luaState, color.GetRed());
        lua_pushnumber(luaState, color.GetGreen());
        lua_pushnumber(luaState, color.GetBlue());
        lua_pushnumber(luaState, color.GetAlpha());
        return 4;
    }
#endif // CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    if (entity->IsTypeOf(T2D::GetTypeId())) {
        T2D* t2d = static_cast<T2D*>(entity);
        const Color& color = f2d(t2d);
        lua_pushnumber(luaState, color.GetRed());
        lua_pushnumber(luaState, color.GetGreen());
        lua_pushnumber(luaState, color.GetBlue());
        lua_pushnumber(luaState, color.GetAlpha());
        return 4;
    }
#endif // CANDERA_2D_ENABLED

    return 0;
}

#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_ENABLED)
#define CANDERA_LUASCRIPTSYSTEM_SET_ENTITYTYPE3D_2D_TUPLE4(entityType3D, entityType2D, tupleType, luaFunctionName, functionName) \
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_3D(entityType3D, tupleType, FEATSTD_CONCAT2(luaFunctionName, 3D), functionName)              \
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_2D(entityType2D, tupleType, FEATSTD_CONCAT2(luaFunctionName, 2D), functionName)              \
                                                                                                                                 \
Int LuaBinding3D2D::luaFunctionName(lua_State* luaState)                                                                         \
{                                                                                                                                \
    return SetTuple4<entityType3D, entityType2D, tupleType>(luaState, FEATSTD_CONCAT2(luaFunctionName, 3D)(),                    \
        FEATSTD_CONCAT2(luaFunctionName, 2D)());                                                                                 \
}
#elif defined(CANDERA_3D_ENABLED)
//only 3D enabled
#define CANDERA_LUASCRIPTSYSTEM_SET_ENTITYTYPE3D_2D_TUPLE4(entityType3D, entityType2D, tupleType, luaFunctionName, functionName) \
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_3D(entityType3D, tupleType, FEATSTD_CONCAT2(luaFunctionName, 3D), functionName)              \
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_2D(entityType2D, tupleType, FEATSTD_CONCAT2(luaFunctionName, 2D), functionName)              \
                                                                                                                                 \
Int LuaBinding3D2D::luaFunctionName(lua_State* luaState)                                                                         \
{                                                                                                                                \
    return LuaBinding::SetTuple4<entityType3D, tupleType>(luaState, FEATSTD_CONCAT2(luaFunctionName, 3D)());                     \
}
#elif defined(CANDERA_2D_ENABLED)
//only 2D enabled
#define CANDERA_LUASCRIPTSYSTEM_SET_ENTITYTYPE3D_2D_TUPLE4(entityType3D, entityType2D, tupleType, luaFunctionName, functionName) \
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_3D(entityType3D, tupleType, FEATSTD_CONCAT2(luaFunctionName, 3D), functionName)              \
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_2D(entityType2D, tupleType, FEATSTD_CONCAT2(luaFunctionName, 2D), functionName)              \
                                                                                                                                 \
Int LuaBinding3D2D::luaFunctionName(lua_State* luaState)                                                                         \
{                                                                                                                                \
    return LuaBinding::SetTuple4<entityType2D, tupleType>(luaState, FEATSTD_CONCAT2(luaFunctionName, 2D)());                     \
}
#endif

template<typename T3D, typename T2D, typename F3D, typename F2D>
Int LuaBinding3D2D::GetRectangle(lua_State* luaState, F3D f3d, F2D f2d)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

#ifdef CANDERA_3D_ENABLED
    if (entity->IsTypeOf(T3D::GetTypeId())) {
        T3D* t3d = static_cast<T3D*>(entity);
        const Rectangle& value = f3d(t3d);
        lua_pushnumber(luaState, value.GetLeft());
        lua_pushnumber(luaState, value.GetTop());
        lua_pushnumber(luaState, value.GetWidth());
        lua_pushnumber(luaState, value.GetHeight());
        return 4;
    }
#endif // CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    if (entity->IsTypeOf(T2D::GetTypeId())) {
        T2D* t2d = static_cast<T2D*>(entity);
        const Rectangle& value = f2d(t2d);
        lua_pushnumber(luaState, value.GetLeft());
        lua_pushnumber(luaState, value.GetTop());
        lua_pushnumber(luaState, value.GetWidth());
        lua_pushnumber(luaState, value.GetHeight());
        return 4;
    }
#endif // CANDERA_2D_ENABLED

    return 0;
}

#define CANDERA_LUASCRIPTSYSTEM_GET_ENTITYTYPE3D_2D_RECTANGLE(entityType3D, entityType2D, luaFunctionName, functionName)                                                        \
CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR_3D(entityType3D, Rectangle, FEATSTD_CONCAT2(luaFunctionName, 3D), functionName)                                                             \
CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR_2D(entityType2D, Rectangle, FEATSTD_CONCAT2(luaFunctionName, 2D), functionName)                                                             \
CANDERA_LUASCRIPTSYSTEM_FUNCTION_T3D_T2D(entityType3D, entityType2D, luaFunctionName, GetRectangle, FEATSTD_CONCAT2(luaFunctionName, 3D), FEATSTD_CONCAT2(luaFunctionName, 2D))

// ----------------------------------------------------------------------------
// Transformable(2D)

template<
#ifdef CANDERA_3D_ENABLED
    typename F3D
#endif
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_ENABLED)
    ,
#endif
#ifdef CANDERA_2D_ENABLED
    typename F2D
#endif
>
Int LuaBinding3D2D::SetVector(lua_State* luaState,
#ifdef CANDERA_3D_ENABLED
F3D f3d
#endif
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_ENABLED)
,
#endif
#ifdef CANDERA_2D_ENABLED
F2D f2d
#endif
)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Float x = static_cast<Float>(luaL_checknumber(luaState, 2));
    Float y = static_cast<Float>(luaL_checknumber(luaState, 3));

#ifdef CANDERA_3D_ENABLED
    if (entity->IsTypeOf(Node::GetTypeId())) {
        Node* node = static_cast<Node*>(entity);
        Float z = static_cast<Float>(luaL_checknumber(luaState, 4));
        f3d(node, Vector3(x, y, z));
        return 0;
    }
#endif // CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    if (entity->IsTypeOf(Node2D::GetTypeId())) {
        Node2D* node2D = static_cast<Node2D*>(entity);
        f2d(node2D, Vector2(x, y));
    }
#endif // CANDERA_2D_ENABLED

    return 0;
}

template<
#ifdef CANDERA_3D_ENABLED
    typename F3D
#endif
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_ENABLED)
    ,
#endif
#ifdef CANDERA_2D_ENABLED
    typename F2D
#endif
>
Int LuaBinding3D2D::GetVector(lua_State* luaState,
#ifdef CANDERA_3D_ENABLED
F3D f3d
#endif
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_ENABLED)
,
#endif
#ifdef CANDERA_2D_ENABLED
F2D f2d
#endif
)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

#ifdef CANDERA_3D_ENABLED
    if (entity->IsTypeOf(Node::GetTypeId())) {
        Node* node = static_cast<Node*>(entity);
        const Vector3& vector = f3d(node);
        lua_pushnumber(luaState, vector.GetX());
        lua_pushnumber(luaState, vector.GetY());
        lua_pushnumber(luaState, vector.GetZ());
        return 3;
    }
#endif // CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    if (entity->IsTypeOf(Node2D::GetTypeId())) {
        Node2D* node2D = static_cast<Node2D*>(entity);
        const Vector2& vector = f2d(node2D);
        lua_pushnumber(luaState, vector.GetX());
        lua_pushnumber(luaState, vector.GetY());
        return 2;
    }
#endif // CANDERA_2D_ENABLED

    return 0;
}

template<
#ifdef CANDERA_3D_ENABLED
    typename F3D
#endif
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_ENABLED)
    ,
#endif
#ifdef CANDERA_2D_ENABLED
    typename F2D
#endif
>
Int LuaBinding3D2D::SetRotation(lua_State* luaState,
#ifdef CANDERA_3D_ENABLED
F3D f3d
#endif
#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_ENABLED)
,
#endif
#ifdef CANDERA_2D_ENABLED
F2D f2d
#endif
)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Float x = static_cast<Float>(luaL_checknumber(luaState, 2));

#ifdef CANDERA_3D_ENABLED
    if (entity->IsTypeOf(Node::GetTypeId())) {
        Node* node = static_cast<Node*>(entity);
        Float y = static_cast<Float>(luaL_checknumber(luaState, 3));
        Float z = static_cast<Float>(luaL_checknumber(luaState, 4));
        f3d(node, Vector3(x, y, z));
        return 0;
    }
#endif // CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    if (entity->IsTypeOf(Node2D::GetTypeId())) {
        Node2D* node2D = static_cast<Node2D*>(entity);
        f2d(node2D, x);
    }
#endif // CANDERA_2D_ENABLED

    return 0;
}

#define CANDERA_LUASCRIPTSYSTEM_TRANSFORMABLE_SET_VECTOR(luaFunctionName)                                                                      \
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_3D(Node, Vector3, FEATSTD_CONCAT2(luaFunctionName, 3D), luaFunctionName)                                   \
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_2D(Node2D, Vector2, FEATSTD_CONCAT2(luaFunctionName, 2D), luaFunctionName)                                 \
CANDERA_LUASCRIPTSYSTEM_FUNCTION_3D_2D(luaFunctionName, SetVector, FEATSTD_CONCAT2(luaFunctionName, 3D), FEATSTD_CONCAT2(luaFunctionName, 2D))

#define CANDERA_LUASCRIPTSYSTEM_TRANSFORMABLE_GET_VECTOR(luaFunctionName)                                                                      \
CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR_3D(Node, Vector3, FEATSTD_CONCAT2(luaFunctionName, 3D), luaFunctionName)                                   \
CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR_2D(Node2D, Vector2, FEATSTD_CONCAT2(luaFunctionName, 2D), luaFunctionName)                                 \
CANDERA_LUASCRIPTSYSTEM_FUNCTION_3D_2D(luaFunctionName, GetVector, FEATSTD_CONCAT2(luaFunctionName, 3D), FEATSTD_CONCAT2(luaFunctionName, 2D))

CANDERA_LUASCRIPTSYSTEM_TRANSFORMABLE_SET_VECTOR(SetPosition)
CANDERA_LUASCRIPTSYSTEM_TRANSFORMABLE_GET_VECTOR(GetPosition)
CANDERA_LUASCRIPTSYSTEM_TRANSFORMABLE_SET_VECTOR(SetScale)
CANDERA_LUASCRIPTSYSTEM_TRANSFORMABLE_GET_VECTOR(GetScale)
CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1058, Candera::Matrix3x2::Translate, False positive because macro does not generate code that calls Rotate from Candera::Matrix3x2.)
CANDERA_SUPPRESS_LINT_FOR_SYMBOL(64, Candera::Matrix3x2 = Candera::Node2D, False positive because macro does not generate code that includes type Candera::Matrix3x2.)
CANDERA_SUPPRESS_LINT_FOR_SYMBOL(64, float = struct, False positive no float gets mached with struct in code generated from macro.)
CANDERA_LUASCRIPTSYSTEM_TRANSFORMABLE_SET_VECTOR(Translate)
CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1058, Candera::Matrix3x2::Scale, False positive because macro does not generate code that calls Rotate from Candera::Matrix3x2.)
CANDERA_SUPPRESS_LINT_FOR_SYMBOL(64, Candera::Matrix3x2 = Candera::Node2D, False positive because macro does not generate code that includes type Candera::Matrix3x2.)
CANDERA_SUPPRESS_LINT_FOR_SYMBOL(64, float = struct, False positive no float gets mached with struct in code generated from macro.)
CANDERA_LUASCRIPTSYSTEM_TRANSFORMABLE_SET_VECTOR(Scale)

#define CANDERA_LUASCRIPTSYSTEM_TRANSFORMABLE_ROTATE_VECTOR(luaFunctionName)                                                                     \
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_3D(Node, Vector3, FEATSTD_CONCAT2(luaFunctionName, 3D), luaFunctionName)                                     \
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_2D(Node2D, Float, FEATSTD_CONCAT2(luaFunctionName, 2D), luaFunctionName)                                     \
CANDERA_LUASCRIPTSYSTEM_FUNCTION_3D_2D(luaFunctionName, SetRotation, FEATSTD_CONCAT2(luaFunctionName, 3D), FEATSTD_CONCAT2(luaFunctionName, 2D))

CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1058, Candera::Matrix3x2::SetRotation, False positive because macro does not generate code that calls Rotate from Candera::Matrix3x2.)
CANDERA_SUPPRESS_LINT_FOR_SYMBOL(64, Candera::Matrix3x2 = Candera::Node2D, False positive because macro does not generate code that includes type Candera::Matrix3x2.)
CANDERA_LUASCRIPTSYSTEM_TRANSFORMABLE_ROTATE_VECTOR(SetRotation)
CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1058, Candera::Matrix3x2::Rotate, False positive because macro does not generate code that calls Rotate from Candera::Matrix3x2.)
CANDERA_SUPPRESS_LINT_FOR_SYMBOL(64, Candera::Matrix3x2 = Candera::Node2D, False positive because macro does not generate code that includes type Candera::Matrix3x2.)
CANDERA_LUASCRIPTSYSTEM_TRANSFORMABLE_ROTATE_VECTOR(Rotate)

Int LuaBinding3D2D::GetRotation(lua_State* luaState)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

#ifdef CANDERA_3D_ENABLED
    if (entity->IsTypeOf(Node::GetTypeId())) {
        Node* node = static_cast<Node*>(entity);
        const Vector3& vector = node->GetRotation();
        lua_pushnumber(luaState, vector.GetX());
        lua_pushnumber(luaState, vector.GetY());
        lua_pushnumber(luaState, vector.GetZ());
        return 3;
    }
#endif // CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    if (entity->IsTypeOf(Node2D::GetTypeId())) {
        Node2D* node2D = static_cast<Node2D*>(entity);
        Float rotation = node2D->GetRotation();
        lua_pushnumber(luaState, rotation);
        return 1;
    }
#endif // CANDERA_2D_ENABLED

    return 0;
}


// ----------------------------------------------------------------------------
// Node(2D)

CANDERA_LUASCRIPTSYSTEM_TRANSFORMABLE_GET_VECTOR(GetWorldPosition)

CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_3D(Node, bool, FEATSTD_CONCAT2(SetRenderingEnabled, 3D), SetRenderingEnabled)
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_2D(Node2D, bool, FEATSTD_CONCAT2(SetRenderingEnabled, 2D), SetRenderingEnabled)
CANDERA_LUASCRIPTSYSTEM_FUNCTION_T3D_T2D(Node, Node2D, SetRenderingEnabled, SetBool, SetRenderingEnabled3D, SetRenderingEnabled2D)

CANDERA_LUASCRIPTSYSTEM_IS_ENTITYTYPE3D_2D_BOOL(Node, Node2D, IsRenderingEnabled, IsRenderingEnabled)
CANDERA_LUASCRIPTSYSTEM_IS_ENTITYTYPE3D_2D_BOOL(Node, Node2D, IsEffectiveRenderingEnabled, IsEffectiveRenderingEnabled)

CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_3D(Node, Float, FEATSTD_CONCAT2(SetAlphaValue, 3D), SetAlphaValue)
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_2D(Node2D, Float, FEATSTD_CONCAT2(SetAlphaValue, 2D), SetAlphaValue)
CANDERA_LUASCRIPTSYSTEM_FUNCTION_T3D_T2D(Node, Node2D, SetAlphaValue, SetFloat, SetAlphaValue3D, SetAlphaValue2D)

CANDERA_LUASCRIPTSYSTEM_GET_ENTITYTYPE3D_2D_FLOAT(Node, Node2D, GetAlphaValue, GetAlphaValue)
CANDERA_LUASCRIPTSYSTEM_GET_ENTITYTYPE3D_2D_FLOAT(Node, Node2D, GetEffectiveAlphaValue, GetEffectiveAlphaValue)
CANDERA_LUASCRIPTSYSTEM_SET_ENTITYTYPE3D_2D_INT(Node, Node2D, SetRenderOrderRank, SetRenderOrderRank)
CANDERA_LUASCRIPTSYSTEM_GET_ENTITYTYPE3D_2D_INT(Node, Node2D, GetRenderOrderRank, GetRenderOrderRank)

// ----------------------------------------------------------------------------
// Camera(2D)

struct IsCameraF {
    bool operator() (const ScriptEntity* entity) const {
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(944, "It is intended that one argument always evaluates to false for 2D/3D only builds.")
        return (
#ifdef CANDERA_3D_ENABLED
            (entity->IsTypeOf(Camera::GetTypeId()))
#else
            false
#endif
            ||
#ifdef CANDERA_2D_ENABLED
            (entity->IsTypeOf(Camera2D::GetTypeId()))
#else
            false
#endif
            );
    }
};

CANDERA_LUASCRIPTSYSTEM_FUNCTION(LuaBinding3D2D, IsCamera, IsF, IsCameraF)
CANDERA_LUASCRIPTSYSTEM_SET_ENTITYTYPE3D_2D_TUPLE4(Camera, Camera2D, Rectangle, SetCameraViewport, SetViewport)
CANDERA_LUASCRIPTSYSTEM_GET_ENTITYTYPE3D_2D_RECTANGLE(Camera, Camera2D, GetCameraViewport, GetViewport)
CANDERA_LUASCRIPTSYSTEM_SET_ENTITYTYPE3D_2D_TUPLE4(Camera, Camera2D, Rectangle, SetCameraScissorRectangle, SetScissorRectangle)
CANDERA_LUASCRIPTSYSTEM_GET_ENTITYTYPE3D_2D_RECTANGLE(Camera, Camera2D, GetCameraScissorRectangle, GetScissorRectangle)
CANDERA_LUASCRIPTSYSTEM_SET_ENTITYTYPE3D_2D_BOOL(Camera, Camera2D, SetCameraScissoringEnabled, SetScissoringEnabled)
CANDERA_LUASCRIPTSYSTEM_IS_ENTITYTYPE3D_2D_BOOL(Camera, Camera2D, IsCameraScissoringEnabled, IsScissoringEnabled)
CANDERA_LUASCRIPTSYSTEM_GET_ENTITYTYPE3D_2D_INT(Camera, Camera2D, GetCameraSequenceNumber, GetSequenceNumber)
CANDERA_LUASCRIPTSYSTEM_SET_ENTITYTYPE3D_2D_BOOL(Camera, Camera2D, SetCameraSwapEnabled, SetSwapEnabled)
CANDERA_LUASCRIPTSYSTEM_IS_ENTITYTYPE3D_2D_BOOL(Camera, Camera2D, IsCameraSwapEnabled, IsSwapEnabled)


Int LuaBinding3D2D::SetCameraSequenceNumber(lua_State* luaState)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Int value = static_cast<Int>(luaL_checkinteger(luaState, 2));

#ifdef CANDERA_3D_ENABLED
    if (entity->IsTypeOf(Camera::GetTypeId())) {
        Camera* camera = static_cast<Camera*>(entity);
        camera->SetSequenceNumber(value);
        return 0;
    }
#endif // CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    if (entity->IsTypeOf(Camera2D::GetTypeId())) {
        Camera2D* camera = static_cast<Camera2D*>(entity);
        camera->SetSequenceNumber(static_cast<Int16>(value));
    }
#endif // CANDERA_2D_ENABLED

    return 0;
}

#ifdef CANDERA_3D_ENABLED
struct SetCameraColorClearEnabled3D {
    void operator() (Camera* entity, bool enabled) const
    {
        ClearMode clearMode = entity->GetClearMode();
        clearMode.SetColorClearEnabled(enabled);
        entity->SetClearMode(clearMode);
    }
};

struct IsCameraColorClearEnabled3D {
    bool operator() (const Camera* entity) const { return entity->GetClearMode().IsColorClearEnabled(); }
};
#endif

CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_2D(Camera2D, bool, SetCameraColorClearEnabled2D, SetClearColorEnabled)
CANDERA_LUASCRIPTSYSTEM_FUNCTION_T3D_T2D(Camera, Camera2D, SetCameraColorClearEnabled, SetBool, SetCameraColorClearEnabled3D, SetCameraColorClearEnabled2D)

CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR_2D(Camera2D, bool, IsCameraColorClearEnabled2D, IsClearColorEnabled)
CANDERA_LUASCRIPTSYSTEM_FUNCTION_T3D_T2D(Camera, Camera2D, IsCameraColorClearEnabled, GetBool, IsCameraColorClearEnabled3D, IsCameraColorClearEnabled2D)

#ifdef CANDERA_3D_ENABLED
struct SetCameraClearColor3D {
    void operator() (Camera* entity, const Color& color) const
    {
        ClearMode clearMode = entity->GetClearMode();
        clearMode.SetClearColor(color);
        entity->SetClearMode(clearMode);
    }
};
#endif

CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR_2D(Camera2D, Color, SetCameraClearColor2D, SetClearColor)

#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_ENABLED)
Int LuaBinding3D2D::SetCameraClearColor(lua_State* luaState)
{
    return SetTuple4<Camera, Camera2D, Color>(luaState, SetCameraClearColor3D(), SetCameraClearColor2D());
}
#elif defined(CANDERA_3D_ENABLED)
//only 3D enabled
Int LuaBinding3D2D::SetCameraClearColor(lua_State* luaState)
{
    return SetColor<Camera>(luaState, SetCameraClearColor3D());
}
#elif defined(CANDERA_2D_ENABLED)
//only 2D enabled
Int LuaBinding3D2D::SetCameraClearColor(lua_State* luaState)
{
    return LuaBinding::SetTuple4<Camera2D, Color>(luaState, SetCameraClearColor2D());
}
#endif

#ifdef CANDERA_3D_ENABLED
struct GetCameraClearColor3D {
    const Color& operator() (const Camera* entity) const
    {
        const ClearMode& clearMode = entity->GetClearMode();
        return clearMode.GetClearColor();
    }
};
#endif

#ifdef CANDERA_2D_ENABLED
struct GetCameraClearColor2D {
    const Color& operator() (const Camera2D* entity) const { return entity->GetClearColor(); }
};
#endif

CANDERA_LUASCRIPTSYSTEM_FUNCTION_T3D_T2D(Camera, Camera2D, GetCameraClearColor, GetColor, GetCameraClearColor3D, GetCameraClearColor2D)

} // namespace Internal

} // namespace Scripting

} // namespace Candera
