//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_SCRIPTING_LUABINDING3D_H
#define CANDERA_SCRIPTING_LUABINDING3D_H

#include <CanderaScripting/LuaScriptSystem.h>
#include <CanderaScripting/Lua/LuaBinding.h>

namespace Candera {

namespace Scripting {

namespace Internal {

/** @addtogroup LuaScripting
 *  @{
 */

/**
 * @brief  LuaBinding3D exposes Candera3D functionality in Lua. See LuaBinding for further details.
 */

class LuaBinding3D : LuaBinding
{
public:
    static void AddFunctions(lua_State* luaState);
    static void Init3D(lua_State* luaState);

private:
    FEATSTD_MAKE_CLASS_STATIC(LuaBinding3D);

    static const luaL_Reg s_additionalFunctions[];
    static const CanderaReferenceType s_canderaObjectReferenceTypes3D[];

    // Appearance -----------------------------------------------------------------------------------------------------------

    template<typename F> static Int CopyAppearanceProperty(lua_State* luaState, F f);

    /** @addtogroup LuaBindingAppearance
     * @{
     *  <hr>\section CanderaLuaHasAppearance HasAppearance
     *  \code
     *  local hasAppearance = Candera.HasAppearance(id)
     *  \endcode
     *  Returns 'true' if the node of the script component identified by 'id' has an appearance, 'nil' otherwise.
     ** @} */
    static Int HasAppearance(lua_State* luaState);

    /** @addtogroup LuaBindingAppearance
     * @{
     *  <hr>\section CanderaLuaSetAppearanceFrom SetAppearanceFrom
     *  \code
     *  Candera.SetAppearanceFrom(destinationId, sourceId)
     *  \endcode
     *  Sets the appearance from the node of the script component identified by 'sourceId' as the appearance in
     *  the node of the script component identified by 'destinationId'.
     ** @} */
    static Int SetAppearanceFrom(lua_State* luaState);

    /** @addtogroup LuaBindingAppearance
     * @{
     *  <hr>\section CanderaLuaHasMaterial HasMaterial
     *  \code
     *  local hasMaterial = Candera.HasMaterial(id)
     *  \endcode
     *  Returns 'true' if the node of the script component identified by 'id' has a material, 'nil' otherwise.
     ** @} */
    static Int HasMaterial(lua_State* luaState);                // Appearance::GetMaterial() != 0

    /** @addtogroup LuaBindingAppearance
     * @{
     *  <hr>\section CanderaLuaSetMaterialFrom SetMaterialFrom
     *  \code
     *  Candera.SetMaterialFrom(destinationId, sourceId)
     *  \endcode
     *  Sets the material from the node of the script component identified by 'sourceId' as the material in
     *  the node of the script component identified by 'destinationId'.
     ** @} */
    static Int SetMaterialFrom(lua_State* luaState);            // Appearance::Get/SetMaterial

    /** @addtogroup LuaBindingAppearance
     * @{
     *  <hr>\section CanderaLuaHasTexture HasTexture
     *  \code
     *  local hasTexture = Candera.HasTexture(id, unit)
     *  \endcode
     *  Returns 'true' if the node of the script component identified by 'id' has a texture at texture unit number 'unit', 'nil' otherwise.
     ** @} */
    static Int HasTexture(lua_State* luaState);                 // Appearance::GetTexture(unit) != 0

    /** @addtogroup LuaBindingAppearance
     * @{
     *  <hr>\section CanderaLuaSetTextureFrom SetTextureFrom
     *  \code
     *  Candera.SetTextureFrom(destinationId, destinationUnit, sourceId, sourceUnit)
     *  \endcode
     *  Sets the texture from the node of the script component identified by 'sourceId' at texture unit number 'sourceUnit'
     *  as the texture in the node of the script component identified by 'destinationId' at texture unit number 'destinationUnit'.
     ** @} */
    static Int SetTextureFrom(lua_State* luaState);             // Appearance::Get/SetTexture

    /** @addtogroup LuaBindingAppearance
     * @{
     *  <hr>\section CanderaLuaHasRenderMode HasRenderMode
     *  \code
     *  local hasRenderMode = Candera.HasRenderMode(id)
     *  \endcode
     *  Returns 'true' if the node of the script component identified by 'id' has a render mode, 'nil' otherwise.
     ** @} */
    static Int HasRenderMode(lua_State* luaState);              // Appearance::GetRenderMode() != 0

    /** @addtogroup LuaBindingAppearance
     * @{
     *  <hr>\section CanderaLuaSetRenderModeFrom SetRenderModeFrom
     *  \code
     *  Candera.SetRenderModeFrom(destinationId, sourceId)
     *  \endcode
     *  Sets the render mode from the node of the script component identified by 'sourceId' as the render mode in
     *  the node of the script component identified by 'destinationId'.
     ** @} */
    static Int SetRenderModeFrom(lua_State* luaState);          // Appearance::Get/SetRenderMode

    /** @addtogroup LuaBindingAppearance
     * @{
     *  <hr>\section CanderaLuaHasShader HasShader
     *  \code
     *  local hasShader = Candera.HasShader(id)
     *  \endcode
     *  Returns 'true' if the node of the script component identified by 'id' has a shader, 'nil' otherwise.
     ** @} */
    static Int HasShader(lua_State* luaState);                  // Appearance::GetShader() != 0

     /** @addtogroup LuaBindingAppearance
     * @{
     *  <hr>\section CanderaLuaSetShaderFrom SetShaderFrom
     *  \code
     *  Candera.SetShaderFrom(destinationId, sourceId)
     *  \endcode
     *  Sets the shader from the node of the script component identified by 'sourceId' as the shader in
     *  the node of the script component identified by 'destinationId'.
     ** @} */
    static Int SetShaderFrom(lua_State* luaState);              // Appearance::Get/SetShader

    /** @addtogroup LuaBindingAppearance
     * @{
     *  <hr>\section CanderaLuaHasShaderParamSetter HasShaderParamSetter
     *  \code
     *  local hasShaderParamSetter = Candera.HasShaderParamSetter(id)
     *  \endcode
     *  Returns 'true' if the node of the script component identified by 'id' has a shader parameter setter, 'nil' otherwise.
     ** @} */
    static Int HasShaderParamSetter(lua_State* luaState);       // Appearance::GetShaderParamSetter() != 0

    /** @addtogroup LuaBindingAppearance
     * @{
     *  <hr>\section CanderaLuaSetShaderParamSetterFrom SetShaderParamSetterFrom
     *  \code
     *  Candera.SetShaderParamSetterFrom(destinationId, sourceId)
     *  \endcode
     *  Sets the shader parameter setter from the node of the script component identified by 'sourceId' as the shader parameter setter in
     *  the node of the script component identified by 'destinationId'.
     ** @} */
    static Int SetShaderParamSetterFrom(lua_State* luaState);   // Appearance::Get/SetShaderParamSetter


    // Material -------------------------------------------------------------------------------------------------------------

    template<typename T, typename F> static Int SetMaterialColor(lua_State* luaState, F f);
    template<typename T, typename F> static Int GetMaterialColor(lua_State* luaState, F f);
    template<typename T, typename F> static Int SetMaterialFloat(lua_State* luaState, F f);
    template<typename T, typename F> static Int GetMaterialFloat(lua_State* luaState, F f);

    /** @addtogroup LuaBindingMaterial
     * @{
     *  <hr>\section CanderaLuaSetMaterialDiffuse SetMaterialDiffuse
     *  \code
     *  Candera.SetMaterialDiffuse(id, red, green, blue, alpha)
     *  \endcode
     *  Sets the diffuse color of the material of the node of the script component identified by 'id'.
     *  The alpha value of the diffuse color represents the alpha value of the material and
     *  can be set with \ref CanderaLuaSetMaterialAlphaValue "SetMaterialAlphaValue", separately.
     ** @} */
    static Int SetMaterialDiffuse(lua_State* luaState);         // Material::SetDiffuse

    /** @addtogroup LuaBindingMaterial
     * @{
     *  <hr>\section CanderaLuaGetMaterialDiffuse GetMaterialDiffuse
     *  \code
     *  local red, green, blue, alpha = Candera.GetMaterialDiffuse(id)
     *  \endcode
     *  Retrieves the diffuse color of the material of the node of the script component identified by 'id'.
     ** @} */
    static Int GetMaterialDiffuse(lua_State* luaState);         // Material::GetDiffuse

    /** @addtogroup LuaBindingMaterial
     * @{
     *  <hr>\section CanderaLuaSetMaterialAmbient SetMaterialAmbient
     *  \code
     *  Candera.SetMaterialAmbient(id, red, green, blue, alpha)
     *  \endcode
     *  Sets the ambient color of the material of the node of the script component identified by 'id'.
     ** @} */
    static Int SetMaterialAmbient(lua_State* luaState);         // Material::SetAmbient

    /** @addtogroup LuaBindingMaterial
     * @{
     *  <hr>\section CanderaLuaGetMaterialAmbient GetMaterialAmbient
     *  \code
     *  local red, green, blue, alpha = Candera.GetMaterialAmbient(id)
     *  \endcode
     *  Retrieves the ambient color of the material of the node of the script component identified by 'id'.
     ** @} */
    static Int GetMaterialAmbient(lua_State* luaState);         // Material::GetAmbient

    /** @addtogroup LuaBindingMaterial
     * @{
     *  <hr>\section CanderaLuaSetMaterialEmissive SetMaterialEmissive
     *  \code
     *  Candera.SetMaterialEmissive(id, red, green, blue, alpha)
     *  \endcode
     *  Sets the emissive color of the material of the node of the script component identified by 'id'.
     ** @} */
    static Int SetMaterialEmissive(lua_State* luaState);        // Material::SetEmissive

    /** @addtogroup LuaBindingMaterial
     * @{
     *  <hr>\section CanderaLuaGetMaterialEmissive GetMaterialEmissive
     *  \code
     *  local red, green, blue, alpha = Candera.GetMaterialEmissive(id)
     *  \endcode
     *  Retrieves the emissive color of the material of the node of the script component identified by 'id'.
     ** @} */
    static Int GetMaterialEmissive(lua_State* luaState);        // Material::GetEmissive

    /** @addtogroup LuaBindingMaterial
     * @{
     *  <hr>\section CanderaLuaSetMaterialSpecular SetMaterialSpecular
     *  \code
     *  Candera.SetMaterialSpecular(id, red, green, blue, alpha)
     *  \endcode
     *  Sets the specular color of the material of the node of the script component identified by 'id'.
     ** @} */
    static Int SetMaterialSpecular(lua_State* luaState);        // Material::SetSpecular

    /** @addtogroup LuaBindingMaterial
     * @{
     *  <hr>\section CanderaLuaGetMaterialSpecular GetMaterialSpecular
     *  \code
     *  local red, green, blue, alpha = Candera.GetMaterialSpecular(id)
     *  \endcode
     *  Retrieves the specular color of the material of the node of the script component identified by 'id'.
     ** @} */
    static Int GetMaterialSpecular(lua_State* luaState);        // Material::GetSpecular

    /** @addtogroup LuaBindingMaterial
     * @{
     *  <hr>\section CanderaLuaSetMaterialSpecularPower SetMaterialSpecularPower
     *  \code
     *  Candera.SetMaterialSpecularPower(id, specularPower)
     *  \endcode
     *  Sets the specular power of the material of the node of the script component identified by 'id'.
     *  The specular power defines the shininess of the material for specular highlights.
     ** @} */
    static Int SetMaterialSpecularPower(lua_State* luaState);   // Material::SetSpecularPower

    /** @addtogroup LuaBindingMaterial
     * @{
     *  <hr>\section CanderaLuaGetMaterialSpecularPower GetMaterialSpecularPower
     *  \code
     *  local specularPower = Candera.GetMaterialSpecularPower(id)
     *  \endcode
     *  Retrieves the specular power of the material of the node of the script component identified by 'id'.
     *  The specular power defines the shininess of the material for specular highlights.
     ** @} */
    static Int GetMaterialSpecularPower(lua_State* luaState);   // Material::GetSpecularPower

    /** @addtogroup LuaBindingMaterial
     * @{
     *  <hr>\section CanderaLuaSetMaterialAlphaValue SetMaterialAlphaValue
     *  \code
     *  Candera.SetMaterialAlphaValue(id, alphaValue)
     *  \endcode
     *  Sets the alpha component of diffuse color of the material of the node of the script component identified by 'id',
     *  which defines the alpha value related to the material.
     *  An alphaValue of 0.0 means fully transparent (invisible), 1.0 means fully opaque.
     ** @} */
    static Int SetMaterialAlphaValue(lua_State* luaState);      // Material::SetAlphaValue

    /** @addtogroup LuaBindingMaterial
     * @{
     *  <hr>\section CanderaLuaGetMaterialAlphaValue GetMaterialAlphaValue
     *  \code
     *  local alphaValue = Candera.GetMaterialAlphaValue(id)
     *  \endcode
     *  Retrieves the alpha component of diffuse color of the material of the node of the script component identified by 'id',
     *  which defines the alpha value related to the material.
     *  An alphaValue of 0.0 means fully transparent (invisible), 1.0 means fully opaque.
     ** @} */
    static Int GetMaterialAlphaValue(lua_State* luaState);      // Material::GetAlphaValue

    // RenderMode -----------------------------------------------------------------------------------------------------------
    template<typename T, typename F> static Int SetRenderModeBool(lua_State* luaState, F f);
    template<typename T, typename F> static Int GetRenderModeBool(lua_State* luaState, F f);

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaSetRenderModeColorWriteEnabled SetRenderModeColorWriteEnabled
     *  \code
     *  Candera.SetRenderModeColorWriteEnabled(id, enableRed, enableGreen, enableBlue, enableAlpha)
     *  \endcode
     *  Enable (true) or disable (false) writing of the incoming fragment's color value into the color buffer
     *  in the render mode of the node of the script component identified by 'id'.
     ** @} */
    static Int SetRenderModeColorWriteEnabled(lua_State* luaState);         // RenderMode::SetColorWriteEnabled

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaSetRenderModeColorWriteRedEnabled SetRenderModeColorWriteRedEnabled
     *  \code
     *  Candera.SetRenderModeColorWriteRedEnabled(id, isColorWriteRedEnabled)
     *  \endcode
     *  Enable (true) or disable (false) writing of the incoming fragment's color red value into color buffer
     *  in the render mode of the node of the script component identified by 'id'.
     ** @} */
    static Int SetRenderModeColorWriteRedEnabled(lua_State* luaState);      // RenderMode::SetColorWriteRedEnabled

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaIsRenderModeColorWriteRedEnabled IsRenderModeColorWriteRedEnabled
     *  \code
     *  local isColorWriteRedEnabled = Candera.IsRenderModeColorWriteRedEnabled(id)
     *  \endcode
     *  Retrieves whether writing color red into the color buffer is enabled (true) or not (nil)
     *  in the render mode of the node of the script component identified by 'id'.
     ** @} */
    static Int IsRenderModeColorWriteRedEnabled(lua_State* luaState);       // RenderMode::IsColorWriteRedEnabled

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaSetRenderModeColorWriteGreenEnabled SetRenderModeColorWriteGreenEnabled
     *  \code
     *  Candera.SetRenderModeColorWriteGreenEnabled(id, isColorWriteGreenEnabled)
     *  \endcode
     *  Enable (true) or disable (false) writing of the incoming fragment's color green value into color buffer
     *  in the render mode of the node of the script component identified by 'id'.
     ** @} */
    static Int SetRenderModeColorWriteGreenEnabled(lua_State* luaState);    // RenderMode::SetColorWriteGreenEnabled

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaIsRenderModeColorWriteGreenEnabled IsRenderModeColorWriteGreenEnabled
     *  \code
     *  local isColorWriteGreenEnabled = Candera.IsRenderModeColorWriteGreenEnabled(id)
     *  \endcode
     *  Retrieves whether writing color green into the color buffer is enabled (true) or not (nil)
     *  in the render mode of the node of the script component identified by 'id'.
     ** @} */
    static Int IsRenderModeColorWriteGreenEnabled(lua_State* luaState);     // RenderMode::IsColorWriteGreenEnabled

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaSetRenderModeColorWriteBlueEnabled SetRenderModeColorWriteBlueEnabled
     *  \code
     *  Candera.SetRenderModeColorWriteBlueEnabled(id, isColorWriteBlueEnabled)
     *  \endcode
     *  Enable (true) or disable (false) writing of the incoming fragment's color blue value into color buffer
     *  in the render mode of the node of the script component identified by 'id'.
     ** @} */
    static Int SetRenderModeColorWriteBlueEnabled(lua_State* luaState);     // RenderMode::SetColorWriteBlueEnabled

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaIsRenderModeColorWriteBlueEnabled IsRenderModeColorWriteBlueEnabled
     *  \code
     *  local isColorWriteBlueEnabled = Candera.IsRenderModeColorWriteBlueEnabled(id)
     *  \endcode
     *  Retrieves whether writing color blue into the color buffer is enabled (true) or not (nil)
     *  in the render mode of the node of the script component identified by 'id'.
     ** @} */
    static Int IsRenderModeColorWriteBlueEnabled(lua_State* luaState);      // RenderMode::IsColorWriteBlueEnabled

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaSetRenderModeColorWriteAlphaEnabled SetRenderModeColorWriteAlphaEnabled
     *  \code
     *  Candera.SetRenderModeColorWriteAlphaEnabled(id, isColorWriteAlphaEnabled)
     *  \endcode
     *  Enable (true) or disable (false) writing of the incoming fragment's color alpha value into color buffer
     *  in the render mode of the node of the script component identified by 'id'.
     ** @} */
    static Int SetRenderModeColorWriteAlphaEnabled(lua_State* luaState);    // RenderMode::SetColorWriteAlphaEnabled

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaIsRenderModeColorWriteAlphaEnabled IsRenderModeColorWriteAlphaEnabled
     *  \code
     *  local isColorWriteAlphaEnabled = Candera.IsRenderModeColorWriteAlphaEnabled(id)
     *  \endcode
     *  Retrieves whether writing alpha into the color buffer is enabled (true) or not (nil)
     *  in the render mode of the node of the script component identified by 'id'.
     ** @} */
    static Int IsRenderModeColorWriteAlphaEnabled(lua_State* luaState);     // RenderMode::IsColorWriteAlphaEnabled

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaSetRenderModeDepthWriteEnabled SetRenderModeDepthWriteEnabled
     *  \code
     *  Candera.SetRenderModeDepthWriteEnabled(id, isDepthWriteEnabled)
     *  \endcode
     *  Enable (true) or disable (false) writing of the incoming fragment's depth value into depth buffer
     *  in the render mode of the node of the script component identified by 'id'.
     *  If the depth buffer test is disabled this setting will be ignored and no writing
     *  to the depth buffer will occur.
     ** @} */
    static Int SetRenderModeDepthWriteEnabled(lua_State* luaState);         // RenderMode::SetDepthWriteEnabled

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaIsRenderModeDepthWriteEnabled IsRenderModeDepthWriteEnabled
     *  \code
     *  local isDepthWriteEnabled = Candera.IsRenderModeDepthWriteEnabled(id)
     *  \endcode
     *  Retrieves whether writing depth into the depth buffer is enabled (true) or not (nil)
     *  in the render mode of the node of the script component identified by 'id'.
     ** @} */
    static Int IsRenderModeDepthWriteEnabled(lua_State* luaState);          // RenderMode::IsDepthWriteEnabled

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaSetRenderModeDepthTestEnabled SetRenderModeDepthTestEnabled
     *  \code
     *  Candera.SetRenderModeDepthTestEnabled(id, isDepthTestEnabled)
     *  \endcode
     *  Enable (true) or disable (false) the test of the incoming fragment's depth value against the depth buffer
     *  in the render mode of the node of the script component identified by 'id'.
     ** @} */
    static Int SetRenderModeDepthTestEnabled(lua_State* luaState);          // RenderMode::SetDepthTestEnabled

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaIsRenderModeDepthTestEnabled IsRenderModeDepthTestEnabled
     *  \code
     *  local isDepthTestEnabled = Candera.IsRenderModeDepthTestEnabled(id)
     *  \endcode
     *  Retrieves whether the test of the incoming fragment's depth value against the depth buffer is enabled (true) or not (nil)
     *  in the render mode of the node of the script component identified by 'id'.
     ** @} */
    static Int IsRenderModeDepthTestEnabled(lua_State* luaState);           // RenderMode::IsDepthTestEnabled

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaSetRenderModeBlendingEnabled SetRenderModeBlendingEnabled
     *  \code
     *  Candera.SetRenderModeBlendingEnabled(id, isBlendingEnabled)
     *  \endcode
     *  Enable (true) or disable (false) blending of the incoming fragment with the color buffer
     *  in the render mode of the node of the script component identified by 'id'.
     ** @} */
    static Int SetRenderModeBlendingEnabled(lua_State* luaState);           // RenderMode::SetBlendingEnabled

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaIsRenderModeBlendingEnabled IsRenderModeBlendingEnabled
     *  \code
     *  local isBlendingEnabled = Candera.IsRenderModeBlendingEnabled(id)
     *  \endcode
     *  Retrieves whether blending of the incoming fragment with the color buffer is enabled (true) or not (nil)
     *  in the render mode of the node of the script component identified by 'id'.
     ** @} */
    static Int IsRenderModeBlendingEnabled(lua_State* luaState);            // RenderMode::IsBlendingEnabled

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaSetRenderModeStencilTestEnabled SetRenderModeStencilTestEnabled
     *  \code
     *  Candera.SetRenderModeStencilTestEnabled(id, isStencilTestEnabled)
     *  \endcode
     *  Enable (true) or disable (false) the stencil test of the incoming fragment with the stencil buffer
     *  in the render mode of the node of the script component identified by 'id'.
     ** @} */
    static Int SetRenderModeStencilTestEnabled(lua_State* luaState);        // RenderMode::SetStencilTestEnabled

    /** @addtogroup LuaBindingRenderMode
     * @{
     *  <hr>\section CanderaLuaIsRenderModeStencilTestEnabled IsRenderModeStencilTestEnabled
     *  \code
     *  local isStencilTestEnabled = Candera.IsRenderModeStencilTestEnabled(id)
     *  \endcode
     *  Retrieves whether the stencil test of the incoming fragment with the stencil buffer is enabled (true) or not (nil)
     *  in the render mode of the node of the script component identified by 'id'.
     ** @} */
    static Int IsRenderModeStencilTestEnabled(lua_State* luaState);         // RenderMode::IsStencilTestEnabled


    // Light ----------------------------------------------------------------------------------------------------------------

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaIsLight IsLight
     *  \code
     *  local isLight = Candera.IsLight(id)
     *  \endcode
     *  Returns 'true' if the node of the script component identified by 'id' is a light, 'nil' otherwise.
     ** @} */
    static Int IsLight(lua_State* luaState);

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaSetLightAmbient SetLightAmbient
     *  \code
     *  Candera.SetLightAmbient(id, red, green, blue, alpha)
     *  \endcode
     *  Set ambient color of the light of the script component identified by 'id'.
     *  Relates to the Material's ambient color. The alpha value is ignored.
     ** @} */
    static Int SetLightAmbient(lua_State* luaState);            // Light::SetAmbient

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaGetLightAmbient GetLightAmbient
     *  \code
     *  local red, green, blue, alpha = Candera.GetLightAmbient(id)
     *  \endcode
     *  Retrieves the ambient color of the light of the script component identified by 'id'.
     *  Relates to the Material's ambient color. The alpha value is ignored.
     ** @} */
    static Int GetLightAmbient(lua_State* luaState);            // Light::GetAmbient

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaSetLightDiffuse SetLightDiffuse
     *  \code
     *  Candera.SetLightDiffuse(id, red, green, blue, alpha)
     *  \endcode
     *  Set diffuse color of the light of the script component identified by 'id'.
     *  Relates to the Material's ambient color. The alpha value is ignored.
     ** @} */
    static Int SetLightDiffuse(lua_State* luaState);            // Light::SetDiffuse

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaGetLightDiffuse GetLightDiffuse
     *  \code
     *  local red, green, blue, alpha = Candera.GetLightDiffuse(id)
     *  \endcode
     *  Retrieves the diffuse color of the light of the script component identified by 'id'.
     *  Relates to the Material's ambient color. The alpha value is ignored.
     ** @} */
    static Int GetLightDiffuse(lua_State* luaState);            // Light::GetDiffuse

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaSetLightSpecular SetLightSpecular
     *  \code
     *  Candera.SetLightSpecular(id, red, green, blue, alpha)
     *  \endcode
     *  Set specular color of the light of the script component identified by 'id'.
     *  Relates to the Material's ambient color. The alpha value is ignored.
     ** @} */
    static Int SetLightSpecular(lua_State* luaState);           // Light::SetSpecular

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaGetLightSpecular GetLightSpecular
     *  \code
     *  local red, green, blue, alpha = Candera.GetLightSpecular(id)
     *  \endcode
     *  Retrieves the specular color of the light of the script component identified by 'id'.
     *  Relates to the Material's ambient color. The alpha value is ignored.
     ** @} */
    static Int GetLightSpecular(lua_State* luaState);           // Light::GetSpecular

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaSetLightDirection SetLightDirection
     *  \code
     *  Candera.SetLightDirection(id, x, y, z)
     *  \endcode
     *  Set direction of the light of the script component identified by 'id'.
     *  The default direction of the light is along the negative Z axis of the light's local coordinate system.
     *  The direction has no effect on lights of type ambient or point.
     ** @} */
    static Int SetLightDirection(lua_State* luaState);          // Light::SetDirection

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaGetLightDirection GetLightDirection
     *  \code
     *  local x,y,z = Candera.GetLightDirection(id)
     *  \endcode
     *  Retrieves the direction of the light of the script component identified by 'id'.
     *  The default direction of the light is along the negative Z axis of the light's local coordinate system.
     *  The direction has no effect on lights of type ambient or point.
     ** @} */
    static Int GetLightDirection(lua_State* luaState);          // Light::GetDirection

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaGetLightWorldDirection GetLightWorldDirection
     *  \code
     *  local x,y,z = Candera.GetLightWorldDirection(id)
     *  \endcode
     *  Retrieves the direction vector of the light of the script component identified by 'id' in world space.
     *  Multiplies the direction vector with the world rotation matrix of this light.
     ** @} */
    static Int GetLightWorldDirection(lua_State* luaState);     // Light::GetWorldDirection

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaSetLightRange SetLightRange
     *  \code
     *  Candera.SetLightRange(id, range)
     *  \endcode
     *  Sets the range of light emission from the light of the script component identified by 'id'.
     *  Default range is 1000.0;
     ** @} */
    static Int SetLightRange(lua_State* luaState);              // Light::SetRange

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaGetLightRange GetLightRange
     *  \code
     *  local range = Candera.GetLightRange(id)
     *  \endcode
     *  Retrieves the range of light emission from the light of the script component identified by 'id'.
     *  Default range is 1000.0;
     ** @} */
    static Int GetLightRange(lua_State* luaState);              // Light::GetRange

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaSetLightSpotAngle SetLightSpotAngle
     *  \code
     *  Candera.SetLightSpotAngle(id, spotAngle)
     *  \endcode
     *  Sets the spot angle of the cone of the light of the script component identified by 'id' in degrees.
     *  The lighting is restricted to the cone set.
     *  The spot angle has an effect on lights of type spotlight, only. The value must lie within [0.0, 90.0].
     *  Default value is 45 degrees.
     ** @} */
    static Int SetLightSpotAngle(lua_State* luaState);          // Light::SetSpotAngle

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaGetLightSpotAngle GetLightSpotAngle
     *  \code
     *  local spotAngle = Candera.GetLightSpotAngle(id)
     *  \endcode
     *  Retrieves the spot angle of the cone of the light of the script component identified by 'id' in degrees.
     *  The lighting is restricted to the cone set.
     *  The spot angle has an effect on lights of type spotlight, only. The value must lie within [0.0, 90.0].
     *  Default value is 45 degrees.
     ** @} */
    static Int GetLightSpotAngle(lua_State* luaState);          // Light::GetSpotAngle

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaSetLightSpotExponent SetLightSpotExponent
     *  \code
     *  Candera.SetLightSpotExponent(id, spotExponent)
     *  \endcode
     *  Sets the spot exponent of the light of the script component identified by 'id'.
     *  The spot exponent controls the distribution of this light within the spot cone.
     *  Larger values lead to a more concentrated cone. The value must lie within [0.0, 128.0]
     *  The default spot exponent is 0.0, resulting in a uniform light distribution.
     ** @} */
    static Int SetLightSpotExponent(lua_State* luaState);       // Light::SetSpotExponent

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaGetLightSpotExponent GetLightSpotExponent
     *  \code
     *  local spotExponent = Candera.GetLightSpotExponent(id)
     *  \endcode
     *  Retrieves the spot exponent of the light of the script component identified by 'id'.
     *  The spot exponent controls the distribution of this light within the spot cone.
     *  Larger values lead to a more concentrated cone. The value must lie within [0.0, 128.0]
     *  The default spot exponent is 0.0, resulting in a uniform light distribution.
     ** @} */
    static Int GetLightSpotExponent(lua_State* luaState);       // Light::GetSpotExponent

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaSetLightAttenuation SetLightAttenuation
     *  \code
     *  Candera.SetLightAttenuation(id, constant, linear, quadratic)
     *  \endcode
     *  Sets the attenuation coefficients for the light of the script component identified by 'id'.
     *  The attenuation factor is: 1 / (constant + (linear * distance) + (quadratic * distance^2)).
     *  The default attenuation coefficients are (1, 0, 0), resulting in no attenuation.
     ** @} */
    static Int SetLightAttenuation(lua_State* luaState);        // Light::SetAttenuation

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaGetLightAttenuation GetLightAttenuation
     *  \code
     *  local constant, linear, quadratic = Candera.GetLightAttenuation(id)
     *  \endcode
     *  Retrieves the attenuation coefficients for the light of the script component identified by 'id'.
     *  The attenuation factor is: 1 / (constant + (linear * distance) + (quadratic * distance^2)).
     *  The default attenuation coefficients are (1, 0, 0), resulting in no attenuation.
     ** @} */
    static Int GetLightAttenuation(lua_State* luaState);        // Light::GetAttenuation

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaSetLightAttenuationEnabled SetLightAttenuationEnabled
     *  \code
     *  Candera.SetLightAttenuationEnabled(id, isLightAttenuationEnabled)
     *  \endcode
     *  Enable (true) or disable (false) attenuation of the light of the script component identified by 'id'.
     *  If disabled, objects illumination appears or disappears abruptly as they enter and exit the light's range.
     *  Has no effect on ambient and directional light.
     *  Attenuation is disabled by default.
     ** @} */
    static Int SetLightAttenuationEnabled(lua_State* luaState); // Light::SetAttenuationEnabled

    /** @addtogroup LuaBindingLight
     * @{
     *  <hr>\section CanderaLuaIsLightAttenuationEnabled IsLightAttenuationEnabled
     *  \code
     *  local isLightAttenuationEnabled = Candera.IsLightAttenuationEnabled(id)
     *  \endcode
     *  Retrieves whether the attenuation of the light of the script component identified by 'id' is enabled (true) or not (nil).
     *  If disabled, objects illumination appears or disappears abruptly as they enter and exit the light's range.
     *  Has no effect on ambient and directional light.
     *  Attenuation is disabled by default.
     ** @} */
    static Int IsLightAttenuationEnabled(lua_State* luaState);  // Light::IsAttenuationEnabled


    // Camera ---------------------------------------------------------------------------------------------------------------

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaSetCameraLookAtVector SetCameraLookAtVector
     *  \code
     *  Candera.SetCameraLookAtVector(id, x, y, z)
     *  \endcode
     *  Sets the camera's look-at vector of the script component identified by 'id'. Initial value is x:0, y:0, z:-1.
     *  The stored look-at vector will be normalized.
     *  The up-vector will be recalculated based on current up-vector (see \ref CanderaLuaSetCameraUpVector "SetUpVector").
     *  If the passed look-at vector is collinear with the up vector, the up vector is set to - or + old look-at vector,
     *  depending on whether the new look-at is in the same or the opposite direction as the old up vector.
     ** @} */
    static Int SetCameraLookAtVector(lua_State* luaState);                  // Camera::SetLookAtVector

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaGetCameraLookAtVector GetCameraLookAtVector
     *  \code
     *  local x,y,z = Candera.GetLightWorldDirection(id)
     *  \endcode
     *  Retrieves normalized look-at vector of the camera of the script component identified by 'id'.
     ** @} */
    static Int GetCameraLookAtVector(lua_State* luaState);                  // Camera::GetLookAtVector

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaGetCameraWorldLookAtVector GetCameraWorldLookAtVector
     *  \code
     *  local x,y,z = Candera.GetCameraWorldLookAtVector(id)
     *  \endcode
     *  Retrieves normalized look-at vector in world space of the camera of the script component identified by 'id'.
     ** @} */
    static Int GetCameraWorldLookAtVector(lua_State* luaState);             // Camera::GetWorldLookAtVector

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaSetCameraUpVector SetCameraUpVector
     *  \code
     *  Candera.SetCameraUpVector(id, x, y, z)
     *  \endcode
     *  Sets the up vector of the camera of the script component identified by 'id'.
     *  The up vector is made orthogonal (calculated as the vector
     *  in the same plane as the look-at vector and the input up vector), and normalized. Subsequent calls to
     *  "SetLookAtVector" will also perform this "make orthogonal" operation.
     ** @} */
    static Int SetCameraUpVector(lua_State* luaState);                      // Camera::SetUpVector

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaGetCameraUpVector GetCameraUpVector
     *  \code
     *  local x,y,z = Candera.GetCameraUpVector(id)
     *  \endcode
     *  Retrieves normalized up-vector of the camera of the script component identified by 'id'.
     ** @} */
    static Int GetCameraUpVector(lua_State* luaState);                      // Camera::GetUpVector

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaGetCameraWorldUpVector GetCameraWorldUpVector
     *  \code
     *  local x,y,z = Candera.GetCameraWorldUpVector(id)
     *  \endcode
     *  Retrieves normalized up vector in world space of the camera of the script component identified by 'id'.
     ** @} */
    static Int GetCameraWorldUpVector(lua_State* luaState);                 // Camera::GetWorldUpVector

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaGetCameraRightVector GetCameraRightVector
     *  \code
     *  local x,y,z = Candera.GetCameraRightVector(id)
     *  \endcode
     *  Retrieves normalized right vector of the camera of the script component identified by 'id'.
     ** @} */
    static Int GetCameraRightVector(lua_State* luaState);                   // Camera::GetRightVector

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaGetCameraWorldRightVector GetCameraWorldRightVector
     *  \code
     *  local x,y,z = Candera.GetCameraWorldRightVector(id)
     *  \endcode
     *  Retrieves normalized right-vector in world space of the camera of the script component identified by 'id'.
     ** @} */
    static Int GetCameraWorldRightVector(lua_State* luaState);              // Camera::GetWorldRightVector

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaCameraLookAtWorldPoint CameraLookAtWorldPoint
     *  \code
     *  Candera.CameraLookAtWorldPoint(id, x, y, z)
     *  \endcode
     *  Points the camera of the script component identified by 'id' at a location in world space.
     *  This helper method generates the look-at-vector for the camera, based on its current position and the target point.
     ** @} */
    static Int CameraLookAtWorldPoint(lua_State* luaState);                 // Camera::LookAtWorldPoint

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaCameraRotateAroundWorldAxis CameraRotateAroundWorldAxis
     *  \code
     *  Candera.CameraRotateAroundWorldAxis(id, x, y, z, angleDegrees)
     *  \endcode
     *  Rotates the camera of the script component identified by 'id' around an arbitrary world axis.
     *  The rotation around the world axis is specified by the angle in degrees.
     ** @} */
    static Int CameraRotateAroundWorldAxis(lua_State* luaState);            // Camera::RotateAroundWorldAxis

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaSetCameraViewingFrustumCullingEnabled SetCameraViewingFrustumCullingEnabled
     *  \code
     *  Candera.SetCameraViewingFrustumCullingEnabled(id, isViewingFrustumCullingEnabled)
     *  \endcode
     *  Enables or disables viewing frustum culling of the camera of the script component identified by 'id'. Enabled per default.
     ** @} */
    static Int SetCameraViewingFrustumCullingEnabled(lua_State* luaState);  // Camera::SetViewingFrustumCullingEnabled

    /** @addtogroup LuaBindingCamera
     * @{
     *  <hr>\section CanderaLuaIsCameraViewingFrustumCullingEnabled IsCameraViewingFrustumCullingEnabled
     *  \code
     *  local isViewingFrustumCullingEnabled = Candera.IsCameraViewingFrustumCullingEnabled(id)
     *  \endcode
     *  Retrieves whether viewing frustum culling of the camera of the script component identified by 'id' is enabled (true)
     *  or not (nil).
     ** @} */
    static Int IsCameraViewingFrustumCullingEnabled(lua_State* luaState);   // Camera::IsViewingFrustumCullingEnabled


    // ShaderParamSetter ----------------------------------------------------------------------------------------------------

    /** @addtogroup LuaBindingShaderParamSetter
     * @{
     *  <hr>\section CanderaLuaSetShaderUniform SetShaderUniform
     *  \code
     *  Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.Float, x)
     *  Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.FloatVec2, x, y)
     *  Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.FloatVec3, x, y, z)
     *  Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.FloatVec4, x, y, z, w)
     *  Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.Integer, x)
     *  Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.IntegerVec2, x, y)
     *  Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.IntegerVec3, x, y, z)
     *  Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.IntegerVec4, x, y, z, w)
     *  Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.Bool, x)
     *  Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.BoolVec2, x, y)
     *  Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.BoolVec3, x, y, z)
     *  Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.BoolVec4, x, y, z, w)
     *  Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.Sampler2D, x)
     *  Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.SamplerCube, x)
     *  \endcode
     *  Sets the uniform of the given name and type with values in the shader parameter setter from the appearance of the node of the script
     *  component identified by 'id'.
     ** @} */
    static Int SetShaderUniform(lua_State* luaState);   // ShaderParamSetter::SetUniform

    /** @addtogroup LuaBindingShaderParamSetter
     * @{
     *  <hr>\section CanderaLuaGetShaderUniformType GetShaderUniformType
     *  \code
     *  local uniformType = Candera.GetShaderUniformType(id, 'uniformName')
     *  \endcode
     *  Gets the type of the uniform specified by 'uniformName' from the appearance of the node of the script component identified by 'id'.
     ** @} */
    static Int GetShaderUniformType(lua_State* luaState);   // ShaderParamSetter::GetUniformType

    /** @addtogroup LuaBindingShaderParamSetter
     * @{
     *  <hr>\section CanderaLuaGetShaderUniform GetShaderUniform
     *  \code
     *  local x = Candera.GetShaderUniform(id, 'uniformName')
     *  local x,y = Candera.GetShaderUniform(id, 'uniformName')
     *  local x,y,z = Candera.GetShaderUniform(id, 'uniformName')
     *  local x,y,z,w = Candera.GetShaderUniform(id, 'uniformName')
     *  \endcode
     *  Gets the value(s) (depending on the type) of the uniform specified by 'uniformName' from the appearance of the node of the script component
     *  identified by 'id'.
     ** @} */
    static Int GetShaderUniform(lua_State* luaState);   // ShaderParamSetter::GetUniformData
};

/** @} */ // end of LuaScripting

} // namespace Internal

} // namespace Scripting

} // namespace Candera

#endif
