//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_SERIALIZATION_INTERFACE_H)
#define CANDERA_SERIALIZATION_INTERFACE_H

#include <Candera/Environment.h>

#include <CanderaPlatform/OS/StringPlatform.h>
#include <CanderaPlatform/OS/MathPlatform.h>

#include <FeatStd/Util/NumericUtil.h>

namespace Candera {
/// @addtogroup CanderaPlatformOS
/// @{
    /**
    * @brief Incapsulates serialization routines
    */

    namespace Internal {
        template <typename T, bool IsSigned = (::FeatStd::Internal::IsSigned<T>::Value == true), bool IsUnsigned = (::FeatStd::Internal::IsUnsigned<T>::Value == true)>
        struct TypedSerializationPlatform {
        };

        template <typename T>       
        struct TypedSerializationPlatform<T, false, true>
        {
            static UInt ToBinary(const Char* buf, T& dest, Int* count = 0)
            {
                //start dest from 0
                dest = 0;

                if (buf == 0) {
                    return 0;
                }
                const Char* p = buf;

                //fail if there is no number
                if ((*p < '0') || (*p > '9')) {
                    return 0;
                }

                //read number
                while((*p >= '0') && (*p <= '9')){
                    dest = ((dest * 10) + *p) - '0';
                    p++;
                }

                //store number of characters processed.
                if (count != 0) {
                    *count = FeatStd::Internal::NumericConversion<Int>(p - buf);
                }

                return 1;
            }
        };

        template <typename T>        
        struct TypedSerializationPlatform<T, true, false>
        {
            static UInt ToBinary(const Char* buf, T& dest, Int* count = 0)
            {
                dest = 0;

                if (buf == 0) {
                    return 0;
                }

                const Char* p = buf;
                bool negSign = false;

                //check sign
                if (*p == '-') {
                    ++p;
                    negSign = true;
                }
                if (*p == '+') {
                    ++p;
                }

                //remove spaces
                while((*p == ' ') || (*p == '\t') || (*p == '\n') || (*p == '\r')) {
                    ++p;
                }
                SizeType signCount = p - buf;

                UInt uidest;
                Int uicount = 0;
                if (TypedSerializationPlatform<UInt>::ToBinary(p, uidest, &uicount) != 1) {
                    dest = 0;
                }
                else {
                    dest = Int(uidest);
                }
                SizeType stcount = uicount;

                //add sign
                if (negSign){
                    dest = -dest;
                }
                stcount += signCount;

                if (count != 0) {
                    *count = FeatStd::Internal::NumericConversion<Int>(stcount);
                }

                return 1;
            }
        };
    }

    struct SerializationPlatform {
        static UInt ToBinary(const Char* buf, Float& dest, Int* count = 0);
        //static UInt ToBinary(const Char* buf, Int& dest, Int* count = 0) 
        //{ return ToBinary<Int>(buf, dest, count); }
        //static UInt ToBinary(const Char* buf, UInt& dest, Int* count = 0)
        //{ return ToBinary<UInt>(buf, dest, count); }
        static UInt ToBinary(const Char* buf, Char* dest, Int* count = 0);

        template <typename T>
        static UInt ToBinary(const Char* buf, T& dest, Int* count = 0) {
            return ::Candera::Internal::TypedSerializationPlatform<T>::ToBinary(buf, dest, count);
        }

        template <typename Iterator>
        static UInt ToBinary(const Char* buf, const Char* separators, Iterator begin, Iterator end, Int* count = 0);

        static UInt FromBinary(Char* buf, UInt sizeOfBuf, Float src, Int* count = 0);
        static UInt FromBinary(Char* buf, UInt sizeOfBuf, Int src, Int* count = 0);
        static UInt FromBinary(Char* buf, UInt sizeOfBuf, UInt src, Int* count = 0);
        static UInt FromBinary(Char* buf, UInt sizeOfBuf, const Char* src, Int* count = 0);

        template <typename Iterator>
        static UInt FromBinary(Char* buf, UInt sizeOfBuf, const Char* separators, Iterator begin, Iterator end, Int* count = 0);

    private:
        static const Char* SkipSeparators(const Char* buf, const Char* separators);
        static Char* WriteSeparators(Char* buf, UInt sizeOfBuf, const Char* separators);
    }; //struct SerializationPlatform
       

template <typename Iterator>
UInt SerializationPlatform::ToBinary(const Char* buf, const Char* separators, Iterator begin, Iterator end, Int* count)
{
    const Char* p = buf;
    UInt items = 0;

    p = SkipSeparators(p, separators);
    for (Iterator it = begin; it != end; it ++){
        Int read;
        if (ToBinary(p, *it, &read) == 1) {
            items ++;
            p += read;
            p = SkipSeparators(p, separators);
        }
        else {
            break;
        }
    }

    if (count) {
        *count = FeatStd::Internal::NumericConversion<Int>(p - buf); 
    }
    return items;
}

template <typename Iterator>
UInt SerializationPlatform::FromBinary(Char* buf, UInt sizeOfBuf, const Char* separators, Iterator begin, Iterator end, Int* count)
{
    Char* p = buf;
    UInt items = 0;

    for (Iterator it = begin; it != end; it ++){
        Int written;
        if (it != begin) {
            Char* sepp = WriteSeparators(p, sizeOfBuf, separators);
            sizeOfBuf -= FeatStd::Internal::NumericConversion<UInt>(sepp - p);
            p = sepp;
        }
        if (FromBinary(p, sizeOfBuf, *it, &written) == 1) {
            items ++;
            p += written;
            sizeOfBuf -= written;
        }
        else {
            break;
        }
    }

    if (count) {
        *count = FeatStd::Internal::NumericConversion<UInt>(p - buf);
    }
    return items;
}
/// @}
} //namespace Candera
#endif //CANDERA_SERIALIZATION_INTERFACE_H
