//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_PlatformPerfCounter_h)
#define Candera_PlatformPerfCounter_h

#include <Candera/Environment.h>
#include <CanderaPlatform/OS/PlatformInclude.h>

#include CANDERA_OS_INCLUDE(PerfCounter)

namespace Candera {
/// @addtogroup CanderaPlatformOS
/// @{
    /** Provides high resolution performance counter support for Candera.
     */
    struct PerfCounterPlatform {
        /**
         * Provides the current time in 20us units. To get the actual time in microseconds,
         * call function @see PerfCounterPlatform::ToMicroSec. This time is only to be
         * used for time measurements.
         * @return The current time in 20us units.
         */
        static UInt32 Now() {
            return Platform::PerfCounter::Now();
        }

        /** return elapsed time of given performance counter
            @param counter the counter variable setup by @see PerfCounterPlatform::Start
            @return elapsed time in 20us units */
        static UInt32 Elapsed(UInt32 counter) {
            return Platform::PerfCounter::Elapsed(counter);
        }

        /** return elapsed time of given performance counter for time periods smaller than 1sec
            @param counter the counter variable setup by @see PerfCounterPlatform::Start
            @return elapsed time in 20us units */
        static UInt16 ElapsedShort(UInt32 counter) {
            return static_cast<UInt16>(Elapsed(counter));
        }

        /**
         * Computes the duration of a time period delimited by two 32-bit
         * unsigned integer values.
         *
         * @param start Start time of the period to compute.
         * @param end   End time of the period to compute.
         * @return The duration starting from start and ending at end.
         */
        static UInt32 Duration(UInt32 start, UInt32 end) {
            return Platform::PerfCounter::Duration(start, end);
        }

        /**
         * Computes the duration of a time period delimited by two 16-bit
         * unsigned integer values.
         *
         * @param start Start time of the period to compute.
         * @param end   End time of the period to compute.
         * @return The duration starting from start and ending at end.
         */
        static UInt32 ShortDuration(UInt16 start, UInt16 end) {
            return Platform::PerfCounter::ShortDuration(start, end);
        }

        /** Converts time returned by @see PerfCounterPlatform::Elapsed and PerfCounterPlatform::ElapsedShort to us
            @param duration in 20 us units.
            @return time elapsed in us */
        static UInt32 ToMicroSec(UInt32 duration) {
            return duration * CANDERA_PERFCOUNTER_RESOLUTION;
        }
    };
/// @}
}   // namespace PerfCounter
#endif  // Candera_PlatformPerfCounter_h
