//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GENIVI_WINDOW_SURFACE_RENDER_TARGET_H)
#define GENIVI_WINDOW_SURFACE_RENDER_TARGET_H

#include <CanderaPlatform/Device/Common/Internal/GDU/GduBaseRenderTarget.h>
#include <CanderaPlatform/Device/Common/Internal/GDU/GduRenderTarget3DAttachment.h>
#include <CanderaPlatform/Device/Common/Internal/GDU/GduRenderTarget2DAttachment.h>
#include <CanderaPlatform/Device/Common/Internal/GDU/GduContext2DAttachment.h>
#include <CanderaPlatform/Device/Common/Internal/EGL/EglContextProvider.h>
#include <CanderaPlatform/Device/Common/Internal/EGL/EglWindowSurfaceProvider.h>
#include <CanderaPlatform/Device/Common/Internal/EGL/EglAntiAliasedWindowSurfaceProperties.h>
#include <CanderaPlatform/Device/Common/Internal/GL/GlFlushSync.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Context2DOver3D.h>

#include <CanderaPlatform/Device/Genivi/Target/ILM/GeniviWaylandFrameBuffer.h>//"GeniviWaylandFrameBuffer.h"
#include <CanderaPlatform/Device/Genivi/Target/ILM/GeniviSurface.h>//"GeniviSurface.h"

#include <CanderaPlatform/Device/Genivi/Target/GeniviEglDisplay.h>//"GeniviEglDisplay.h"
#include <Candera/System/Diagnostics/VideoMemoryStatistic.h>
#include <FeatStd/Diagnostics/Measurable.h>

namespace Candera
{

namespace GeniviWindowSurfaceRenderTargetNameSpace
{
    // EglContextProvider requires a class from which to
    // retrieve a window and a display.
    // Get the display from an GeniviEglDisplay, and the window from GeniviLayer.
    typedef EglWindowSurfaceProvider<
        GeniviEglDisplay,
        GeniviSurface>
            ContextProviderBase;

    // Assemble the context provider.
    typedef EglContextProvider<
        ContextProviderBase,
        GeniviSurface,
        EglAntiAliasedWindowSurfaceProperties>
            ContextProvider;

    // Assemble the render target.
    // Support EglAntiAliasedWindowSurfaceProperties feature set.
    // The layer is forwarded to EglWindowSurfaceProvider.
    // Do synchronization by finalizing. No synchronization needed between
    // 2D and 3D.
    typedef GduBaseRenderTarget<
        ContextProvider,
        GeniviWaylandFrameBuffer,
        EglAntiAliasedWindowSurfaceProperties,
        GlFlushSync,
        GeniviSurface>
            RenderTargetBase;

#ifdef CANDERA_2D_ENABLED
    // Attach a 2D component to the 3D context.
    // This binds a Context2DOver3D.
    // The size is retrieved form the layer.
    // This can be bound to the RenderTargetBase because the
    // FrameBuffer doesn't need to know about the 2D context.
    typedef GduContext2DAttachment<
        RenderTargetBase,
        GeniviSurface,
        EglAntiAliasedWindowSurfaceProperties,
        Internal::Context2DOver3D,
        GeniviSurface>
            RenderTargetBase2D;
#else
    typedef RenderTargetBase RenderTargetBase2D;
#endif //CANDERA_2D_ENABLED

    // Add the required attachments.
    // This surface needs to support the 2 render target types.
    typedef GduRenderTarget3DAttachment<
        RenderTargetBase2D>
            RenderTarget3D;
#ifdef CANDERA_2D_ENABLED
    typedef GduRenderTarget2DAttachment<
        RenderTarget3D>
            RenderTarget;
#else

    typedef RenderTarget3D RenderTarget;
#endif //CANDERA_2D_ENABLED
}

/**

 * @brief GeniviWindowSurfaceRenderTarget represents a Genivi window surface with dedicated interface to render into.
 *
 * This object can be used as RenderTarget2D/3D, and it's content is flushed to the display. It is useful
 * for on-screen rendering. The interface for this is missing.
 * Note that this class should not be accessed directly by the application, instead the interface of
 * GeniviFrameBufferObject should be used.
 */
class GeniviWindowSurfaceRenderTarget : public GeniviWindowSurfaceRenderTargetNameSpace::RenderTarget,
        public FeatStd::Diagnostics::Measurable
{
public:
    GeniviWindowSurfaceRenderTarget() : m_mustReleasePool(false), m_size(0) {}
    ~GeniviWindowSurfaceRenderTarget() { Unload(); }

    bool Upload(Int displayId, Support& support, EglAntiAliasedWindowSurfaceProperties& properties)
    {
        // Find a proper context resource pool to attach to the window surface.
        // Use the default ContextRecourcePool for all surfaces as this is required for Candera 2D over 3D and text rendering.
        ContextResourcePool *pool = &ContextResourcePool::GetDefault();
        if (pool == 0) {
            pool = Internal::GduTools::GetContextResourcePoolWithoutGdu(
                displayId,
                c_geniviWindowSurfaceType);
            SetContextResourcePool(pool);
            m_mustReleasePool = true;
        }
        if (pool == 0) {
            CANDERA_DEVICE_LOG_ERROR("No Context resource pool available.");
            return false;
        }

        if (!Base::Upload(displayId, support, properties)) {
            return false;
        }

        UInt bytesPerPixel = (properties.GetRedSize()
                + properties.GetGreenSize()
                + properties.GetBlueSize()
                + properties.GetAlphaSize()
                + properties.GetDepthSize()
                + properties.GetStencilSize()) >> 3; // divide through 8 to get bytes.
        //OpenGL ES requires double buffering, so assume two framebuffers.
        //Note: On some devices this can be triple buffering as well.
        m_size = bytesPerPixel * support.GetWidth() * support.GetHeight() * 2;

        Diagnostics::VideoMemoryStatistic::OnWindowSurfaceUploaded(*this);

        return true;
    }

    void Unload()
    {
        Base::Unload();
        if (m_mustReleasePool) {
            SetContextResourcePool(0);
            m_mustReleasePool = false;
        }

        m_size = 0;
        Diagnostics::VideoMemoryStatistic::OnWindowSurfaceUnloaded(*this);
    }

    virtual UInt GetSize() const {
        return m_size;
    }

private:
    bool m_mustReleasePool;
    UInt m_size;
};

}


#endif
