//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <CanderaPlatform/OS/StringPlatform.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageTrace.h>
#include <CanderaPlatform/Device/Common/EGL/EglInclude.h>

#include <FeatStd/Util/PointerUtil.h>

#include "GeniviWaylandContext.h"
#if (defined(USE_WESTON_IVI_SHELL))
#include "WaylandInc.h"
#endif

namespace Candera {
// =============================================================================
// Implement server info protocol to query surface handle from wayland and ILM.
// =============================================================================
struct ServerInfoListener {
    void (* GetConnectionId)(void * data,
         struct serverinfo * serverinfo,
         uint32_t connectionId);
};

// -----------------------------------------------------------------------------
static const struct wl_interface * types[] = {
    NULL,
};

// -----------------------------------------------------------------------------
static const struct wl_message ServerInfoRequests[] = {
    { "GetConnectionId", "", types + 0 },
};

// -----------------------------------------------------------------------------
static const struct wl_message ServerInfoEvents[] = {
    { "ConnectionId", "u", types + 0 },
};

// -----------------------------------------------------------------------------
extern "C" WL_EXPORT const struct wl_interface ServerInfoInterface = {
    "serverinfo",
    1,
    sizeof(ServerInfoRequests) / sizeof(ServerInfoRequests[0]), ServerInfoRequests,
    sizeof(ServerInfoEvents) / sizeof(ServerInfoEvents[0]), ServerInfoEvents,
};

// -----------------------------------------------------------------------------
enum ServerInfoProtocol{
    GetConnectionIdReq = 0
};

// -----------------------------------------------------------------------------
static ServerInfoListener s_serverInfoListener;

// =============================================================================
// Implement Candera::GeniviWaylandContext
// =============================================================================

// -----------------------------------------------------------------------------
GeniviWaylandContext::GeniviWaylandContext():
    m_listener(),
    m_handle(0),
    m_compositor(0),
    m_registry(0),
    m_connectionId(0)
#if (defined(USE_WESTON_IVI_SHELL))
    ,m_iviApplication(0)
#endif
{
}

// -----------------------------------------------------------------------------
GeniviWaylandContext::~GeniviWaylandContext()
{
    m_compositor = 0;
    m_handle = 0;
    m_registry = 0;
    m_connectionId = 0;
#if (defined(USE_WESTON_IVI_SHELL))
    m_iviApplication = 0;
#endif
}

// -----------------------------------------------------------------------------
bool GeniviWaylandContext::Upload(EGLNativeDisplayType handle)
{
    if (m_handle != handle) {
        Unload();
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(740, "Unusual pointer cast [MISRA C++ Rule 5-2-7]: Conversion is intended and valid.")
        m_handle = reinterpret_cast<struct wl_display *>(handle);
    }

    if (m_handle == 0) {
        CANDERA_DEVICE_LOG_ERROR("Wayland context could not be uploaded with display handle 0.");
        return false;
    }

    // Get compositor proxy from wayland display.
    m_registry = wl_display_get_registry(m_handle);

    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(916, "Pointer assignment conversion [MISRA C++ Rule 5-2-7]: Conversion is intended and valid.")
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(64, "Error -- Type mismatch (assignment): false positive")
    m_listener.global = WaylandGlobalListener;
    m_listener.global_remove = WaylandGlobalRemoveListener;
    static_cast<void>(wl_registry_add_listener(m_registry, &m_listener, this));

    s_serverInfoListener.GetConnectionId = WaylandServerInfoListener;

    static_cast<void>(wl_display_dispatch(m_handle));
    static_cast<void>(wl_display_roundtrip(m_handle));

    if (m_compositor == 0) {
        CANDERA_DEVICE_LOG_ERROR("Could not acquire compositor.");
        return false;
    }
    return true;
}

// -----------------------------------------------------------------------------
void GeniviWaylandContext::Unload()
{
    if (m_compositor != 0) {
        wl_compositor_destroy(m_compositor);
    }

    if (m_handle != 0) {
        wl_display_disconnect(m_handle);
    }
}

// -----------------------------------------------------------------------------
wl_compositor * GeniviWaylandContext::GetCompositor() const
{
    return m_compositor;
}

// -----------------------------------------------------------------------------
UInt32 GeniviWaylandContext::GetConnectionId() const
{
    return m_connectionId;
}

// -----------------------------------------------------------------------------
void * GeniviWaylandContext::CreateNativeDisplayHandle(const Char * displayId)
{
    //Connect to wayland server.
    m_handle = wl_display_connect(displayId);
    if (m_handle == 0) {
        CANDERA_DEVICE_LOG_ERROR("Could not acquire handle to native display.");
    }
    return m_handle;
}

// -----------------------------------------------------------------------------
void * GeniviWaylandContext::GetNativeDisplayHandle() const
{
    return m_handle;
}

#if (defined(USE_WESTON_IVI_SHELL))
void * GeniviWaylandContext::GetIviApplication()
{
   return m_iviApplication;
}
#endif

// -----------------------------------------------------------------------------
void GeniviWaylandContext::WaylandGlobalListener(void * data,
                                                 struct wl_registry * waylandRegistry,
                                                 uint32_t name,
                                                 const Char * interface,
                                                 uint32_t version)
{
    GeniviWaylandContext * context = reinterpret_cast<GeniviWaylandContext *>(data);
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(64, "Error -- Type mismatch (assignment): false positive")
#if (defined(USE_WESTON_IVI_SHELL))
    if (Candera::StringPlatform::CompareStrings("ivi_application", interface) == 0) {
       context->m_iviApplication = FeatStd::Internal::PointerToPointer<struct ivi_application *>(wl_registry_bind(waylandRegistry, 
                                                                                          name, 
                                                                                          &ivi_application_interface, 
                                                                                          1));
    }
    else 
#endif
    if (Candera::StringPlatform::CompareStrings("wl_compositor", interface) == 0) {
        context->m_compositor = FeatStd::Internal::PointerToPointer<struct wl_compositor *>(wl_registry_bind(waylandRegistry,
                                                                                          name,
                                                                                          &wl_compositor_interface,
                                                                                          version));
    }
    else {
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(64, "Error -- Type mismatch (assignment): false positive")
        if (Candera::StringPlatform::CompareStrings("serverinfo", interface) == 0) {
            s_serverInfoListener.GetConnectionId = WaylandServerInfoListener;

            wl_proxy * lServerInfo = FeatStd::Internal::PointerToPointer<struct wl_proxy *>(wl_registry_bind(waylandRegistry,
                                                                                          name,
                                                                                          &ServerInfoInterface,
                                                                                          1));
            
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(740, "Unusual pointer cast [MISRA C++ Rule 5-2-7]: Conversion is intended and valid.")
            static_cast<void>(wl_proxy_add_listener(lServerInfo,
                                  reinterpret_cast<void (**) (void)>(&s_serverInfoListener),
                                  data));

            wl_proxy_marshal(lServerInfo, GetConnectionIdReq);
        }
    }
}

// -----------------------------------------------------------------------------
void GeniviWaylandContext::WaylandGlobalRemoveListener(void * /*data*/,
                                                       struct wl_registry * /*waylandRegistry*/,
                                                       uint32_t /*name*/)
{
}

// -----------------------------------------------------------------------------
void GeniviWaylandContext::WaylandServerInfoListener(void * data,
                                                     struct serverinfo * /*serverInfo*/,
                                                     uint32_t connectionId)
{
    Candera::GeniviWaylandContext * ctx = FeatStd::Internal::PointerToPointer<Candera::GeniviWaylandContext *>(data);
    ctx->m_connectionId = connectionId;
}

}
