//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GENIVI_TARGET_DISPLAY_H)
#define GENIVI_TARGET_DISPLAY_H

#include <Candera/System/Container/Vector.h>

#include <CanderaPlatform/Device/Genivi/GeniviDisplay.h>//"GeniviDisplay.h"
#include "GeniviWaylandContext.h"

namespace Candera
{
/** @addtogroup GeniviGDITarget
 *  @{
 */

/**
 * @brief The class GeniviTargetDisplay represents a display on the "Genivi" platform.
 *        As addition to base GeniviDisplay functionality, it registers itself as
 *        IVI layer manager (ILM) client and also creates a wayland context for this display.
 */
class GeniviTargetDisplay : public GeniviDisplay
{
    friend class GeniviDevicePackageInterface;
    friend class GeniviSurface;

    public:

    /**
     * @brief Maps an ilm surface id to a wayland surface, which
     * is especially useful when it is needed to determine
     * which wl_surface is rendered on which ilm surface,
     * or vice versa which ilm surface id displays a certain
     * wl_surface.
     */
    struct IlmWaylandMapItem {
        wl_surface* m_waylandSurface;
        UInt32 m_ilmSurfaceId;
    };

    /**
     * Destructor
     */
    ~GeniviTargetDisplay();

    /**
     * Create the video resources required for manipulating a display.
     * Registers itself as ILM client and creates a wayland context.
     * @param settings common parameters used in initialization of a display.
     * @return true if everything uploaded successfully, false otherwise.
     */
    virtual bool UploadInternal(const CommonSettings& settings);

    /**
     * An alternative to Upload. Native resources are created outside, and only
     * a handle is attached to the display object. This function creates the associations
     * needed internally by the graphic package. (e.g. a simulated display).
     * @param intendedWidth intended width display. (in most cases it should correspond
     *        to actual width of the display)
     * @param intendedHeight intended height display. (in most cases it should correspond
     *        to actual height of the display)
     * @param handle native handle of the display.
     * @return true if everything uploaded successfully, false otherwise.
     */
    virtual bool AttachNativeHandle(Int intendedWidth, Int intendedHeight, void* handle);

    /**
     * Unload all video resources and/or the associations created by AttachHandle.
     */
    virtual void UnloadInternal();

    /**
     * @brief Gets the wayland context of this GeniviTargetDisplay.
     *        Used to create GeniviSurfaces.
     * @return The wayland context of this GeniviTargetDisplay.
     */
    GeniviWaylandContext* GetWaylandContext();

    /**
     * @brief Sets the id of the wayland display to connect to.
     * @param displayId The id of the display to connect to.
     */
    void SetWaylandDisplayId(const Char* displayId);

    /**
     * @brief Gets the id of the wayland display to connect to.
     * @return The id of the display to connect to.
     */
    const Char* GetWaylandDisplayId() const;

    /**
     *  Get number of ilmWaylandMapItems available.
     *  @return Number of ilmWaylandMapItems available.
     */
    UInt32 GetNumberOfIlmWaylandMapItems() const;

    /**
     *  Get IlmWaylandMapItem at index index.
     *  @param index Index to get IlmWaylandItem at.
     *  @return Pointer to IlmWaylandItem, 0 if nothing exists at index.
     */
    IlmWaylandMapItem* GetIlmWaylandMapItem(UInt32 index);



private:
    GeniviTargetDisplay(Int displayId);

    const Char* m_waylandDisplayId;
    GeniviWaylandContext m_waylandContext;

    FeatStd::Internal::Vector<IlmWaylandMapItem> m_ilmWaylandMap;


};

/** @}*/ //end of GeniviGDITarget

}

#endif
