//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <CanderaPlatform/Device/Common/Base/DevicePackageTrace.h>

#include "GeniviSurface.h"
#include <CanderaPlatform/Device/Genivi/GeniviSurfaceProperties.h>
#include "GeniviTargetDisplay.h"
#include "GeniviWaylandContext.h"
#include "WaylandInc.h"

#include <ilm/ilm_client.h>
#include <ilm/ilm_control.h>

#if defined(CGIDEVICE_SCREENBROKER_ENABLED)
    #include <ScreenBroker/Client/ClientApi.h>
    #include <ScreenBroker/Client/ClientUtil.h>
#endif

namespace Candera
{

GeniviSurface::CallbackFunction GeniviSurface::s_callbackFunction = 0;

// -----------------------------------------------------------------------------
EGLNativeWindowType GeniviSurface::CreateNativeWindow(Int displayId)
{
    FEATSTD_LINT_SYMBOL(550, lObject, "[MISRA C++ Rule 0-1-4] lObject is not accessed is a false positive")
    
    Candera::GeniviTargetDisplay * lDisplay = static_cast<GeniviTargetDisplay *>(
                                                  Candera::GeniviDevicePackageInterface::GetInstance().GetDisplay(displayId));
    if (0 == lDisplay) {
        CANDERA_DEVICE_LOG_ERROR("Couldn't retrieve target display for display %d", displayId);
        FEATSTD_DEBUG_ASSERT(false);
        return 0;
    }

    if (0 == m_properties) {
        CANDERA_DEVICE_LOG_ERROR("Invalid properties for render target on display %d", displayId);
        FEATSTD_DEBUG_ASSERT(false);
        return 0;
    }

    // Calculate window dimension based on the given dimensions of the render target and the display
    Int lWindowWidth = ((m_properties->GetWidth() < lDisplay->GetParameters().width) && (m_properties->GetWidth() > 0 )) ?
                           m_properties->GetWidth() :
                           lDisplay->GetParameters().width;

    Int lWindowHeight = ((m_properties->GetHeight() < lDisplay->GetParameters().height) && (m_properties->GetHeight() > 0 )) ?
                            m_properties->GetHeight() :
                            lDisplay->GetParameters().height;

    // Request wayland compositor for surface creation
    GeniviWaylandContext * lContext = lDisplay->GetWaylandContext();
    if (0 == lContext) {
        CANDERA_DEVICE_LOG_ERROR("Could not acquire wayland context for display %d!", displayId);
        return 0;
    }
    if (0 == lContext->GetCompositor()) {
        CANDERA_DEVICE_LOG_ERROR("Invalid wayland compositor referenced by display %d!", displayId);
        return 0;
    }

    // Create wayland surface
    t_ilm_layer lLayerId = static_cast<t_ilm_layer>(m_properties->GetLayerId());
    t_ilm_surface lSurfaceId = static_cast<t_ilm_surface>(m_properties->GetSurfaceId());
    m_waylandSurface = wl_compositor_create_surface(lContext->GetCompositor());
    if (0 == m_waylandSurface) {
        CANDERA_DEVICE_LOG_ERROR("Could not acquire a new wayland surface instance for surface %u!", lSurfaceId);
        return 0;
    }

    // Register surface to ILM
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(740, "Unusual pointer cast [MISRA C++ Rule 5-2-7]: Conversion is intended and valid.")
#if ((WAYLAND_VERSION_MAJOR == 1) && (WAYLAND_VERSION_MINOR >= 9))
#if (defined(USE_WESTON_IVI_SHELL)) //Weston with IVI-Shell and Genivi extensions
    uint32_t lIlmNativeHandle = reinterpret_cast<t_ilm_nativehandle>(m_waylandSurface);
#elif (defined(USE_LAYERMANAGER)) //LayerManager 1.x on Wayland 1.11
    struct wl_proxy * proxy = reinterpret_cast<struct wl_proxy *>(m_waylandSurface);
    uint32_t lIlmNativeHandle = (lContext->GetConnectionId() << 16) | static_cast<uint32_t>(wl_proxy_get_id(proxy));
#else
#error NO Compositor set. This device package requires either USE_WESTON_IVI_SHELL or USE_LAYERMANAGER defined for Wayland 1.11
#endif
#else //Legacy case for ILM 1.x on Wayland 1.1
    FEATSTD_LINT_CURRENT_SCOPE(438, "Violates MISRA C++ 2008 Required Rule 0-1-9: false positive, lObject is accessed")
    struct wl_object * lObject = reinterpret_cast<struct wl_object *>(m_waylandSurface);
    uint32_t lIlmNativeHandle = (lContext->GetConnectionId() << 16) | static_cast<uint32_t>(lObject->id);
#endif

    ilmErrorTypes lIlmError = ILM_FAILED;

#if (defined(USE_WESTON_IVI_SHELL)) //Weston with IVI-Shell and Genivi extensions
    // see https://github.com/GENIVI/wayland-ivi-extension/blob/master/ivi-layermanagement-api/ilmClient/src/ilm_client_wayland_platform.c

    ivi_surface *surf = ivi_application_surface_create((ivi_application*)lContext->GetIviApplication(), lSurfaceId, m_waylandSurface); //use  m_waylandSurface instead of casting lIlmNativeHandle back to wl_surface*
    lIlmError = surf == 0 ? ILM_FAILED : ILM_SUCCESS;
#else
    lIlmError = ilm_surfaceCreate(static_cast<t_ilm_nativehandle>(lIlmNativeHandle),
                                                lWindowWidth,
                                                lWindowHeight,
                                                ILM_PIXELFORMAT_RGBA_8888,
                                                &lSurfaceId);
#endif //(defined(USE_WESTON_IVI_SHELL)) 

    if (ILM_FAILED == lIlmError) {
        CANDERA_DEVICE_LOG_ERROR("Failed to create ILM surface %u!", lSurfaceId);
        return 0;
    }

    // Set some surface properties in ILM. This is not required for Weston with IVI Shell
#if (defined(USE_LAYERMANAGER)) //LayerManager 1.x on Wayland 1.11
#if !defined(CGIDEVICE_SCREENBROKER_ENABLED)
#if defined(CGIDEVICE_ILM_USE_ASSET_SURFACE_PROPERTIES)
    static_cast<void>(ilm_layerAddSurface(lLayerId, lSurfaceId));
    static_cast<void>(ilm_surfaceSetVisibility(lSurfaceId, m_properties->IsSurfaceVisible()));
#endif
#endif

    static_cast<void>(ilm_surfaceSetOpacity(lSurfaceId, m_properties->GetSurfaceOpacity()));

    static_cast<void>(ilm_surfaceSetDestinationRectangle(lSurfaceId,
                                       m_properties->GetX(),
                                       m_properties->GetY(),
                                       lWindowWidth,
                                       lWindowHeight));

    static_cast<void>(ilm_surfaceSetSourceRectangle(lSurfaceId,
                                  0,
                                  0,
                                  lWindowWidth,
                                  lWindowHeight));

#endif
    // Do a central commit for all ILM changes applied in prior statements
    lIlmError = ilm_commitChanges();
	
    if (ILM_SUCCESS == lIlmError) {
        CANDERA_DEVICE_LOG_INFO("Successfully created surface %u and added to layer %u",
                                lSurfaceId,
                                lLayerId);

        CANDERA_DEVICE_LOG_INFO("  Initial settings of surface %u: dimension=%dx%d, opacity=%.3f, visible=%s",
                                lSurfaceId,
                                lWindowWidth,
                                lWindowHeight,
                                m_properties->GetSurfaceOpacity(),
                                m_properties->IsSurfaceVisible() ? "yes" : "no");
    } else {
        CANDERA_DEVICE_LOG_ERROR("Failed to configure surface %u (ILM error: %d)!",
                                 lSurfaceId,
                                 lIlmError);
        return 0;
    }

#if defined(CGIDEVICE_SCREENBROKER_ENABLED)
    // Register ILM suurface ID for this screenbroker client instance at screenbroker
    ScreenBroker::RequestArg lRequestArg(ScreenBroker::ClientUtil::GenerateRequestId(),
                                         static_cast<ScreenBroker::UInt32>(-1),
                                         static_cast<ScreenBroker::UInt32>(lIlmNativeHandle));
    ScreenBroker::ClientApi::RegisterSurface(lRequestArg, lLayerId, lSurfaceId);
#endif

    // Store mapping of wayland surface instance to ILM surface ID,
    // as this is not provided via wayland but needed in input handling.
    GeniviTargetDisplay::IlmWaylandMapItem lItem;
    lItem.m_ilmSurfaceId = lSurfaceId;
    lItem.m_waylandSurface = m_waylandSurface;
    static_cast<void>(lDisplay->m_ilmWaylandMap.Add(lItem));

    // Create wayland EGL window instance
    EGLNativeWindowType lWindow =  reinterpret_cast<EGLNativeWindowType>(wl_egl_window_create(m_waylandSurface, lWindowWidth, lWindowHeight));
    if (0 != lWindow) {
        CANDERA_DEVICE_LOG_INFO("Created wayland EGL surface instance (%p) for surface %u", lWindow, lSurfaceId);
    } else {
        CANDERA_DEVICE_LOG_ERROR("Could not create egl wayland surface instance for surface %u!", lSurfaceId);
    }

    return lWindow;
}

// -----------------------------------------------------------------------------
GeniviSurface::GeniviSurface()
    : m_window(0),
      m_properties(0),
      m_display(-1),
      m_waylandSurface(0)
{
}

// -----------------------------------------------------------------------------
GeniviSurface::~GeniviSurface()
{
    Unload();
}

// -----------------------------------------------------------------------------
Int GeniviSurface::GetHeight() const
{
    return (0 != m_properties) ? m_properties->GetHeight() : 0;
}

// -----------------------------------------------------------------------------
Int GeniviSurface::GetWidth() const
{
    return (0 != m_properties) ? m_properties->GetWidth() : 0;
}

// -----------------------------------------------------------------------------
Int GeniviSurface::GetX() const
{
    Int posX = 0;
        if (0 != m_properties) {
            posX = m_properties->GetX();
            ilmErrorTypes err = ILM_FAILED;

   #if (defined(USE_WESTON_IVI_SHELL))
            ilmLayerProperties lLayerProperties;
            err = ilm_getPropertiesOfLayer(m_properties->GetLayerId(), &lLayerProperties);

            if (err == ILM_SUCCESS) {
                posX += lLayerProperties.sourceX;
            }
   #elif (defined(USE_LAYERMANAGER))
            t_ilm_uint pos[2] = { 0, 0 };
            err = ilm_layerGetPosition(m_properties->GetLayerId(), pos);

            if (err == ILM_SUCCESS) {
                posX += static_cast<Int>(pos[0]);
            }
    #endif


        }
    return posX;
}

// -----------------------------------------------------------------------------
Int GeniviSurface::GetY() const
{
    Int posY = 0;
        if (0 != m_properties) {
            posY = m_properties->GetY();
            ilmErrorTypes err = ILM_FAILED;

    #if (defined(USE_WESTON_IVI_SHELL))
            ilmLayerProperties lLayerProperties;
            err = ilm_getPropertiesOfLayer(m_properties->GetLayerId(), &lLayerProperties);

            if (err == ILM_SUCCESS) {
                posY += lLayerProperties.sourceY;
            }
    #elif (defined(USE_LAYERMANAGER))
            t_ilm_uint pos[2] = { 0, 0 };
            err = ilm_layerGetPosition(m_properties->GetLayerId(),pos);

            if (err == ILM_SUCCESS) {
                posY +=  static_cast<Int>(pos[1]);
            }
    #endif

        }
    return posY;
}

// -----------------------------------------------------------------------------
void GeniviSurface::ApplyChanges(const GeniviSurfaceProperties& /*surfaceProperties*/)
{
    FEATSTD_LINT_NEXT_EXPRESSION(1762, "Violates MISRA C++ 2008 Required Rule 9-3-3: Empty stub on target")
}

// -----------------------------------------------------------------------------
bool GeniviSurface::Upload(Int displayId, GeniviSurfaceProperties & surfaceProperties)
{
     if (m_display >= 0) {
        //already uploaded
        return false;
    }

    GeniviDisplay * lDisplay = GeniviDevicePackageInterface::GetInstance().GetDisplay(displayId);
    if (lDisplay == 0) {
        return false;
    }

    if (surfaceProperties.GetWidth() < 0) {
        surfaceProperties.SetWidth(lDisplay->GetParameters().width);
    }

    if (surfaceProperties.GetHeight() < 0) {
        surfaceProperties.SetHeight(lDisplay->GetParameters().height);
    }

    m_properties = &surfaceProperties;
    
    EGLNativeWindowType lWindow = CreateNativeWindow(displayId);

    if (lWindow == 0) {
        CANDERA_DEVICE_LOG_ERROR("Failed to create native window.");
        return false;
    }

    return AttachNativeWindowHandle(lWindow, displayId, surfaceProperties);
}

// -----------------------------------------------------------------------------
bool GeniviSurface::AttachNativeWindowHandle(EGLNativeWindowType handle,
                                         Int displayId,
                                         GeniviSurfaceProperties& /*surfaceProperties*/)
{
    if (m_display >= 0) {
        //already uploaded
        return false;
    }

    //check if displayId is valid
    GeniviDisplay * lDisplay = GeniviDevicePackageInterface::GetInstance().GetDisplay(displayId);
    if (lDisplay == 0) {
        return false;
    }

    m_display = displayId;
    m_window = handle;

    return true;

}

// -----------------------------------------------------------------------------
void GeniviSurface::Unload()
{
    if ((0 != m_properties) && (0 != m_window)) {

        // Remove ilm surface.
        t_ilm_layer lLayerId = m_properties->GetLayerId();
        t_ilm_surface lSurfaceId = m_properties->GetSurfaceId();
        static_cast<void>(ilm_layerRemoveSurface(lLayerId, lSurfaceId));
        static_cast<void>(ilm_surfaceRemove(lSurfaceId));
        static_cast<void>(ilm_commitChanges());

        // Deregister ILM surface ID for this screenbroker client instance at screenbroker
#if defined(CGIDEVICE_SCREENBROKER_ENABLED)
        ScreenBroker::RequestArg lRequestArg(ScreenBroker::ClientUtil::GenerateRequestId(),
                                             static_cast<ScreenBroker::UInt32>(-1),
                                             static_cast<ScreenBroker::UInt32>(0));
        ScreenBroker::ClientApi::DeregisterSurface(lRequestArg, lSurfaceId);
#endif

        // GeniviSurface unload callback
        if (s_callbackFunction != 0) {
            s_callbackFunction(lSurfaceId);
        }

        // Destroy wayland egl window.
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(740, "Unusual pointer cast [MISRA C++ Rule 5-2-7]: Conversion is intended and valid.")
        wl_egl_window_destroy(reinterpret_cast<struct wl_egl_window *>(m_window));

        if (m_waylandSurface != 0) {
            Candera::GeniviTargetDisplay * lDisplay = static_cast<GeniviTargetDisplay *>(Candera::GeniviDevicePackageInterface::GetInstance().GetDisplay(m_display));
            Int index = -1;
            for (SizeType i = 0; i < lDisplay->m_ilmWaylandMap.Size(); i++) {
                if ((lDisplay->m_ilmWaylandMap[i].m_waylandSurface == m_waylandSurface) && (lDisplay->m_ilmWaylandMap[i].m_ilmSurfaceId == UInt32(lSurfaceId))) {
                    index = i;
                }
            }

            if (index != -1) {
                static_cast<void>(lDisplay->m_ilmWaylandMap.Remove(index));
            }

            wl_surface_destroy(m_waylandSurface);
        }

        m_window = 0;
        m_display = -1;
    }
}

}
