//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GENIVI_SURFACE_PROPERTIES_H)
#define GENIVI_SURFACE_PROPERTIES_H

#include <Candera/Environment.h>

#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnitMetaInfo.h>

namespace Candera
{

/** @addtogroup GeniviGDI
 *  @{
 */
 
/**
 * @brief The class GeniviSurfaceProperties is a class representing the basic properties
 * of a Genivi surface.
 */
class GeniviSurfaceProperties
{
    public:

        /**
         * Constructor
         */
        GeniviSurfaceProperties();

        /**
         * Destructor
         */
        virtual ~GeniviSurfaceProperties();

        // Properties ----------------------------------------------------------------------------------------

        /**
         * Set the width of the window associated to this surface.
         * @param width value to use as width or negative value to use display width.
         */
        void SetWidth(Int width) { m_width = width; }
        /**
         * Get the width of the window associated to this surface.
         * @return value set by SetWidth or the default -1.
         */
        Int GetWidth() const { return m_width; }

        /**
         * Set the height of the window associated to this surface.
         * @param height value to use as height or negative value to use display height.
         */
        void SetHeight(Int height) { m_height = height; }
        /**
         * Get the height of the window associated to this surface.
         * @return value set by SetHeight or the default -1.
         */
        Int GetHeight() const { return m_height; }

        /**
         * Set the position in the X direction of the window associated to this surface.
         * @param x value to use as X position.
         */
        void SetX(Int x) { m_x = x; }
        /**
         * Get the position in the X direction of the window associated to this surface.
         * @return value set by SetX or the default 0.
         */
        Int GetX() const { return m_x; }

        /**
         * Set the position in the Y direction of the window associated to this surface.
         * @param y value to use as Y position.
         */
        void SetY(Int y) { m_y = y; }
        /**
         * Get the position in the Y direction of the window associated to this surface.
         * @return value set by SetY or the default 0.
         */
        Int GetY() const { return m_y; }

        /**
         * Set the ILM layer index to associate this surface to.
         * @param layerId The ILM layer index to associate this surface to.
         */
        void SetLayerId(UInt32 layerId) { m_layerId = layerId; }
        /**
         * Gets the index of the ILM layer this surface is associated to.
         * @return the index of the ILM layer this surface is associated to.
         */
        UInt32 GetLayerId() const { return m_layerId; }

        /**
         * Set the ILM surface index to associate this surface to.
         * @param surfaceId The ILM surface index to associate this surface to.
         */
        void SetSurfaceId(UInt32 surfaceId);
        /**
         * Gets the index of the ILM surface this surface is associated to.
         * @return the index of the ILM surface this surface is associated to.
         */
        UInt32 GetSurfaceId() const { return m_surfaceId; }

        /**
         * Sets the opacity of the ILM surface.
         * @param opacity Opacity of the ILM surface.
         */
        void SetSurfaceOpacity(Float opacity) { m_surfaceOpacity = opacity; }
        /**
         * Gets the opacity of the ILM surface.
         * @return  Opacity of the ILM surface.
         */
        Float GetSurfaceOpacity() const { return m_surfaceOpacity; }

        /**
         * Sets if ILM surface is visible or not.
         * @param visibility Visibility of the ILM surface.
         */
        void SetSurfaceVisibility(bool visibility) { m_isSurfaceVisibile = visibility; }
        /**
         * Gets if ILM surface is visible or not.
         * @return visibility Visibility of the ILM surface.
         */
        bool IsSurfaceVisible() const { return m_isSurfaceVisibile; }

        /**
         * Enable or disable this surface. The surface is enabled after information
         * on the attached render target was updated (i.e. when RenderTarget::SwapBuffers is called).
         * The surface is disabled the first time the display is updated. (i.e. as soon as possible.)
         * Note GraphicDeviceUnit::ApplyChanges must be called for this change to take effect.
         * @param enabled true to enable, false to disable.
         */
        void SetEnabled(bool enabled) { m_enabled = enabled; }
        /**
         * Check if surface is enabled.
         * @return value set by SetEnabled or the default true.
         */
        bool IsEnabled() const { return m_enabled; }

        /// @cond excluded from doxygen
        CdaForeignGduDef(GeniviSurfaceProperties)
            CdaGduProperties()

                CdaGduProperty(Width, Int, GetWidth, SetWidth)
                CdaDescription("Width of the render target (-1 indicates that it is derived from the display width).");
                CdaGduPropertyEnd()

                CdaGduProperty(Height, Int, GetHeight, SetHeight)
                CdaDescription("Height of the render target (-1 indicates that it is derived from the display height).");
                CdaGduPropertyEnd()

                CdaGduProperty(X, Int, GetX, SetX)
                CdaDescription("Position in the X direction of the window associated to this surface");
                CdaUpdateType(OnApplyUpdateType)
                CdaGduPropertyEnd()

                CdaGduProperty(Y, Int, GetY, SetY)
                CdaDescription("Position in the Y direction of the window associated to this surface");
                CdaUpdateType(OnApplyUpdateType)
                CdaGduPropertyEnd()

                CdaGduProperty(LayerId, UInt32, GetLayerId, SetLayerId)
                CdaDescription("ILM layer index to associate this surface to");
                CdaGduPropertyEnd()

                CdaGduProperty(SurfaceId, UInt32, GetSurfaceId, SetSurfaceId)
                CdaDescription("ILM surface index to associate this surface to");
                CdaGduPropertyEnd()

                CdaGduProperty(SurfaceOpacity, Float, GetSurfaceOpacity, SetSurfaceOpacity)
                CdaDescription("Opacity of the ILM surface");
                CdaGduPropertyEnd()

                CdaGduProperty(SurfaceVisibility, bool, IsSurfaceVisible, SetSurfaceVisibility)
                CdaDescription("Visibility of the ILM surface");
                CdaGduPropertyEnd()

                CdaGduProperty(Enabled, bool, IsEnabled, SetEnabled)
                CdaDescription("Enable or disable the surface. The surface is enabled after information on the attached render target was updated (i.e. when RenderTarget::SwapBuffers is called). The surface is disabled the first time the display is updated. (i.e. as soon as possible.)");
                CdaGduPropertyEnd()

            CdaGduPropertiesEnd()
        CdaGduDefEnd()
        /// @endcond

    private:

        bool m_enabled;

        bool m_isSurfaceVisibile;

        Int m_width;
        Int m_height;
        Int m_x;
        Int m_y;

        UInt32 m_layerId;
        UInt32 m_surfaceId;
        Float m_surfaceOpacity;

};

/** @}*/ //end of GeniviGDI

}

#endif
