//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GENIVI_DISPLAY_H)
#define GENIVI_DISPLAY_H

#include <CanderaPlatform/Device/Common/Base/Display.h>
#include <GeniviDevicePackageInterface.h>

namespace Candera
{
    class GeniviDevicePackageInterface;
    class GeniviSimulatedDisplay;

/** @addtogroup GeniviGDI
 *  @{
 */

/**
 * @brief The class GeniviDisplay represents a display on the "Genivi" platform.
 */
class GeniviDisplay : public Display
{
    friend class GeniviDevicePackageInterface;

    public:
        /**
         * Destructor
         */
        virtual ~GeniviDisplay();

        /**
         * An alternative to Upload. Native resources are created outside, and only
         * a handle is attached to the display object. This function creates the associations
         * needed internally by the graphic package. (e.g. a simulated display).
         * @param intendedWidth intended width display. (in most cases it should correspond
         *        to actual width of the display)
         * @param intendedHeight intended height display. (in most cases it should correspond
         *        to actual height of the display)
         * @param handle native handle of the display.
         * @return true if everything uploaded successfully, false otherwise.
         */
        virtual bool AttachNativeHandle(Int intendedWidth, Int intendedHeight, void* handle);

        /**
         * Retrieve a pointer to a simulated display object.
         * @return a pointer simulated display object, or 0 if a simulation is not available.
         */
        virtual SimulatedDisplay* ToSimulatedDisplay();

        /**
         * Retrieve a pointer to the native handle attached by @see AttachNativeHandle, or created
         * by @see Upload.
         * @return a native display handle, or 0 if none is available.
         */
        virtual void* GetNativeHandle();

        /**
         * Retrieve a pointer to a simulated display object.
         * @return a pointer simulated display object, or 0 if a simulation is not available.
         */
        GeniviSimulatedDisplay* ToGeniviSimulatedDisplay();

        /**
         * Retrieve the display parameters of this display.
         * @return the display parameters of this display.
         */
        const DisplayParam& GetParameters() const { return m_param; }

        /**
         * Apply the current properties set via meta-info interface to the device.
         * Implements virtual method from DeviceMetaInfoHost.
         */
        virtual void ApplyChanges();

        //void SetLayerOrder(Int32 lo) { m_layerOrder = lo; }
        //Int32 GetLayerOrder() const { return m_layerOrder; }

    protected:
        /**
         * Constructor
         */
        GeniviDisplay(Int displayId);

        DisplayParam m_param;

        //Int32 m_layerOrder;

        bool m_attached;

        /**
         * See Display::UploadInternal for more details.
         */
        virtual bool UploadInternal(const CommonSettings& settings);

        /**
         * See Display::UnloadInternal for more details.
         */
        virtual void UnloadInternal();
};

/** @}*/ //end of GeniviGDI

}

#endif
