//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_GlTypeMapper_h
#define Candera_GlTypeMapper_h


#include <CanderaPlatform/Device/Common/OpenGLES/GlInclude.h>

#include <Candera/Environment.h>
#include <Candera/EngineBase/Common/CanderaObject.h>

#ifdef CANDERA_3D_ENABLED
    #include <Candera/Engine3D/Core/VertexGeometry.h>
    #include <Candera/Engine3D/Core/Texture.h>
    #include <Candera/Engine3D/Core/RenderBuffer.h>

    #include <CanderaPlatform/Device/Common/Base/DirectTextureImage.h>
#else
    #include <CanderaPlatform/Device/Common/Stubs/DummyCandera3D.h>
#endif

#include <Candera/EngineBase/Common/Bitmap.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>

namespace Candera {

    // Specifies an invalid parameter for GL constants, used for not supported Open GL ES extension tokens.
    static const GLenum c_glInvalidParameter = 0xFFffFFff;

/** @addtogroup CommonDevice
 *  @{
 */

const GLenum CullFaceModeMapping[] =
{
    GL_FRONT,// RenderMode::FrontFaceCulling
    GL_BACK,            // RenderMode::BackFaceCulling
    GL_FRONT_AND_BACK   // Not supported by Candera as inefficient. To set an object invisible, see Node::SetRenderingEnabled().
};

const GLenum StencilPlaneMapping[] =
{
    GL_FRONT,// RenderMode::FrontFace
    GL_BACK,            // RenderMode::BackFace
    GL_FRONT_AND_BACK   // RenderMode::FrontAndBackFace
};

const GLenum StencilOperationMapping[] =
{
    GL_ZERO,// RenderMode::SetToZero
    GL_KEEP,// RenderMode::Keep
    GL_REPLACE,// RenderMode::Replace
    GL_INCR,// RenderMode::Increment
    GL_DECR,// RenderMode::Decrement
    GL_INVERT,// RenderMode::Invert
    GL_INCR_WRAP,// RenderMode::IncrementWrap
    GL_DECR_WRAP,// RenderMode::DecrementWrap
};

const GLenum WindingOrderMapping[] =
{
    GL_CW,// RenderMode::ClockWise
    GL_CCW              // RenderMode::CounterClockWise
};

const GLenum ComparisonFunctionMapping[] =
{
    GL_NEVER,// RenderMode::Never
    GL_LESS,// RenderMode::Less
    GL_EQUAL,// RenderMode::Equal
    GL_LEQUAL,// RenderMode::LessEqual
    GL_GREATER,// RenderMode::Greater
    GL_NOTEQUAL,// RenderMode::NotEqual
    GL_GEQUAL,// RenderMode::GreaterEqual
    GL_ALWAYS// RenderMode::Always
};

const GLenum BlendFactorMapping[] =
{
    GL_ZERO,// RenderMode::Zero
    GL_ONE,// RenderMode::One
    GL_SRC_COLOR,// RenderMode::SourceColor
    GL_ONE_MINUS_SRC_COLOR,// RenderMode::InverseSourceColor
    GL_SRC_ALPHA,// RenderMode::SourceAlpha
    GL_ONE_MINUS_SRC_ALPHA,// RenderMode::InverseSourceAlpha
    GL_DST_COLOR,// RenderMode::DestColor
    GL_ONE_MINUS_DST_COLOR,// RenderMode::InverseDestColor
    GL_DST_ALPHA,// RenderMode::DestAlpha
    GL_ONE_MINUS_DST_ALPHA,// RenderMode::InverseDestAlpha
    GL_CONSTANT_COLOR,// RenderMode::ConstantColor
    GL_ONE_MINUS_CONSTANT_COLOR,// RenderMode::InverseConstantColor
    GL_CONSTANT_ALPHA,// RenderMode::ConstantAlpha
    GL_ONE_MINUS_CONSTANT_ALPHA,// RenderMode::InverseConstantAlpha
    GL_SRC_ALPHA_SATURATE// RenderMode::SourceAlphaSaturate
};

const GLenum BlendOperationMapping[] =
{
    GL_FUNC_ADD,              // RenderMode::Add
    GL_FUNC_SUBTRACT,         // RenderMode::Subtract
    GL_FUNC_REVERSE_SUBTRACT, // RenderMode::ReverseSubtract
#if GL_EXT_blend_minmax
    GL_MIN_EXT,               // RenderMode::Min
    GL_MAX_EXT                // RenderMode::Max
#else
    c_glInvalidParameter,
    c_glInvalidParameter
#endif
};

const GLenum TextureUnitMapping[] =
{
    GL_TEXTURE0,                    // Texture Unit 0 = 0
    GL_TEXTURE1,                    // Texture Unit 1 = 1
    GL_TEXTURE2,                    // Texture Unit 2 = 2
    GL_TEXTURE3,                    // Texture Unit 3 = 3
    GL_TEXTURE4,                    // Texture Unit 4 = 4
    GL_TEXTURE5,                    // Texture Unit 5 = 5
    GL_TEXTURE6,                    // Texture Unit 6 = 6
    GL_TEXTURE7                     // Texture Unit 7 = 7
};

const GLenum TextureMinMagFilterMapping[] =
{
    GL_NEAREST,                     // Texture::MinMagFilter::Nearest = 0
    GL_LINEAR                       // Texture::MinMagFilter::Linear = 1
};

const GLenum TextureWrapModeMapping[] =
{
    GL_REPEAT,                      // Texture::WrapMode::Repeat = 0
    GL_CLAMP_TO_EDGE,               // Texture::WrapMode::ClampToEdge = 1
    GL_MIRRORED_REPEAT              // Texture::WrapMode::MirroredRepeat = 2
};

const GLenum TextureTargetTypeMapping[] =
{
    GL_TEXTURE_2D,                  //TextureImage::TextureTargetType::Texture2D = 0
    GL_TEXTURE_CUBE_MAP,            //TextureImage::TextureTargetType::TextureCubeMap = 1
    GL_TEXTURE_EXTERNAL_OES         //TextureImage::TextureTargetType::TextureExternalOes = 2
};

const GLenum TextureCubeMapTargetFaceMapping[] =
{
    GL_TEXTURE_CUBE_MAP_POSITIVE_X, //TextureImage::TargetFace::CubeMapPositiveXFace = 0
    GL_TEXTURE_CUBE_MAP_NEGATIVE_X, //TextureImage::TargetFace::CubeMapNegativeXFace = 1
    GL_TEXTURE_CUBE_MAP_POSITIVE_Y, //TextureImage::TargetFace::CubeMapPositiveYFace = 2
    GL_TEXTURE_CUBE_MAP_NEGATIVE_Y, //TextureImage::TargetFace::CubeMapNegativeYFace = 3
    GL_TEXTURE_CUBE_MAP_POSITIVE_Z, //TextureImage::TargetFace::CubeMapPositiveZFace = 4
    GL_TEXTURE_CUBE_MAP_NEGATIVE_Z  //TextureImage::TargetFace::CubeMapNegativeZFace = 5
};

const GLenum VertexGeometryUsageMapping[] =
{
    GL_STATIC_DRAW,                 // VertexBuffer::BufferUsage::StaticWrite = 0
    GL_DYNAMIC_DRAW,                // VertexBuffer::BufferUsage::DynamicWrite = 1
    GL_STREAM_DRAW                  // VertexBuffer::BufferUsage::InfrequentDraw = 2
};

const GLenum VertexDataTypeMapping[] =
{
    GL_FLOAT,             // Float32_1
    GL_FLOAT,             // Float32_2
    GL_FLOAT,             // Float32_3
    GL_FLOAT,             // Float32_4
    GL_UNSIGNED_BYTE,     // UInt8_1
    GL_UNSIGNED_BYTE,     // UInt8_1_N
    GL_UNSIGNED_BYTE,     // UInt8_2
    GL_UNSIGNED_BYTE,     // UInt8_2_N
    GL_UNSIGNED_BYTE,     // UInt8_3
    GL_UNSIGNED_BYTE,     // UInt8_3_N
    GL_UNSIGNED_BYTE,     // UInt8_4
    GL_UNSIGNED_BYTE,     // UInt8_4_N
    GL_BYTE,              // Int8_1
    GL_BYTE,              // Int8_1_N
    GL_BYTE,              // Int8_2
    GL_BYTE,              // Int8_2_N
    GL_BYTE,              // Int8_3
    GL_BYTE,              // Int8_3_N
    GL_BYTE,              // Int8_4
    GL_BYTE,              // Int8_4_N
    GL_UNSIGNED_SHORT,    // UInt16_1
    GL_UNSIGNED_SHORT,    // UInt16_1_N
    GL_UNSIGNED_SHORT,    // UInt16_2
    GL_UNSIGNED_SHORT,    // UInt16_2_N
    GL_UNSIGNED_SHORT,    // UInt16_3
    GL_UNSIGNED_SHORT,    // UInt16_3_N
    GL_UNSIGNED_SHORT,    // UInt16_4
    GL_UNSIGNED_SHORT,    // UInt16_4_N
    GL_SHORT,             // Int16_1
    GL_SHORT,             // Int16_1_N
    GL_SHORT,             // Int16_2
    GL_SHORT,             // Int16_2_N
    GL_SHORT,             // Int16_3
    GL_SHORT,             // Int16_3_N
    GL_SHORT,             // Int16_4
    GL_SHORT,             // Int16_4_N
    GL_HALF_FLOAT_OES,    // Float16_1
    GL_HALF_FLOAT_OES,    // Float16_2
    GL_HALF_FLOAT_OES,    // Float16_3
    GL_HALF_FLOAT_OES,    // Float16_4
    GL_UNSIGNED_INT,      // UInt32_1
    GL_UNSIGNED_INT,      // UInt32_1_N
    GL_UNSIGNED_INT,      // UInt32_2
    GL_UNSIGNED_INT,      // UInt32_2_N
    GL_UNSIGNED_INT,      // UInt32_3
    GL_UNSIGNED_INT,      // UInt32_3_N
    GL_UNSIGNED_INT,      // UInt32_4
    GL_UNSIGNED_INT,      // UInt32_4_N
    GL_INT,                // Int32_1
    GL_INT,               // Int32_1_N
    GL_INT,               // Int32_2
    GL_INT,               // Int32_2_N
    GL_INT,               // Int32_3
    GL_INT,               // Int32_3_N
    GL_INT,               // Int32_4
    GL_INT,               // Int32_4_N
};

const GLenum RenderBufferTypeMapping[] =
{
    GL_NONE,                            //RenderBuffer::Format::InvalidFormat = 0
    GL_RGB565,                          //RenderBuffer::Format::Rgb565Format = 1
    GL_RGBA4,                           //RenderBuffer::Format::Rgba4Format = 2
    GL_RGB5_A1,                         //RenderBuffer::Format::Rgb5A1Format = 3
    GL_DEPTH_COMPONENT16,               //RenderBuffer::Format::DepthComponent16Format = 4
    GL_STENCIL_INDEX8,                  //RenderBuffer::Format::StencilIndex8Format = 5
    GL_DEPTH_COMPONENT16_NONLINEAR_NV,  //RenderBuffer::Format::NvDepth16NonlinearFormat = 6
    GL_DEPTH_COMPONENT24_OES,           //RenderBuffer::Format::OesDepth24Format = 7
    GL_DEPTH_COMPONENT32_OES            //RenderBuffer::Format::OesDepth32Format = 8
};

const GLenum CoverageOperationTypeMapping[] =
{
    GL_COVERAGE_ALL_FRAGMENTS_NV,       //RenderMode::CoverageOperation::CoverAllFragments
    GL_COVERAGE_EDGE_FRAGMENTS_NV,      //RenderMode::CoverageOperation::CoverEdgeFragments
    GL_COVERAGE_AUTOMATIC_NV            //RenderMode::CoverageOperation::CoverAutomatic
};

const GLenum AttachmentPointMapping[] =
{
    GL_COLOR_ATTACHMENT0,           //ColorAttachment = 0
    GL_DEPTH_ATTACHMENT,            //DepthAttachment = 1
    GL_STENCIL_ATTACHMENT           //StencilAttachment = 2
};


const GLenum DeviceInfoMapping[] =
{
    GL_VENDOR,                      //VendorInfo = 0
    GL_RENDERER,                    //RendererInfo = 1
    GL_VERSION,                     //ApiVersionInfo = 2
    GL_SHADING_LANGUAGE_VERSION,    //ShadingLanguageVersionInfo = 3
    GL_EXTENSIONS                   //ExtensionsInfo = 4
};

const UInt8 VertexDataTypeSizeMapping[] =
{
    1,                // Float32_1
    2,                // Float32_2
    3,                // Float32_3
    4,                // Float32_4
    1,                // UInt8_1
    1,                // UInt8_1_N
    2,                // UInt8_2
    2,                // UInt8_2_N
    3,                // UInt8_3
    3,                // UInt8_3_N
    4,                // UInt8_4
    4,                // UInt8_4_N
    1,                // Int8_1
    1,                // Int8_1_N
    2,                // Int8_2
    2,                // Int8_2_N
    3,                // Int8_3
    3,                // Int8_3_N
    4,                // Int8_4
    4,                // Int8_4_N
    1,                // UInt16_1
    1,                // UInt16_1_N
    2,                // UInt16_2
    2,                // UInt16_2_N
    3,                // UInt16_3
    3,                // UInt16_3_N
    4 ,               // UInt16_4
    4,                // UInt16_4_N
    1,                // Int16_1
    1,                // Int16_1_N
    2,                // Int16_2
    2,                // Int16_2_N
    3,                // Int16_3
    3,                // Int16_3_N
    4,                // Int16_4
    4,                // Int16_4_N
    1,                // Float16_1
    2,                // Float16_2
    3,                // Float16_3
    4,                // Float16_4
    1,                // UInt32_1
    1,                // UInt32_1_N
    2,                // UInt32_2
    2,                // UInt32_2_N
    3,                // UInt32_3
    3,                // UInt32_3_N
    4,                // UInt32_4
    4,                // UInt32_4_N
    1,                // Int32_1
    1,                // Int32_1_N
    2,                // Int32_2
    2,                // Int32_2_N
    3,                // Int32_3
    3,                // Int32_3_N
    4,                // Int32_4
    4,                // Int32_4_N
};

const GLenum VertexPrimitivesMap[] =
{
    GL_TRIANGLES,
    GL_TRIANGLE_STRIP,
    GL_TRIANGLE_FAN,
    GL_LINES,
    GL_LINE_STRIP,
    GL_LINE_LOOP,
    GL_POINTS
};

const GLenum VertexIndexFormatMap[] =
{
    GL_NONE,              // ArrayBuffer = 0,
    GL_UNSIGNED_SHORT,    // UInt16IndexedArrayBuffer = 1,
    GL_UNSIGNED_BYTE,     // UInt8IndexedArrayBuffer = 2,
    GL_UNSIGNED_INT       // UInt32IndexedArrayBuffer = 3,
};

const GLenum HintTargetMapping[] =
{
    GL_GENERATE_MIPMAP_HINT,                    //MipMapGeneration = 0
    GL_FRAGMENT_SHADER_DERIVATIVE_HINT_OES,     //DerivativeCalculation = 1
};

const GLenum HintModeMapping[] =
{
    GL_FASTEST,     //Fastest = 0
    GL_NICEST,      //Nicest = 1
    GL_DONT_CARE,   //DontCare = 2
};

const GLenum DirectTextureFormatMapping[] =
{
#ifdef GL_VIV_direct_texture
    GL_VIV_YV12,    //FormatYV12
#ifndef USE_3PSW_VIVANTE_OPENGLES
    GL_VIV_I420,    //FormatI420
#else
    c_glInvalidParameter,
#endif
    GL_VIV_NV12,    //FormatNV12
    GL_VIV_NV21,    //FormatNV21
    GL_VIV_YUY2,    //FormatYUY2
    GL_VIV_UYVY,    //FormatUYVY
    GL_RGBA,        //FormatRGBA
    GL_BGRA_EXT,    //FormatBGRA
    GL_ALPHA,       //FormatAlpha
    GL_RGB565,      //FormatRGB565
    GL_RGB          //FormatRGB
#else
    c_glInvalidParameter,   //FormatYV12
    c_glInvalidParameter,   //FormatI420
    c_glInvalidParameter,   //FormatNV12
    c_glInvalidParameter,   //FormatNV21
    c_glInvalidParameter,   //FormatYUY2
    c_glInvalidParameter,   //FormatUYVY
    c_glInvalidParameter,   //FormatRGBA
    c_glInvalidParameter,   //FormatBGRA
    c_glInvalidParameter,   //FormatAlpha
    c_glInvalidParameter,   //FormatRGB565
    c_glInvalidParameter    //FormatRGB
#endif
};

const GLenum ShaderObjectTypeMapping[] =
{
    GL_VERTEX_SHADER,
    GL_FRAGMENT_SHADER
};

/**
 * @brief   The GlTypeMapper is a helper class used by RenderDevice to map Candera types to Open GL ES Types.
 */
class GlTypeMapper
{
    public:
        // Map from Candera types to OpenGL types:
        static GLenum MapCullFaceMode(UInt mode) { return CullFaceModeMapping[mode]; }
        static GLenum MapWindingOrder(UInt order) { return WindingOrderMapping[order]; }
        static GLenum MapComparisonFunction(UInt compFunc) { return ComparisonFunctionMapping[compFunc]; }
        static GLenum MapBlendFactor(UInt factor) { return BlendFactorMapping[factor]; }
        static GLenum MapBlendOperation(UInt operation) { return BlendOperationMapping[operation]; }
        static GLenum MapTextureUnit(UInt unit) { return TextureUnitMapping[unit]; }
        static GLenum MapTextureWrapMode(Texture::WrapMode wrapMode) { return TextureWrapModeMapping[wrapMode]; }
        static GLenum MapTextureMinMagFilter(Texture::MinMagFilter filter) { return TextureMinMagFilterMapping[filter]; }
        static GLenum MapTextureTargetType(TextureImage::TextureTargetType target) { return TextureTargetTypeMapping[target]; }
        static GLenum MapTextureCubeMapTargetFace(CubeMapTextureImage::TargetFace face) { return TextureCubeMapTargetFaceMapping[face]; }
        static GLenum MapBitmapPixelFormatToGLFormat(Bitmap::PixelFormat format) { return c_PixelFormatProperties[format].Format; }
        static GLenum MapBitmapPixelFormatToGLType(Bitmap::PixelFormat format) { return c_PixelFormatProperties[format].Type; }
        static GLenum MapBitmapPixelFormatToGLInternalFormat(Bitmap::PixelFormat format) { return c_PixelFormatProperties[format].Format; }
        static UInt8 MapBitmapPixelFormatToGLBitsPerPixel(Bitmap::PixelFormat format) { return c_PixelFormatProperties[format].BitsPerPixel; }
        static GLenum MapVertexGeometryUsage(VertexGeometry::BufferUsage usage) { return VertexGeometryUsageMapping[usage]; }
        static GLenum MapVertexDataType(VertexGeometry::VertexDataType type) { return VertexDataTypeMapping[type]; }
        static GLenum MapStencilPlane(UInt plane) { return StencilPlaneMapping[plane]; }
        static GLenum MapStencilOperation(UInt operation) { return StencilOperationMapping[operation]; }
        static GLenum MapRenderBufferFormat(RenderBuffer::Format format) { return RenderBufferTypeMapping[format]; }
        static GLenum MapCoverageOperation(RenderMode::CoverageOperation operation) { return CoverageOperationTypeMapping[operation]; }
        static GLenum MapDeviceInfo(RenderDevice::DeviceInfo deviceInfo) { return DeviceInfoMapping[deviceInfo]; }
        static UInt8  MapVertexDataTypeSize(VertexGeometry::VertexDataType type) { return VertexDataTypeSizeMapping[type]; }
        static GLenum MapVertexPrimitiveType(VertexBuffer::PrimitiveType type) { return VertexPrimitivesMap[type]; }
        static GLenum MapVertexIndexFormat(VertexGeometry::BufferType format) { return VertexIndexFormatMap[format]; }
        static GLenum MapHintTarget(RenderDevice::HintTarget type) { return HintTargetMapping[type]; }
        static GLenum MapHintMode(RenderDevice::HintMode type) { return HintModeMapping[type]; }
        static GLenum MapDirectTextureFormat(DirectTextureImage::Format format) { return DirectTextureFormatMapping[format]; }
        static GLenum MapShaderObjectType(Shader::ObjectType objectType) { return ShaderObjectTypeMapping[objectType]; }

        // Map from OpenGL types to Candera types:
        static Shader::UniformType MapGlUniformType(GLenum glType);
        static RenderMode::BlendFactor MapGlBlendFactor(GLenum glFactor);
        static RenderMode::BlendOperation MapGlBlendOperation(GLenum glOperation);

    private:
        struct PixelFormatProperties {
            GLenum Format;
            GLenum Type;
            UInt8 BitsPerPixel;
        };

        static const PixelFormatProperties c_PixelFormatProperties[];
};

 /** @} */ // end of CommonDevice

} // namespace Candera

#endif// Candera_GlTypeMapper_h
