//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !(defined GL_SHADER_STORAGE_INTERFACE_H)
#define GL_SHADER_STORAGE_INTERFACE_H

#include <CanderaTypes.h>
#include "FeatStd/MemoryManagement/Heap.h"

namespace Candera
{

   /** @addtogroup CommonDevice
   *  @{
   */
   /**
   * @brief SampleShaderStorage represents an interface used for storing binary shaders to persistent memory.
   */
   class GlShaderStorageInterface{
   public:
      /**
      * ctor
      */
      GlShaderStorageInterface(){}

      /**
      * dtor
      */
      virtual ~GlShaderStorageInterface(){}

      /**
      * @brief StorageObject contains all data necessary to store and retrieve a binary shader from persistent memory.
      * Objects of this type will be created whenever a shader(program) shall be loaded from disk.
      * The object will be destructed immediately after the upload.
      * To avoid unnecessary memory consumption, lifetime of data is handled by the concrete implementation of ShaderStorageInterface.
      * OnRelease is called as soon as resources may be released.
      * The entity that allocates StorageObject::m_data is responsible for cleanup! To do so, a method matching DestructorProc can be implemented and set.
      */
     class StorageObject{
      public:
         enum {
            IdentifierSize = 16, //no size related typos shall occur. As enum because it does not increase code size.
         }; 
         typedef void(*DestructorProc) (StorageObject* object); // function pointer definition for cleanup 
         /**
         * ctor
         * @param vertexShaderText
         * @param fragmentShadertext
         * @param shaderName
         * @param proc Optional implementation of DestructorProc. default is NULL. 
         */
         StorageObject(const Char *vertexShaderText, const Char *fragmentShadertext, const Char *shaderName, DestructorProc proc = 0) : 
             m_data(0), m_binaryFormat(0), m_dataSize(0), m_vertexShadertext(vertexShaderText), m_fragmentShadertext(fragmentShadertext), m_shaderName(shaderName), OnRelease(proc) { FeatStd::Internal::Memory::Set(m_identifier, 0, IdentifierSize); } //identifier may hold up to 16 bytes

         void *m_data; // provider is responsible for deletion. 
         UInt32 m_binaryFormat; // Binary format of the compiled and linked program
         SizeType m_dataSize; // bytes of data in data 
         UInt8 m_identifier[IdentifierSize]; // 128 bit unique identifier in 16 groups of one byte. this is optional
         const Char *m_vertexShadertext; // vertex shader text (mandatory, not 0)
         const Char *m_fragmentShadertext; // fragment shader text (mandatory, not 0)
         const Char *m_shaderName; // name of the shader (optional, may be 0)
         DestructorProc OnRelease; //Pointer to cleanup function. This will be called if the StorageObject is destructed.
         static void DefaultCleanupProc(GlShaderStorageInterface::StorageObject* object) //default implementation of Cleanup Proc
         {
             if (0 != object->m_data){
                 FEATSTD_DELETE_ARRAY(object->m_data);
                 object->m_data = 0;
             }
         }
      };
     /**
     * Retrieve
     * Retrieves the shader program identified by object.m_identifier from persistent storage. Has to be implemented in custom ShaderStorage.
     * @param object StorageObject where m_identifier has to be set to something different from {0}
     * @return true if successful. If true is returned object.m_data, object.m_dataSitze and object.m_m_binaryFormat have to be set.
     */
      virtual bool Retrieve(StorageObject & object) = 0;
      /**
      * Persist
      * Persists a shader to storage. Has to be implemented in custom ShaderStorage.
      * @param object StorageObject where object.m_identifier has to be set to something different from {0}, object.m_data has to be set to a valid chunk of memory containen the shader program, object.m_dataSitze has to be set to the ammount of bytes used in m_data.
      * @return true if successful. 
      */
      virtual bool Persist(StorageObject & object) = 0;

   };


}
#endif // GL_SHADER_STORAGE_INTERFACE_H
