//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GL_FRAME_BUFFER_PROPERTIES_H)
#define GL_FRAME_BUFFER_PROPERTIES_H

#include <CanderaPlatform/Device/Common/OpenGLES/GlFrameBufferFormatProperties.h>
#include <CanderaPlatform/Device/Common/Internal/GDU/GduPropertiesSizeExtension.h>


namespace Candera
{
/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief GlFrameBufferProperties is a class holding the properties needed for
 * initializing a GlTextureFrameBuffer.
 *
 * It features wrappers for the Open GL texture, depth buffer and stencil buffer formats,
 * buffer dimensions, and attachments that allow replacing some of the buffers.
 * The extensions must be ImageSources with the same semantics, for the binding to work.
 * Example:
 * - ExternalColorTexture must provide a texture handle, 
 * - ExternalDepthBuffer and ExternalStencilBuffer must provide render buffers with a formats supported as
 * depth buffer and stencil buffer respectively.
 */

 class GlFrameBufferProperties : public GduPropertiesSizeExtension<GlFrameBufferFormatProperties>
{
    public:
        FEATSTD_TYPEDEF_BASE(GduPropertiesSizeExtension<GlFrameBufferFormatProperties>);

        /**
         * Constructs a GlFrameBufferProperties object.
         */
        GlFrameBufferProperties();
        GlFrameBufferProperties(const GlFrameBufferFormatProperties& base);

        /**
         * Set number of MSAA samples, default is 1. MSAA is disabled when samples is set to a value less than 2.
         */
        void SetMsaaSamples(Int samples);

        /**
         * Get number of MSAA samples.
         */
        Int GetMsaaSamples() const { return m_msaaSamples; }

        /// @cond excluded from doxygen
        CdaForeignGduDef(GlFrameBufferProperties)
            CdaGduProperties()

                CdaGduImportBaseClassProperties(GlFrameBufferFormatProperties, GlFrameBufferFormatProperties)

                CdaGduProperty(Width, Int, GlFrameBufferProperties::Base::GetWidth, GlFrameBufferProperties::Base::SetWidth)
                    CdaDescription("Width of the render target (-1 indicates that it is derived from the display width).")
                    CdaValidityTest(IsWidthValid());
                CdaGduPropertyEnd()

                CdaGduProperty(Height, Int, GlFrameBufferProperties::Base::GetHeight, GlFrameBufferProperties::Base::SetHeight)
                    CdaDescription("Height of the render target (-1 indicates that it is derived from the display height).")
                CdaGduPropertyEnd()

                CdaGduProperty(MsaaSamples, Int, GetMsaaSamples, SetMsaaSamples)
                CdaDescription("MSAA Sample count")
                CdaGduPropertyEnd()

            CdaGduPropertiesEnd()
        CdaGduDefEnd()
        /// @endcond

    private:
        Int m_msaaSamples;

        virtual CdaPropertyMetaInfoErrorType IsWidthValid() const { return CdaValidValue(); }
};

/** @}*/ //end of CommonDevice
}

#endif //def GL_FRAME_BUFFER_PROPERTIES_H
